#
# Copyright (c) 2015-2023 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

__docformat__ = 'restructuredtext'

from pyams_content.component.paragraph import IBaseParagraph, IParagraphContainer, IParagraphContainerTarget
from pyams_content.features.json import JSONBaseConverter
from pyams_content.features.json.interfaces import IJSONConverter
from pyams_i18n.interfaces import INegotiator
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import adapter_config
from pyams_utils.registry import get_global_registry, get_utility


@adapter_config(name='paragraphs',
                required=(IParagraphContainerTarget, IPyAMSLayer),
                provides=IJSONConverter)
class JSONParagraphContainerConverter(JSONBaseConverter):
    """JSON paragraphs container converter"""

    is_inner = True

    @property
    def conversion_target(self):
        return None

    def convert_content(self, params):
        result = []
        registry = get_global_registry()
        for paragraph in IParagraphContainer(self.context).get_visible_paragraphs():
            converter = registry.queryMultiAdapter((paragraph, self.request), IJSONConverter)
            if converter is None:
                continue
            output = converter.to_json(params)
            for name, inner_converter in registry.getAdapters((paragraph, self.request),
                                                              IJSONConverter):
                if (not name) or (name in getattr(converter, 'excluded_converters', ())):
                    continue
                inner_output = inner_converter.to_json(params)
                if not inner_output:
                    continue
                if inner_converter.is_inner:
                    output.update({name: inner_output})
                else:
                    output.update(inner_output)
            result.append(output)
        return result


@adapter_config(required=(IBaseParagraph, IPyAMSLayer),
                provides=IJSONConverter)
class JSONParagraphConverter(JSONBaseConverter):
    """Base JSON paragraph converter"""

    def convert_content(self, params):
        result = super().convert_content(params)
        result.update({
            'factory': self.context.factory_name.lower(),
            'is_anchor': self.context.anchor
        })
        lang = params.get('lang')
        if not lang:
            negotiator = get_utility(INegotiator)
            lang = negotiator.server_language
        self.get_i18n_attribute(self.context, 'title', lang, result)
        return result
