#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from persistent import Persistent
from pyramid.events import subscriber
from pyramid.threadlocal import get_current_registry
from zope.container.contained import Contained
from zope.copy import copy
from zope.interface import implementer
from zope.lifecycleevent import ObjectModifiedEvent
from zope.lifecycleevent.interfaces import IObjectAddedEvent, IObjectModifiedEvent, \
    IObjectRemovedEvent
from zope.schema.fieldproperty import FieldProperty
from zope.schema.vocabulary import SimpleTerm, SimpleVocabulary

from pyams_content.component.paragraph.interfaces import CONTENT_PARAGRAPHS_VOCABULARY, \
    IBaseParagraph, IParagraphContainer, IParagraphContainerTarget, IParagraphFactory, \
    IParagraphFactorySettings, IParagraphTitle, PARAGRAPH_FACTORIES_VOCABULARY
from pyams_content.features.checker import BaseContentChecker
from pyams_content.features.preview.interfaces import IPreviewTarget
from pyams_content.features.renderer import RenderedContentMixin
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION, PUBLISH_CONTENT_PERMISSION
from pyams_content.shared.common.interfaces import IWfSharedContent
from pyams_content.shared.common.interfaces.types import IWfTypedSharedContent
from pyams_form.interfaces.form import IFormContextPermissionChecker
from pyams_i18n.interfaces import II18n
from pyams_utils.adapter import ContextAdapter, adapter_config
from pyams_utils.html import html_to_text
from pyams_utils.registry import query_utility
from pyams_utils.request import check_request
from pyams_utils.traversing import get_parent
from pyams_utils.vocabulary import vocabulary_config
from pyams_workflow.content import HiddenContentPublicationInfo
from pyams_workflow.interfaces import IWorkflowPublicationInfo, IWorkflowState


__docformat__ = 'restructuredtext'


#
# Auto-creation of default paragraphs
#

@subscriber(IObjectAddedEvent, context_selector=IParagraphContainerTarget)
def handle_new_paragraphs_container(event):
    """Handle new paragraphs container

    Subscriber to IObjectAddedEvent for objects implementing IParagraphContainerTarget: when a
    new paragraphs container is created (in version 1), automatically create new paragraphs
    matching paragraphs settings
    """
    # don't create new paragraphs if container is not empty
    container = IParagraphContainer(event.object)
    if len(container) > 0:
        return
    content = get_parent(container, IWfSharedContent)
    # only apply settings to first version
    version_state = IWorkflowState(content, None) if content is not None else None
    if (version_state is not None) and (version_state.version_id > 1):
        return
    # check for typed shared content types
    if IWfTypedSharedContent.providedBy(content):
        datatype = content.get_data_type()
        if datatype is not None:
            source_container = IParagraphContainer(datatype)
            for paragraph in source_container.values():
                container.append(copy(paragraph))
    if len(container) > 0:
        return
    # check for shared content tool auto created paragraphs
    settings = get_parent(container, IParagraphFactorySettings)
    if settings is not None:
        for factory_name in settings.auto_created_paragraphs or ():
            factory = query_utility(IParagraphFactory, name=factory_name)
            if factory is not None:
                container.append(factory.content_type())


#
# Base paragraph classes and subscribers
#

@implementer(IBaseParagraph, IPreviewTarget)
class BaseParagraph(RenderedContentMixin, Persistent, Contained):
    """Base paragraph persistent class"""

    factory_name = None
    icon_class = ''
    icon_hint = ''

    visible = FieldProperty(IBaseParagraph['visible'])
    anchor = FieldProperty(IBaseParagraph['anchor'])
    locked = FieldProperty(IBaseParagraph['locked'])
    title = FieldProperty(IBaseParagraph['title'])

    empty_title = ' -' * 8


@implementer(IParagraphFactory)
class BaseParagraphFactory(object):
    """Base paragraph factory class"""

    name = None
    content_type = None
    secondary_menu = False


class BaseParagraphContentChecker(BaseContentChecker):
    """Base paragraph content checker mixin"""

    @property
    def label(self):
        request = check_request()
        translate = request.localizer.translate
        return II18n(self.context).query_attribute('title', request) or \
            '({0})'.format(translate(self.context.icon_hint).lower())


@vocabulary_config(name=PARAGRAPH_FACTORIES_VOCABULARY)
class ParagraphFactoriesVocabulary(SimpleVocabulary):
    """Paragraph factories vocabulary"""

    def __init__(self, context=None):
        request = check_request()
        registry = request.registry
        translate = request.localizer.translate
        terms = sorted([SimpleTerm(name, title=translate(util.name))
                        for name, util in registry.getUtilitiesFor(IParagraphFactory)],
                       key=lambda x: x.title)
        super(ParagraphFactoriesVocabulary, self).__init__(terms)


@vocabulary_config(name=CONTENT_PARAGRAPHS_VOCABULARY)
class ContentParagraphsVocabulary(SimpleVocabulary):
    """Content paragraphs vocabulary"""

    def __init__(self, context):

        def get_title(paragraph):
            adapter = request.registry.queryMultiAdapter((paragraph, request), IParagraphTitle)
            if adapter is not None:
                return html_to_text(adapter)
            return II18n(paragraph).query_attribute('title', request=request) \
                or BaseParagraph.empty_title

        request = check_request()
        if not IParagraphContainerTarget.providedBy(context):
            context = get_parent(context, IParagraphContainerTarget)
        if context is not None:
            terms = [SimpleTerm(para.__name__,
                                title='§{0}: {1}'.format(index+1, get_title(para)))
                     for index, para in enumerate(IParagraphContainer(context).values())]
        else:
            terms = []
        super(ContentParagraphsVocabulary, self).__init__(terms)


@adapter_config(required=IBaseParagraph,
                provides=IFormContextPermissionChecker)
class BaseParagraphPermissionChecker(ContextAdapter):
    """Paragraph permission checker"""

    @property
    def edit_permission(self):
        content = get_parent(self.context, IParagraphContainerTarget)
        return IFormContextPermissionChecker(content).edit_permission


@adapter_config(name='delete',
                required=IBaseParagraph,
                provides=IFormContextPermissionChecker)
class BaseParagraphDeletePermissionChecker(BaseParagraphPermissionChecker):
    """Paragraph delete permission checker"""

    @property
    def edit_permission(self):
        permission = super().edit_permission
        if self.context.locked and (permission == MANAGE_CONTENT_PERMISSION):
            return PUBLISH_CONTENT_PERMISSION
        return permission


@subscriber(IObjectAddedEvent, context_selector=IBaseParagraph)
def handle_added_paragraph(event):
    """Handle added paragraph"""
    content = get_parent(event.object, IParagraphContainerTarget)
    if content is not None:
        get_current_registry().notify(ObjectModifiedEvent(content))


@subscriber(IObjectModifiedEvent, context_selector=IBaseParagraph)
def handle_modified_paragraph(event):
    """Handle modified paragraph"""
    content = get_parent(event.object, IParagraphContainerTarget)
    if content is not None:
        get_current_registry().notify(ObjectModifiedEvent(content))


@subscriber(IObjectRemovedEvent, context_selector=IBaseParagraph)
def handle_removed_paragraph(event):
    """Handle removed paragraph"""
    content = get_parent(event.object, IParagraphContainerTarget)
    if content is not None:
        get_current_registry().notify(ObjectModifiedEvent(content))


@adapter_config(required=IBaseParagraph,
                provides=IWorkflowPublicationInfo)
def base_paragraph_workflow_publication_info(context):
    """Base paragraph workflow publication info"""
    if not context.visible:
        return HiddenContentPublicationInfo()
    return None
