#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from z3c.form import field
from z3c.form.browser.checkbox import SingleCheckBoxFieldWidget
from zope.interface import Interface, implementer

from pyams_content.component.association.interfaces import IAssociationContainer
from pyams_content.component.association.zmi import AssociationItemAJAXAddForm, \
    AssociationItemAJAXEditForm
from pyams_content.component.association.zmi.interfaces import IAssociationsView
from pyams_content.component.links import ExternalLink, IInternalLinkCustomInfo, InternalLink, \
    MailtoLink
from pyams_content.component.links.interfaces import ICustomInternalLinkTarget, IExternalLink, \
    IInternalLink, ILinkContainerTarget, IMailtoLink
from pyams_content.component.paragraph.zmi import get_json_paragraph_markers_refresh_event
from pyams_content.component.paragraph.zmi.container import ParagraphContainerCounterBase
from pyams_content.component.paragraph.zmi.interfaces import IParagraphContainerTable, \
    IParagraphTitleToolbar
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION
from pyams_content.reference.pictograms.zmi.widget import PictogramSelectFieldWidget
from pyams_form.form import ajax_config
from pyams_form.interfaces.form import IInnerSubForm
from pyams_form.security import ProtectedFormObjectMixin
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.interfaces.viewlet import IToolbarAddingMenu
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.viewlet.toolbar import ToolbarMenuItem
from pyams_utils.adapter import adapter_config
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogAddForm, AdminDialogEditForm, InnerAdminEditForm
from pyams_zmi.interfaces import IPropertiesEditForm


__docformat__ = 'restructuredtext'

from pyams_content import _


#
# Internal links views
#

@viewlet_config(name='internal-links', context=ILinkContainerTarget, layer=IPyAMSLayer,
                view=IParagraphContainerTable, manager=IParagraphTitleToolbar,
                permission=VIEW_SYSTEM_PERMISSION, weight=10)
class InternalLinksCounter(ParagraphContainerCounterBase):
    """Internal links count column"""

    weight = 10
    action_class = 'action internal-links nowrap width-40'
    icon_class = 'fa fa-fw fa-external-link-square fa-rotate-90'
    icon_hint = _("Internal links")

    marker_type = 'internal-links'

    @property
    def count(self):
        return len([file for file in IAssociationContainer(self.context).values()
                    if IInternalLink.providedBy(file)])


@viewlet_config(name='add-internal-link.menu', context=ILinkContainerTarget,
                view=IAssociationsView, layer=IPyAMSLayer, manager=IToolbarAddingMenu, weight=50)
class InternalLinkAddMenu(ProtectedFormObjectMixin, ToolbarMenuItem):
    """Internal link add menu"""

    label = _("Add internal link")
    label_css_class = 'fa fa-fw fa-external-link-square fa-rotate-90'

    url = 'add-internal-link.html'
    modal_target = True


@pagelet_config(name='add-internal-link.html', context=ILinkContainerTarget, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='add-internal-link.json', context=ILinkContainerTarget, layer=IPyAMSLayer,
             base=AssociationItemAJAXAddForm)
class InternalLinkAddForm(AdminDialogAddForm):
    """Internal link add form"""

    legend = _("Add new internal link")
    icon_css_class = 'fa fa-fw fa-external-link-square fa-rotate-90'

    fields = field.Fields(IInternalLink).select('reference', 'force_canonical_url', 'title',
                                                'description',
                                                'pictogram_name')
    fields['force_canonical_url'].widgetFactory = SingleCheckBoxFieldWidget
    fields['pictogram_name'].widgetFactory = PictogramSelectFieldWidget

    edit_permission = MANAGE_CONTENT_PERMISSION

    def create(self, data):
        return InternalLink()

    def add(self, object):
        IAssociationContainer(self.context).append(object)

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        if output:
            output.setdefault('events', []).append(
                get_json_paragraph_markers_refresh_event(self.context, self.request,
                                                         self, InternalLinksCounter))
        return output


@pagelet_config(name='properties.html', context=IInternalLink, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@ajax_config(name='properties.json', context=IInternalLink, layer=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION, base=AssociationItemAJAXEditForm)
@implementer(IPropertiesEditForm)
class InternalLinkPropertiesEditForm(AdminDialogEditForm):
    """Internal link properties edit form"""

    prefix = 'internallink_properties.'

    legend = _("Edit internal link properties")
    icon_css_class = 'fa fa-fw fa-external-link-square fa-rotate-90'
    dialog_class = 'modal-large'

    fields = field.Fields(IInternalLink).select('reference', 'force_canonical_url', 'title',
                                                'description',
                                                'pictogram_name')
    fields['force_canonical_url'].widgetFactory = SingleCheckBoxFieldWidget
    fields['pictogram_name'].widgetFactory = PictogramSelectFieldWidget

    edit_permission = None  # defined by IFormContextPermissionChecker adapter

    def get_ajax_output(self, changes):
        if changes:
            return self.get_associations_table()
        else:
            return super(self.__class__, self).get_ajax_output(changes)


@adapter_config(name='custom',
                context=(ICustomInternalLinkTarget, IPyAMSLayer, InternalLinkPropertiesEditForm),
                provides=IInnerSubForm)
class CustomInternalLinkPropertiesEditForm(InnerAdminEditForm):
    """Custom internal link properties edit form"""

    prefix = 'custom_properties.'

    css_class = 'form-group'
    padding_class = ''

    legend = _("Custom target properties")
    fieldset_class = 'bordered'

    @property
    def fields(self):
        info = IInternalLinkCustomInfo(self.context, None)
        if info is not None:
            return field.Fields(info.properties_interface).omit('properties_interface')
        return field.Fields(Interface)

    weight = 1

    def render(self):
        if not self.fields:
            return ''
        return super(CustomInternalLinkPropertiesEditForm, self).render()


#
# External links views
#

@viewlet_config(name='external-links', context=ILinkContainerTarget, layer=IPyAMSLayer,
                view=IParagraphContainerTable, manager=IParagraphTitleToolbar,
                permission=VIEW_SYSTEM_PERMISSION, weight=11)
class ExternalLinksCounter(ParagraphContainerCounterBase):
    """External links count column"""

    weight = 11
    action_class = 'action external-links nowrap width-40'
    icon_class = 'fa fa-fw fa-link fa-rotate-90'
    icon_hint = _("External links")

    marker_type = 'external-links'

    @property
    def count(self):
        return len([file for file in IAssociationContainer(self.context).values()
                    if IExternalLink.providedBy(file)])


@viewlet_config(name='add-external-link.menu', context=ILinkContainerTarget,
                view=IAssociationsView, layer=IPyAMSLayer, manager=IToolbarAddingMenu, weight=51)
class ExternalLinkAddMenu(ProtectedFormObjectMixin, ToolbarMenuItem):
    """External link add menu"""

    label = _("Add external link")
    label_css_class = 'fa fa-fw fa-link fa-rotate-90'

    url = 'add-external-link.html'
    modal_target = True


@pagelet_config(name='add-external-link.html', context=ILinkContainerTarget, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='add-external-link.json', context=ILinkContainerTarget, layer=IPyAMSLayer,
             base=AssociationItemAJAXAddForm)
class ExternalLinkAddForm(AdminDialogAddForm):
    """External link add form"""

    legend = _("Add new external link")
    icon_css_class = 'fa fa-fw fa-link fa-rotate-90'

    fields = field.Fields(IExternalLink).select('url', 'title', 'description', 'pictogram_name',
                                                'language')
    fields['pictogram_name'].widgetFactory = PictogramSelectFieldWidget

    edit_permission = MANAGE_CONTENT_PERMISSION

    def create(self, data):
        return ExternalLink()

    def add(self, object):
        IAssociationContainer(self.context).append(object)

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        if output:
            output.setdefault('events', []).append(
                get_json_paragraph_markers_refresh_event(self.context, self.request,
                                                         self, ExternalLinksCounter))
        return output


@pagelet_config(name='properties.html', context=IExternalLink, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@ajax_config(name='properties.json', context=IExternalLink, layer=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION, base=AssociationItemAJAXEditForm)
@implementer(IPropertiesEditForm)
class ExternalLinkPropertiesEditForm(AdminDialogEditForm):
    """External link properties edit form"""

    prefix = 'externallink_properties.'

    legend = _("Edit external link properties")
    icon_css_class = 'fa fa-fw fa-link fa-rotate-90'
    dialog_class = 'modal-large'

    fields = field.Fields(IExternalLink).select('url', 'title', 'description', 'pictogram_name',
                                                'language')
    fields['pictogram_name'].widgetFactory = PictogramSelectFieldWidget

    edit_permission = None  # defined by IFormContextPermissionChecker adapter

    def get_ajax_output(self, changes):
        if changes:
            return self.get_associations_table()
        else:
            return super(self.__class__, self).get_ajax_output(changes)


#
# Mailto links views
#

@viewlet_config(name='mailto-links', context=ILinkContainerTarget, layer=IPyAMSLayer,
                view=IParagraphContainerTable, manager=IParagraphTitleToolbar,
                permission=VIEW_SYSTEM_PERMISSION, weight=12)
class MailtoLinksCounter(ParagraphContainerCounterBase):
    """Mailto links count column"""

    weight = 12
    action_class = 'action mailto-links nowrap width-40'
    icon_class = 'fa fa-fw fa-envelope-o'
    icon_hint = _("Mailto links")

    marker_type = 'mailto-links'

    @property
    def count(self):
        return len([file for file in IAssociationContainer(self.context).values()
                    if IMailtoLink.providedBy(file)])


@viewlet_config(name='add-mailto-link.menu', context=ILinkContainerTarget, view=IAssociationsView,
                layer=IPyAMSLayer, manager=IToolbarAddingMenu, weight=52)
class MailtoLinkAddMenu(ProtectedFormObjectMixin, ToolbarMenuItem):
    """Mailto link add menu"""

    label = _("Add mailto link")
    label_css_class = 'fa fa-fw fa-envelope-o'

    url = 'add-mailto-link.html'
    modal_target = True


@pagelet_config(name='add-mailto-link.html', context=ILinkContainerTarget, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='add-mailto-link.json', context=ILinkContainerTarget, layer=IPyAMSLayer,
             base=AssociationItemAJAXAddForm)
class MailtoLinkAddForm(AdminDialogAddForm):
    """Mailto link add form"""

    legend = _("Add new mailto link")
    icon_css_class = 'fa fa-fw fa-envelope-o'

    fields = field.Fields(IMailtoLink).select('address', 'address_name', 'title', 'description',
                                              'pictogram_name')
    fields['pictogram_name'].widgetFactory = PictogramSelectFieldWidget

    edit_permission = MANAGE_CONTENT_PERMISSION

    def create(self, data):
        return MailtoLink()

    def add(self, object):
        IAssociationContainer(self.context).append(object)

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        if output:
            output.setdefault('events', []).append(
                get_json_paragraph_markers_refresh_event(self.context, self.request,
                                                         self, MailtoLinksCounter))
        return output


@pagelet_config(name='properties.html', context=IMailtoLink, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@ajax_config(name='properties.json', context=IMailtoLink, layer=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION, base=AssociationItemAJAXEditForm)
@implementer(IPropertiesEditForm)
class MailtoLinkPropertiesEditForm(AdminDialogEditForm):
    """Mailto link properties edit form"""

    prefix = 'mailtolink_properties.'

    legend = _("Edit mailto link properties")
    icon_css_class = 'fa fa-fw fa-envelope-o'

    fields = field.Fields(IMailtoLink).select('address', 'address_name', 'title', 'description',
                                              'pictogram_name')
    fields['pictogram_name'].widgetFactory = PictogramSelectFieldWidget

    edit_permission = None  # defined by IFormContextPermissionChecker adapter

    def get_ajax_output(self, changes):
        if changes:
            return self.get_associations_table()
        else:
            return super(self.__class__, self).get_ajax_output(changes)
