#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from transaction.interfaces import ITransactionManager
from z3c.form import button
from z3c.form.interfaces import INPUT_MODE
from zope.interface import implementer

from pyams_content.component.illustration import IBasicIllustration
from pyams_content.component.illustration.interfaces import IBasicIllustrationTarget, IIllustration, \
    IIllustrationParagraph, ILLUSTRATION_PARAGRAPH_TYPE
from pyams_content.component.paragraph.interfaces import IParagraphContainerTarget
from pyams_content.component.paragraph.zmi import BaseParagraphAJAXAddForm, BaseParagraphAJAXEditForm, \
    BaseParagraphAddForm, BaseParagraphAddMenu, BaseParagraphPropertiesEditForm, IParagraphInnerEditFormButtons, \
    get_json_paragraph_refresh_event
from pyams_content.component.paragraph.zmi.container import ParagraphContainerCounterBase
from pyams_content.component.paragraph.zmi.interfaces import IParagraphContainerTable, IParagraphContainerView, \
    IParagraphInnerEditor, IParagraphTitleToolbar
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION
from pyams_form.form import ajax_config
from pyams_form.interfaces.form import IInnerForm
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.event import get_json_form_refresh_event
from pyams_skin.interfaces.viewlet import IToolbarAddingMenu
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import adapter_config
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION
from pyams_viewlet.viewlet import viewlet_config

from pyams_content import _


#
# Illustration
#

@viewlet_config(name='add-illustration.menu', context=IParagraphContainerTarget, view=IParagraphContainerView,
                layer=IPyAMSLayer, manager=IToolbarAddingMenu, weight=60)
class IllustrationAddMenu(BaseParagraphAddMenu):
    """Illustration add menu"""

    label = _("Illustration...")
    label_css_class = 'fa fa-fw fa-file-image-o'
    url = 'add-illustration.html'
    paragraph_type = ILLUSTRATION_PARAGRAPH_TYPE


@pagelet_config(name='add-illustration.html', context=IParagraphContainerTarget, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='add-illustration.json', context=IParagraphContainerTarget, layer=IPyAMSLayer,
             base=BaseParagraphAJAXAddForm)
class IllustrationAddForm(BaseParagraphAddForm):
    """Illustration add form"""

    legend = _("Add new illustration")
    dialog_class = 'modal-large'

    content_interface = IIllustrationParagraph

    @property
    def fields(self):
        return super(IllustrationAddForm, self).fields \
            .select('data', 'title', 'alt_title', 'author', 'description', 'renderer')


@pagelet_config(name='properties.html', context=IIllustrationParagraph, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='properties.json', context=IIllustrationParagraph, layer=IPyAMSLayer,
             base=BaseParagraphAJAXEditForm)
class IllustrationPropertiesEditForm(BaseParagraphPropertiesEditForm):
    """Illustration properties edit form"""

    prefix = 'illustration_properties.'

    legend = _("Edit illustration properties")
    dialog_class = 'modal-large'

    content_interface = IIllustrationParagraph

    @property
    def fields(self):
        return super(IllustrationPropertiesEditForm, self).fields \
            .select('data', 'title', 'alt_title', 'author', 'description', 'renderer')

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        if 'title' in changes.get(IBasicIllustration, ()):
            output.setdefault('events', []).append(get_json_paragraph_refresh_event(self.context, self.request))
        return output


@adapter_config(context=(IIllustrationParagraph, IPyAMSLayer), provides=IParagraphInnerEditor)
@ajax_config(name='inner-properties.json', context=IIllustrationParagraph, layer=IPyAMSLayer,
             base=BaseParagraphAJAXEditForm)
@implementer(IInnerForm)
class IllustrationInnerEditForm(IllustrationPropertiesEditForm):
    """Illustration inner edit form"""

    legend = None

    @property
    def buttons(self):
        if self.mode == INPUT_MODE:
            return button.Buttons(IParagraphInnerEditFormButtons)
        else:
            return button.Buttons()

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        updated = changes.get(IBasicIllustration, ())
        if 'title' in updated:
            output.setdefault('events', []).append(get_json_paragraph_refresh_event(self.context, self.request))
        updated = changes.get(IIllustrationParagraph, ())
        if 'data' in updated:
            # we have to commit transaction to be able to handle blobs...
            ITransactionManager(self.context).get().commit()
            output.setdefault('events', []).append(get_json_form_refresh_event(self.context, self.request,
                                                                               IllustrationInnerEditForm))
        return output


#
# Paragraph container illustration marker
#

@viewlet_config(name='illustration', context=IIllustrationParagraph, layer=IPyAMSLayer, view=IParagraphContainerTable,
                manager=IParagraphTitleToolbar, permission=VIEW_SYSTEM_PERMISSION, weight=5)
@viewlet_config(name='illustration', context=IBasicIllustrationTarget, layer=IPyAMSLayer, view=IParagraphContainerTable,
                manager=IParagraphTitleToolbar, permission=VIEW_SYSTEM_PERMISSION, weight=5)
class ParagraphIllustrationMarker(ParagraphContainerCounterBase):
    """Paragraph illustration marker column"""

    weight = 5
    action_class = 'action illustration nowrap width-40'
    icon_class = 'fa fa-fw fa-picture-o'
    icon_hint = _("Illustration")

    marker_type = 'illustration'

    @property
    def count(self):
        illustration = IIllustration(self.context, None)
        if illustration and illustration.data:
            for value in illustration.data.values():
                if value:
                    return 1
        return 0
