#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from transaction.interfaces import ITransactionManager
from z3c.form import field
from zope.interface import implementer

from pyams_content.component.illustration.interfaces import IBasicIllustration, IBasicIllustrationTarget, \
    IIllustration, IIllustrationTarget, ILinkIllustrationTarget
from pyams_content.component.illustration.zmi.paragraph import ParagraphIllustrationMarker
from pyams_content.component.paragraph import IBaseParagraph
from pyams_content.component.paragraph.zmi import get_json_paragraph_markers_refresh_event
from pyams_content.features.renderer.zmi.widget import RendererFieldWidget
from pyams_content.zmi import pyams_content
from pyams_form.interfaces.form import IInnerSubForm, IWidgetsPrefixViewletsManager
from pyams_form.security import ProtectedFormObjectMixin
from pyams_skin.event import get_json_form_refresh_event, get_json_widget_refresh_event
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.viewlet.toolbar import JsToolbarAction
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config
from pyams_utils.fanstatic import get_resource_path
from pyams_utils.interfaces.data import IObjectData
from pyams_viewlet.viewlet import EmptyViewlet, Viewlet, viewlet_config
from pyams_zmi.form import InnerAdminEditForm
from pyams_zmi.interfaces import IPropertiesEditForm

from pyams_content import _


#
# Illustration properties inner edit form
#

@adapter_config(name='illustration',
                context=(IBasicIllustrationTarget, IPyAMSLayer, IPropertiesEditForm),
                provides=IInnerSubForm)
class BasicIllustrationPropertiesInnerEditForm(InnerAdminEditForm):
    """Basic illustration properties inner edit form"""

    prefix = 'basic_illustration_form.'

    css_class = 'form-group'
    padding_class = ''

    legend = _("Illustration")

    fields = field.Fields(IBasicIllustration).omit('__parent__', '__name__')

    weight = 10

    @property
    def legend_class(self):
        if IBaseParagraph.providedBy(self.context):
            return 'illustration switcher no-y-padding padding-right-10 pull-left width-auto'
        else:
            return 'illustration switcher no-y-padding'

    @property
    def fieldset_class(self):
        result = 'margin-top-10 padding-y-5'
        if not IBaseParagraph.providedBy(self.context):
            result += ' bordered'
        return result

    hide_widgets_prefix_div = True

    def getContent(self):
        return IIllustration(self.context)

    def check_mode(self):
        if self.parent_form is not None:
            self.mode = self.parent_form.mode

    @property
    def switcher_state(self):
        if not IBaseParagraph.providedBy(self.context):
            content = self.getContent()
            if content.has_data():
                return 'open'

    def get_ajax_output(self, changes):
        output = super(BasicIllustrationPropertiesInnerEditForm, self).get_ajax_output(changes)
        events = output.setdefault('events', [])
        updated = changes.get(IBasicIllustration, ())
        if 'data' in updated:
            # we have to commit transaction to be able to handle blobs...
            ITransactionManager(self.context).get().commit()
            if IBaseParagraph.providedBy(self.context):
                if self.getContent().data:
                    events.append(
                        get_json_paragraph_markers_refresh_event(self.context, self.request, self,
                                                                 ParagraphIllustrationMarker))
                else:
                    events.append(
                        get_json_paragraph_markers_refresh_event(self.context, self.request, self,
                                                                 EmptyViewlet,
                                                                 ParagraphIllustrationMarker.marker_type))
            events.append(get_json_form_refresh_event(self.context, self.request, self.__class__))
        return output


@adapter_config(name='illustration',
                context=(IIllustrationTarget, IPyAMSLayer, IPropertiesEditForm),
                provides=IInnerSubForm)
class IllustrationPropertiesInnerEditForm(BasicIllustrationPropertiesInnerEditForm):
    """Illustration properties inner edit form"""

    prefix = 'illustration_form.'

    fields = field.Fields(IIllustration).omit('__parent__', '__name__')
    fields['renderer'].widgetFactory = RendererFieldWidget

    @property
    def legend(self):
        if IBaseParagraph.providedBy(self.context):
            return _("Illustration")
        else:
            return _("Header illustration")

    @property
    def switcher_state(self):
        if not IBaseParagraph.providedBy(self.context):
            return 'open'

    def get_ajax_output(self, changes):
        output = super(IllustrationPropertiesInnerEditForm, self).get_ajax_output(changes)
        if 'renderer' in changes.get(IIllustration, ()):
            output.setdefault('events', []).append(
                get_json_widget_refresh_event(self.context, self.request,
                                              IllustrationPropertiesInnerEditForm, 'renderer'))
        return output


@adapter_config(name='link-illustration',
                context=(ILinkIllustrationTarget, IPyAMSLayer, IPropertiesEditForm),
                provides=IInnerSubForm)
class LinkIllustrationPropertiesInnerEditForm(BasicIllustrationPropertiesInnerEditForm):
    """Link illustration properties inner edit form"""

    prefix = 'link_illustration_form.'

    legend = _("Navigation link illustration")
    legend_class = 'illustration switcher no-y-padding'

    weight = 12

    def getContent(self):
        registry = self.request.registry
        return registry.getAdapter(self.context, IIllustration, name='link')


@viewlet_config(name='add-illustration.action',
                context=IBasicIllustrationTarget, layer=IPyAMSLayer,
                view=BasicIllustrationPropertiesInnerEditForm,
                manager=IWidgetsPrefixViewletsManager)
@implementer(IObjectData)
class AddIllustrationAction(ProtectedFormObjectMixin, JsToolbarAction):
    """Add illustration action"""

    group_css_class = 'btn-group clearfix persistent'
    label = _("Add illustration")
    url = 'PyAMS_content.illustration.addIllustration'

    def __new__(cls, context, request, view, manager):
        if not IBaseParagraph.providedBy(context):
            return None
        illustration = IIllustration(context, None)
        if illustration and illustration.data:
            for value in illustration.data.values():
                if value:
                    return None
        return JsToolbarAction.__new__(cls)

    @property
    def object_data(self):
        return {
            'ams-plugins': 'pyams_content',
            'ams-plugin-pyams_content-src': get_resource_path(pyams_content)
        }


@viewlet_config(name='illustration-thumbnail',
                context=IBasicIllustrationTarget, layer=IPyAMSLayer,
                view=BasicIllustrationPropertiesInnerEditForm,
                manager=IWidgetsPrefixViewletsManager)
@template_config(template='templates/illustration-thumbnail.pt', layer=IPyAMSLayer)
class IllustrationThumbnail(Viewlet):
    """Paragraph illustration thumbnail"""

    def __new__(cls, context, request, view, manager):
        if not IBaseParagraph.providedBy(context):
            return None
        illustration = IIllustration(context, None)
        if (not illustration) or not illustration.data:
            return None
        for value in illustration.data.values():
            if value:
                return Viewlet.__new__(cls)
        return None

    def __init__(self, context, request, view, manager):
        super(IllustrationThumbnail, self).__init__(context, request, view, manager)
        self.illustration = IIllustration(context)
