#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from pyramid.renderers import render
from z3c.form import field
from z3c.form.interfaces import NOT_CHANGED
from zope.interface import Interface
from zope.lifecycleevent import ObjectCreatedEvent
from zope.location import locate
from zope.schema.interfaces import WrongType

from pyams_content.component.gallery import IGallery, IGalleryTarget
from pyams_content.component.gallery.file import GalleryFile
from pyams_content.component.gallery.interfaces import GALLERY_FILE_HIDDEN_FIELDS, IBaseGallery, \
    IGalleryFile, IGalleryParagraph
from pyams_content.component.gallery.zmi.interfaces import IGalleryContentsView, \
    IGalleryMediasAddFields
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION
from pyams_content.shared.common.zmi import WfSharedContentPermissionMixin
from pyams_file.file import get_magic_content_type
from pyams_file.interfaces.archive import IArchiveExtractor
from pyams_file.zmi.file import FilePropertiesAction
from pyams_form.form import AJAXAddForm, ajax_config
from pyams_form.group import NamedWidgetsGroup
from pyams_form.security import get_edit_permission
from pyams_i18n.interfaces import II18n
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.event import get_json_refresh_event
from pyams_skin.interfaces.viewlet import IContextActions, IWidgetTitleViewletManager
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.viewlet.toolbar import JsToolbarActionItem, ToolbarAction
from pyams_utils.interfaces import FORBIDDEN_PERMISSION, ICacheKeyValue, VIEW_SYSTEM_PERMISSION
from pyams_utils.registry import query_utility
from pyams_utils.traversing import get_parent
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogAddForm, AdminDialogEditForm


__docformat__ = 'restructuredtext'

from pyams_content import _


@viewlet_config(name='add-media.menu',
                context=IBaseGallery, view=IGalleryContentsView,
                manager=IWidgetTitleViewletManager)
class GalleryMediaAddMenu(WfSharedContentPermissionMixin, ToolbarAction):
    """Gallery media add menu"""

    label = _("Add media(s)")

    url = 'add-media.html'
    modal_target = True
    stop_propagation = True


@viewlet_config(name='add-media.menu',
                context=IGalleryTarget, view=IGalleryContentsView,
                manager=IWidgetTitleViewletManager)
class GalleryTargetMediaAddMenu(GalleryMediaAddMenu):
    """Gallery target media add menu"""

    def get_url(self):
        gallery = IGallery(self.context)
        return absolute_url(gallery, self.request, self.url)


@pagelet_config(name='add-media.html',
                context=IBaseGallery, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='add-media.json',
             context=IBaseGallery, layer=IPyAMSLayer,
             base=AJAXAddForm)
class GalleryMediaAddForm(AdminDialogAddForm):
    """Gallery media add form"""

    legend = _("Add media(s)")
    icon_css_class = 'fa -fa-fw fa-picture-o'

    fields = field.Fields(IGalleryMediasAddFields)
    edit_permission = MANAGE_CONTENT_PERMISSION

    def create(self, data):
        medias = []
        medias_data = data['medias_data']
        if medias_data and (medias_data is not NOT_CHANGED):
            filename = None
            if isinstance(medias_data, (list, tuple)):
                filename, medias_data = medias_data
            if hasattr(medias_data, 'seek'):
                medias_data.seek(0)
            registry = self.request.registry
            content_type = get_magic_content_type(medias_data)
            if isinstance(content_type, bytes):
                content_type = content_type.decode()
            if hasattr(medias_data, 'seek'):
                medias_data.seek(0)
            extractor = query_utility(IArchiveExtractor, name=content_type)
            if extractor is not None:
                extractor.initialize(medias_data)
                for content, filename in extractor.get_contents():
                    try:
                        media = GalleryFile()
                        locate(media, self.context)
                        media.data = filename, content
                    except WrongType:
                        continue
                    else:
                        registry.notify(ObjectCreatedEvent(media))
                        medias.append(media)
            else:
                try:
                    media = GalleryFile()
                    locate(media, self.context)
                    media.data = filename, medias_data if filename else medias_data
                except WrongType:
                    pass
                else:
                    registry.notify(ObjectCreatedEvent(media))
                    medias.append(media)
            for media in medias:
                media.author = data.get('author')
                self.context.append(media, notify=False)
        return medias

    def get_ajax_output(self, changes):
        return {
            'status': 'reload',
            'location': absolute_url(self.context, self.request, 'get-gallery-medias.html'),
            'target': '#gallery_medias_{0}'.format(ICacheKeyValue(self.context))
        }


@viewlet_config(name='file.showhide.action',
                context=IGalleryFile, layer=IPyAMSLayer, view=Interface,
                manager=IContextActions, weight=1,
                permission=VIEW_SYSTEM_PERMISSION)
class GalleryFileShowHideAction(WfSharedContentPermissionMixin, JsToolbarActionItem):
    """Gallery file show/hide action"""

    edit_permission = FORBIDDEN_PERMISSION

    def __init__(self, context, request, view, manager):
        super(GalleryFileShowHideAction, self).__init__(context, request, view, manager)
        gallery = get_parent(context, IBaseGallery)
        if gallery is not None:
            self.edit_permission = get_edit_permission(request, gallery)

    @property
    def label(self):
        if self.request.has_permission(self.edit_permission, context=self.context):
            return _("Show/hide media")
        return ''

    @property
    def label_css_class(self):
        if self.context.visible:
            return 'fa fa-fw fa-eye'
        return 'fa fa-fw fa-eye-slash text-danger'

    hint_gravity = 'nw'

    url = 'PyAMS_content.galleries.switchMediaVisibility'

    def get_url(self):
        if self.request.has_permission(self.edit_permission, context=self.context):
            return self.url


@viewlet_config(name='file.properties.action',
                context=IGalleryFile, layer=IPyAMSLayer, view=Interface,
                manager=IContextActions, permission=VIEW_SYSTEM_PERMISSION, weight=5)
class GalleryFilePropertiesAction(FilePropertiesAction):
    """Media properties action"""

    url = 'gallery-file-properties.html'


@pagelet_config(name='gallery-file-properties.html',
                context=IGalleryFile, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@ajax_config(name='gallery-file-properties.html',
             context=IGalleryFile, layer=IPyAMSLayer)
class GalleryFilePropertiesEditForm(AdminDialogEditForm):
    """Gallery file properties edit form"""

    prefix = 'file_properties.'

    legend = _("Update media properties")
    icon_css_class = 'fa fa-fw fa-picture-o'
    dialog_class = 'modal-large'

    fields = field.Fields(IGalleryFile).omit(*GALLERY_FILE_HIDDEN_FIELDS)

    @property
    def title(self):
        gallery = get_parent(self.context, IBaseGallery)
        return II18n(gallery).query_attribute('title', request=self.request)

    def updateGroups(self):
        self.add_group(NamedWidgetsGroup(self, 'audio_file', self.widgets,
                                         ('sound', 'sound_title', 'sound_description'),
                                         bordered=False,
                                         legend=_("Audio content"),
                                         css_class='inner',
                                         switch=True,
                                         display_mode='auto'))
        super(GalleryFilePropertiesEditForm, self).updateGroups()

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        if 'title' in changes.get(IGalleryFile, ()):
            gallery = get_parent(self.context, IBaseGallery)
            if gallery is not None:
                output.setdefault('events', []).append({
                    'event': 'myams.refresh',
                    'options': {
                        'handler': 'PyAMS_content.galleries.updateMediaTitle',
                        'media_id': 'media_{0}'.format(ICacheKeyValue(self.context)),
                        'title': II18n(self.context).query_attribute('title',
                                                                     request=self.request)
                    }
                })
        paragraph = get_parent(self.context, IGalleryParagraph)
        if paragraph is not None:
            output.setdefault('events', []).append(
                get_json_refresh_event(
                    object_id='media_{0}'.format(ICacheKeyValue(self.context)),
                    content=render('templates/gallery-media-thumbnail.pt', {
                        'gallery': paragraph,
                        'media': self.context
                    }, request=self.request)))
        else:
            gallery = get_parent(self.context, IBaseGallery)
            if gallery is not None:
                output.setdefault('events', []).append(
                    get_json_refresh_event(
                        object_id='media_{0}'.format(ICacheKeyValue(self.context)),
                        content=render('templates/gallery-media-thumbnail.pt', {
                            'gallery': gallery.__parent__,
                            'media': self.context
                        }, request=self.request)))
        return output


@viewlet_config(name='gallery-file-remover.action',
                context=IGalleryFile, layer=IPyAMSLayer, view=Interface,
                manager=IContextActions, weight=90)
class GalleryFileRemoverAction(WfSharedContentPermissionMixin, JsToolbarActionItem):
    """Gallery file remover action"""

    label = _("Remove media...")
    label_css_class = 'fa fa-fw fa-trash'
    hint_gravity = 'nw'

    url = 'PyAMS_content.galleries.removeMedia'

    @property
    def permission(self):
        gallery = get_parent(self.context, IBaseGallery)
        return get_edit_permission(self.request, gallery)
