#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from persistent import Persistent
from pyramid.events import subscriber
from pyramid.threadlocal import get_current_registry
from zope.container.contained import Contained
from zope.interface import alsoProvides, implementer
from zope.lifecycleevent import ObjectModifiedEvent
from zope.lifecycleevent.interfaces import IObjectAddedEvent, IObjectModifiedEvent, \
    IObjectRemovedEvent
from zope.schema.fieldproperty import FieldProperty

from pyams_content.component.gallery.interfaces import IBaseGallery, IGalleryFile
from pyams_content.features.checker import BaseContentChecker
from pyams_content.features.checker.interfaces import IContentChecker, MISSING_LANG_VALUE, \
    MISSING_VALUE
from pyams_content.interfaces import MANAGE_SITE_ROOT_PERMISSION
from pyams_content.shared.common.interfaces import IWfSharedContent
from pyams_file.interfaces import IImage, IResponsiveImage
from pyams_file.property import FileProperty
from pyams_form.interfaces.form import IFormContextPermissionChecker
from pyams_i18n.interfaces import II18n, II18nManager, INegotiator
from pyams_utils.adapter import ContextAdapter, adapter_config
from pyams_utils.registry import get_utility
from pyams_utils.request import check_request
from pyams_utils.traversing import get_parent
from pyams_workflow.content import HiddenContentPublicationInfo
from pyams_workflow.interfaces import IWorkflowPublicationInfo


#
# Gallery file
#

@implementer(IGalleryFile)
class GalleryFile(Persistent, Contained):
    """Gallery file info"""

    _data = FileProperty(IGalleryFile['data'])
    title = FieldProperty(IGalleryFile['title'])
    alt_title = FieldProperty(IGalleryFile['alt_title'])
    description = FieldProperty(IGalleryFile['description'])
    author = FieldProperty(IGalleryFile['author'])
    sound = FileProperty(IGalleryFile['sound'])
    sound_title = FieldProperty(IGalleryFile['sound_title'])
    sound_description = FieldProperty(IGalleryFile['sound_description'])
    visible = FieldProperty(IGalleryFile['visible'])

    @property
    def data(self):
        return self._data

    @data.setter
    def data(self, value):
        self._data = value
        if IImage.providedBy(self._data):
            alsoProvides(self._data, IResponsiveImage)
            
    @data.deleter
    def data(self):
        del self._data


@adapter_config(context=IGalleryFile,
                provides=IWorkflowPublicationInfo)
def gallery_file_publication_info(context):
    """Gallery file publication info"""
    if not context.visible:
        return HiddenContentPublicationInfo()
    return None


@adapter_config(context=IGalleryFile,
                provides=IFormContextPermissionChecker)
class GalleryFilePermissionChecker(ContextAdapter):
    """Gallery file permission checker"""

    @property
    def edit_permission(self):
        gallery = get_parent(self.context, IBaseGallery)
        if gallery is not None:
            return IFormContextPermissionChecker(gallery).edit_permission
        return MANAGE_SITE_ROOT_PERMISSION


@subscriber(IObjectAddedEvent, context_selector=IGalleryFile)
def handle_added_gallery_file(event):
    """Handle added gallery file"""
    content = get_parent(event.object, IWfSharedContent)
    if content is not None:
        get_current_registry().notify(ObjectModifiedEvent(content))


@subscriber(IObjectModifiedEvent, context_selector=IGalleryFile)
def handle_modified_gallery_file(event):
    """Handle modified gallery file"""
    content = get_parent(event.object, IWfSharedContent)
    if content is not None:
        get_current_registry().notify(ObjectModifiedEvent(content))


@subscriber(IObjectRemovedEvent, context_selector=IGalleryFile)
def handle_removed_gallery_file(event):
    """Handle removed gallery file"""
    content = get_parent(event.object, IWfSharedContent)
    if content is not None:
        get_current_registry().notify(ObjectModifiedEvent(content))


@adapter_config(context=IGalleryFile,
                provides=IContentChecker)
class GalleryFileContentChecker(BaseContentChecker):
    """Gallery file content checker"""

    @property
    def label(self):
        request = check_request()
        return II18n(self.context).query_attribute('title', request) or self.context.data.filename

    def inner_check(self, request):
        output = []
        translate = request.localizer.translate
        manager = get_parent(self.context, II18nManager)
        if manager is not None:
            langs = manager.get_languages()
        else:
            negotiator = get_utility(INegotiator)
            langs = (negotiator.server_language,)
        i18n = II18n(self.context)
        for lang in langs:
            for attr in ('title', 'alt_title', 'description'):
                value = i18n.get_attribute(attr, lang, request)
                if not value:
                    field_title = translate(IGalleryFile[attr].title)
                    if len(langs) == 1:
                        output.append(translate(MISSING_VALUE).format(field=field_title))
                    else:
                        output.append(translate(MISSING_LANG_VALUE).format(field=field_title, lang=lang))
        for attr in ('data', 'author'):
            value = getattr(self.context, attr)
            if not value:
                output.append(translate(MISSING_VALUE).format(field=translate(IGalleryFile[attr].title)))
        return output
