#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from pyramid.events import subscriber
from z3c.form import field
from z3c.form.interfaces import IDataExtractedEvent, NOT_CHANGED
from z3c.table.column import GetAttrColumn
from z3c.table.interfaces import IColumn
from zope.dublincore.interfaces import IZopeDublinCore
from zope.interface import Interface
from zope.schema import Choice

from pyams_content.component.association.interfaces import IAssociationContainer
from pyams_content.component.association.zmi import AssociationItemAJAXAddForm, \
    AssociationItemAJAXEditForm, AssociationsTable
from pyams_content.component.association.zmi.interfaces import IAssociationsView
from pyams_content.component.extfile import EXTERNAL_FILES_FACTORIES
from pyams_content.component.extfile.interfaces import IExtAudio, IExtFile, \
    IExtFileContainerTarget, IExtImage, IExtMedia, IExtVideo
from pyams_content.component.extfile.zmi.widget import I18nExtFileTitleFieldWidget
from pyams_content.component.paragraph.zmi import get_json_paragraph_markers_refresh_event
from pyams_content.component.paragraph.zmi.container import ParagraphContainerCounterBase
from pyams_content.component.paragraph.zmi.interfaces import IParagraphContainerTable, \
    IParagraphTitleToolbar
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION
from pyams_form.form import ajax_config
from pyams_form.security import ProtectedFormObjectMixin
from pyams_i18n.interfaces import INegotiator
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.interfaces.viewlet import IToolbarAddingMenu
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.table import I18nColumn
from pyams_skin.viewlet.toolbar import ToolbarMenuDivider, ToolbarMenuItem
from pyams_utils.adapter import adapter_config
from pyams_utils.date import SH_DATETIME_FORMAT, format_datetime
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION
from pyams_utils.registry import query_utility
from pyams_utils.timezone import tztime
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogAddForm, AdminDialogEditForm


__docformat__ = 'restructuredtext'

from pyams_content import _


class IExtFileFactoryChooser(Interface):
    """External file factory chooser interface"""

    factory = Choice(title=_("External file type"),
                     vocabulary='PyAMS files factories',
                     default='file',
                     required=True)


@viewlet_config(name='add-extfile.divider',
                context=IExtFileContainerTarget, layer=IPyAMSLayer, view=IAssociationsView,
                manager=IToolbarAddingMenu, weight=59)
class ExtFileAddMenuDivider(ProtectedFormObjectMixin, ToolbarMenuDivider):
    """External file add menu divider"""


#
# External file view
#

@viewlet_config(name='extfiles',
                context=IExtFileContainerTarget, layer=IPyAMSLayer, view=IParagraphContainerTable,
                manager=IParagraphTitleToolbar, weight=20,
                permission=VIEW_SYSTEM_PERMISSION)
class ExtFilesCounter(ParagraphContainerCounterBase):
    """External files counter"""

    weight = 20
    action_class = 'action extfiles nowrap width-40'
    icon_class = 'fa fa-fw fa-file-text-o'
    icon_hint = _("External files")

    marker_type = 'extfiles'

    @property
    def count(self):
        return len([file for file in IAssociationContainer(self.context).values()
                    if IExtFile.providedBy(file) and not IExtMedia.providedBy(file)])


@viewlet_config(name='add-extfile.menu',
                context=IExtFileContainerTarget, layer=IPyAMSLayer, view=IAssociationsView,
                manager=IToolbarAddingMenu, weight=60)
class ExtFileAddMenu(ProtectedFormObjectMixin, ToolbarMenuItem):
    """External file add menu"""

    label = _("Add external file")
    label_css_class = 'fa fa-fw fa-file-text-o'

    url = 'add-extfile.html'
    modal_target = True


@pagelet_config(name='add-extfile.html',
                context=IExtFileContainerTarget, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='add-extfile.json',
             context=IExtFileContainerTarget, layer=IPyAMSLayer,
             base=AssociationItemAJAXAddForm)
class ExtFileAddForm(AdminDialogAddForm):
    """External file add form"""

    legend = _("Add new external file")
    icon_css_class = 'fa fa-fw fa-file-text-o'

    fields = field.Fields(IExtFile).select('data', 'filename', 'title', 'description',
                                           'author', 'language')
    fields['title'].widgetFactory = I18nExtFileTitleFieldWidget

    edit_permission = MANAGE_CONTENT_PERMISSION

    def create(self, data):
        factory = EXTERNAL_FILES_FACTORIES.get('file')
        if factory is not None:
            return factory[0]()

    def add(self, object):
        IAssociationContainer(self.context).append(object)

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        if output:
            output.setdefault('events', []).append(
                get_json_paragraph_markers_refresh_event(self.context, self.request,
                                                         self, ExtFilesCounter))
        return output


@pagelet_config(name='properties.html',
                context=IExtFile, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@ajax_config(name='properties.json',
             context=IExtFile, layer=IPyAMSLayer,
             base=AssociationItemAJAXEditForm)
class ExtFilePropertiesEditForm(AdminDialogEditForm):
    """External file properties edit form"""

    prefix = 'extfile_properties.'

    legend = _("Update file properties")
    icon_css_class = 'fa fa-fw fa-file-text-o'
    dialog_class = 'modal-large'

    fields = field.Fields(IExtFile).select('data', 'filename', 'title', 'description',
                                           'author', 'language')
    fields['title'].widgetFactory = I18nExtFileTitleFieldWidget

    edit_permission = MANAGE_CONTENT_PERMISSION

    def get_ajax_output(self, changes):
        if changes:
            return self.get_associations_table()
        return super(self.__class__, self).get_ajax_output(changes)


@subscriber(IDataExtractedEvent, form_selector=ExtFilePropertiesEditForm)
def handle_file_properties_edit_form_data(event):
    """Handle file properties edit form data

    This subscriber is used to handle file name extension based on uploaded content
    and on currently selected filename.
    """
    i18n_negotiator = query_utility(INegotiator)
    server_language = i18n_negotiator.server_language if i18n_negotiator is not None \
        else event.form.request.registry.settings.get('pyramid.default_locale_name', 'en')
    data = event.data
    file_upload = data.get('data', {}).get(server_language)
    if (not file_upload) or (file_upload is NOT_CHANGED):
        return
    file_name, file_data = '', None
    if isinstance(file_upload, tuple):
        file_name, file_data = file_upload
    new_name = data.get('filename')
    if not new_name:
        data['filename'] = file_name
    else:
        if '.' in file_name:
            file_ext = file_name.rsplit('.', 1)[1]
            if not new_name.endswith(file_ext):
                base_name = new_name.rsplit('.', 1)[0] if '.' in new_name else new_name
                data['filename'] = '{}.{}'.format(base_name, file_ext)


#
# Images views
#

@viewlet_config(name='extimages',
                context=IExtFileContainerTarget, layer=IPyAMSLayer, view=IParagraphContainerTable,
                manager=IParagraphTitleToolbar, weight=21,
                permission=VIEW_SYSTEM_PERMISSION)
class ExtImagesCounter(ParagraphContainerCounterBase):
    """External image files counter"""

    weight = 21
    action_class = 'action extimages nowrap width-40'
    icon_class = 'fa fa-fw fa-file-image-o'
    icon_hint = _("Images")

    marker_type = 'extimages'

    @property
    def count(self):
        return len([file for file in IAssociationContainer(self.context).values()
                    if IExtImage.providedBy(file)])


@viewlet_config(name='add-extimage.menu',
                context=IExtFileContainerTarget, layer=IPyAMSLayer, view=IAssociationsView,
                manager=IToolbarAddingMenu, weight=61)
class ExtImageAddMenu(ProtectedFormObjectMixin, ToolbarMenuItem):
    """External image add menu"""

    label = _("Add image")
    label_css_class = 'fa fa-fw fa-file-image-o'

    url = 'add-extimage.html'
    modal_target = True


@pagelet_config(name='add-extimage.html',
                context=IExtFileContainerTarget, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='add-extimage.json',
             context=IExtFileContainerTarget, layer=IPyAMSLayer,
             base=AssociationItemAJAXAddForm)
class ExtImageAddForm(ExtFileAddForm):
    """External image add form"""

    legend = _("Add new image")
    icon_css_class = 'fa fa-fw fa-file-image-o'

    fields = field.Fields(IExtImage).select('data', 'filename', 'title', 'description', 'author')
    fields['title'].widgetFactory = I18nExtFileTitleFieldWidget

    def updateWidgets(self, prefix=None):
        super(ExtImageAddForm, self).updateWidgets(prefix)
        if 'title' in self.widgets:
            self.widgets['title'].description = None
        if 'description' in self.widgets:
            self.widgets['description'].description = None

    def create(self, data):
        factory = EXTERNAL_FILES_FACTORIES.get('image')
        if factory is not None:
            return factory[0]()

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        if output:
            output.setdefault('events', []).append(
                get_json_paragraph_markers_refresh_event(self.context, self.request,
                                                         self, ExtImagesCounter))
        return output


@pagelet_config(name='properties.html',
                context=IExtImage, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@ajax_config(name='properties.json',
             context=IExtImage, layer=IPyAMSLayer,
             base=AssociationItemAJAXEditForm)
class ExtImagePropertiesEditForm(ExtFilePropertiesEditForm):
    """External image properties edit form"""

    legend = _("Update image properties")
    icon_css_class = 'fa fa-fw fa-file-image-o'

    fields = field.Fields(IExtImage).select('data', 'filename', 'title', 'description', 'author')
    fields['title'].widgetFactory = I18nExtFileTitleFieldWidget

    def updateWidgets(self, prefix=None):
        super(ExtImagePropertiesEditForm, self).updateWidgets(prefix)
        if 'title' in self.widgets:
            self.widgets['title'].description = None
        if 'description' in self.widgets:
            self.widgets['description'].description = None

    def get_ajax_output(self, changes):
        if changes:
            return self.get_associations_table()
        return super(self.__class__, self).get_ajax_output(changes)


#
# Videos views
#

@viewlet_config(name='extvideos',
                context=IExtFileContainerTarget, layer=IPyAMSLayer, view=IParagraphContainerTable,
                manager=IParagraphTitleToolbar, weight=22,
                permission=VIEW_SYSTEM_PERMISSION)
class ExtVideosCounter(ParagraphContainerCounterBase):
    """External video files counter"""

    weight = 22
    action_class = 'action extvideos nowrap width-40'
    icon_class = 'fa fa-fw fa-file-video-o'
    icon_hint = _("Videos")

    marker_type = 'extvideos'

    @property
    def count(self):
        return len([file for file in IAssociationContainer(self.context).values()
                    if IExtVideo.providedBy(file)])


@viewlet_config(name='add-extvideo.menu',
                context=IExtFileContainerTarget, layer=IPyAMSLayer, view=IAssociationsView,
                manager=IToolbarAddingMenu, weight=62)
class ExtVideoAddMenu(ProtectedFormObjectMixin, ToolbarMenuItem):
    """External video add menu"""

    label = _("Add video")
    label_css_class = 'fa fa-fw fa-file-video-o'

    url = 'add-extvideo.html'
    modal_target = True


@pagelet_config(name='add-extvideo.html', context=IExtFileContainerTarget, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='add-extvideo.json', context=IExtFileContainerTarget, layer=IPyAMSLayer,
             base=AssociationItemAJAXAddForm)
class ExtVideoAddForm(ExtFileAddForm):
    """External video add form"""

    legend = _("Add new video")
    icon_css_class = 'fa fa-fw fa-file-video-o'

    fields = field.Fields(IExtVideo).select('data', 'filename', 'title', 'description',
                                            'author', 'language')
    fields['title'].widgetFactory = I18nExtFileTitleFieldWidget

    def create(self, data):
        factory = EXTERNAL_FILES_FACTORIES.get('video')
        if factory is not None:
            return factory[0]()

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        if output:
            output.setdefault('events', []).append(
                get_json_paragraph_markers_refresh_event(self.context, self.request,
                                                         self, ExtVideosCounter))
        return output


@pagelet_config(name='properties.html',
                context=IExtVideo, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@ajax_config(name='properties.json',
             context=IExtVideo, layer=IPyAMSLayer,
             base=AssociationItemAJAXEditForm)
class ExtVideoPropertiesEditForm(ExtFilePropertiesEditForm):
    """External video properties edit form"""

    legend = _("Update video properties")
    icon_css_class = 'fa fa-fw fa-file-video-o'

    fields = field.Fields(IExtVideo).select('data', 'filename', 'title', 'description',
                                            'author', 'language')
    fields['title'].widgetFactory = I18nExtFileTitleFieldWidget

    def get_ajax_output(self, changes):
        if changes:
            return self.get_associations_table()
        else:
            return super(self.__class__, self).get_ajax_output(changes)


#
# Audio file views
#

@viewlet_config(name='extaudios',
                context=IExtFileContainerTarget, layer=IPyAMSLayer, view=IParagraphContainerTable,
                manager=IParagraphTitleToolbar, weight=23,
                permission=VIEW_SYSTEM_PERMISSION)
class ExtAudiosCounter(ParagraphContainerCounterBase):
    """External audio files counter"""

    weight = 23
    action_class = 'action extaudios nowrap width-40'
    icon_class = 'fa fa-fw fa-file-audio-o'
    icon_hint = _("Audios files")

    marker_type = 'extaudios'

    @property
    def count(self):
        return len([file for file in IAssociationContainer(self.context).values()
                    if IExtAudio.providedBy(file)])


@viewlet_config(name='add-extaudio.menu',
                context=IExtFileContainerTarget, layer=IPyAMSLayer, view=IAssociationsView,
                manager=IToolbarAddingMenu, weight=63)
class ExtAudioAddMenu(ProtectedFormObjectMixin, ToolbarMenuItem):
    """External audio file add menu"""

    label = _("Add audio file")
    label_css_class = 'fa fa-fw fa-file-audio-o'

    url = 'add-extaudio.html'
    modal_target = True


@pagelet_config(name='add-extaudio.html',
                context=IExtFileContainerTarget, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='add-extaudio.json',
             context=IExtFileContainerTarget, layer=IPyAMSLayer,
             base=AssociationItemAJAXAddForm)
class ExtAudioAddForm(ExtFileAddForm):
    """External audio file add form"""

    legend = _("Add new audio file")
    icon_css_class = 'fa fa-fw fa-file-audio-o'

    fields = field.Fields(IExtAudio).select('data', 'filename', 'title',
                                            'description', 'author', 'language')
    fields['title'].widgetFactory = I18nExtFileTitleFieldWidget

    def create(self, data):
        factory = EXTERNAL_FILES_FACTORIES.get('audio')
        if factory is not None:
            return factory[0]()

    def get_ajax_output(self, changes):
        output = super(self.__class__, self).get_ajax_output(changes)
        if output:
            output.setdefault('events', []).append(
                get_json_paragraph_markers_refresh_event(self.context, self.request,
                                                         self, ExtAudiosCounter))
        return output


@pagelet_config(name='properties.html',
                context=IExtAudio, layer=IPyAMSLayer, permission=VIEW_SYSTEM_PERMISSION)
@ajax_config(name='properties.json',
             context=IExtAudio, layer=IPyAMSLayer,
             base=AssociationItemAJAXEditForm)
class ExtAudioPropertiesEditForm(ExtFilePropertiesEditForm):
    """External audio file properties edit form"""

    legend = _("Update audio file properties")
    icon_css_class = 'fa fa-fw fa-file-audio-o'

    fields = field.Fields(IExtAudio).select('data', 'filename', 'title',
                                            'description', 'author', 'language')
    fields['title'].widgetFactory = I18nExtFileTitleFieldWidget

    def get_ajax_output(self, changes):
        if changes:
            return self.get_associations_table()
        else:
            return super(self.__class__, self).get_ajax_output(changes)


#
# External files containers columns
#

@adapter_config(name='created',
                context=(IExtFileContainerTarget, IPyAMSLayer, AssociationsTable),
                provides=IColumn)
class AssociationsTableCreatedColumn(I18nColumn, GetAttrColumn):
    """Associations table creation date column"""

    _header = _("Created")
    weight = 40

    def getValue(self, obj):
        if not IExtFile.providedBy(obj):
            return '--'
        dc = IZopeDublinCore(obj, None)
        if dc is not None:
            return format_datetime(tztime(dc.created), SH_DATETIME_FORMAT, request=self.request)
        return '--'


@adapter_config(name='modified',
                context=(IExtFileContainerTarget, IPyAMSLayer, AssociationsTable),
                provides=IColumn)
class AssociationsTableModifiedColumn(I18nColumn, GetAttrColumn):
    """Associations table modification date column"""

    _header = _("Modified")
    weight = 45

    def getValue(self, obj):
        if not IExtFile.providedBy(obj):
            return '--'
        dc = IZopeDublinCore(obj)
        last = None
        if obj.data:
            for value in obj.data.values():
                if value is None:
                    continue
                file_dc = IZopeDublinCore(value, None)
                if (file_dc is not None) and (file_dc.modified > dc.created):
                    last = max(file_dc.created, file_dc.modified) if last is None \
                        else max(last, file_dc.created, file_dc.modified)
            if last is not None:
                return format_datetime(tztime(last), SH_DATETIME_FORMAT,
                                       request=self.request)
        return '--'
