#
# Copyright (c) 2015-2023 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

__docformat__ = 'restructuredtext'

from datetime import datetime

import iso8601
from pyramid.decorator import reify
from z3c.form.converter import BaseDataConverter
from z3c.form.interfaces import DISPLAY_MODE, IDataConverter, IFieldWidget, INPUT_MODE, NO_VALUE
from z3c.form.widget import FieldWidget, Widget
from zope.interface import implementer_only

from pyams_content.component.calendar.interfaces import IDatetimesWidget
from pyams_content.component.calendar.schema import IDatetimesField
from pyams_form.interfaces import IFormLayer
from pyams_form.widget import widgettemplate_config
from pyams_utils.adapter import adapter_config
from pyams_utils.timezone import gmtime


@adapter_config(required=(IDatetimesField, IDatetimesWidget),
                provides=IDataConverter)
class DatetimesDataConverter(BaseDataConverter):
    """Datetimes field data converter"""

    def toWidgetValue(self, value):
        return value or ()

    def toFieldValue(self, value):
        return [
            gmtime(iso8601.parse_date(dt))
            for dt in value.split(';')
        ]


@widgettemplate_config(mode=INPUT_MODE,
                       layer=IFormLayer,
                       template='templates/datetimes-input.pt')
@widgettemplate_config(mode=DISPLAY_MODE,
                       layer=IFormLayer,
                       template='templates/datetimes-display.pt')
@implementer_only(IDatetimesWidget)
class DatetimesWidget(Widget):
    """Datetimes widget"""

    def extract(self, default=NO_VALUE):
        return super().extract(default) or NO_VALUE

    @reify
    def now(self):
        return gmtime(datetime.utcnow())


@adapter_config(required=(IDatetimesField, IFormLayer),
                provides=IFieldWidget)
def DatetimesFieldWidget(field, request):
    """Datetimes field widget factory"""
    return FieldWidget(field, DatetimesWidget(request))
