#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from persistent import Persistent
from pyramid.events import subscriber
from pyramid.threadlocal import get_current_registry
from zope.container.contained import Contained
from zope.interface import implementer
from zope.lifecycleevent import ObjectModifiedEvent
from zope.lifecycleevent.interfaces import IObjectAddedEvent, IObjectModifiedEvent, IObjectRemovedEvent
from zope.schema.fieldproperty import FieldProperty

from pyams_content.component.association.interfaces import IAssociationContainer, IAssociationContainerTarget, \
    IAssociationItem
from pyams_content.features.json import IJSONConverter, JSONBaseConverter
from pyams_form.interfaces.form import IFormContextPermissionChecker
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import ContextAdapter, adapter_config
from pyams_utils.traversing import get_parent
from pyams_utils.url import absolute_url
from pyams_workflow.content import HiddenContentPublicationInfo
from pyams_workflow.interfaces import IWorkflowPublicationInfo


__docformat__ = 'restructuredtext'


@implementer(IAssociationItem)
class AssociationItem(Persistent, Contained):
    """Base association item persistent class"""

    icon_class = ''
    icon_hint = ''

    visible = FieldProperty(IAssociationItem['visible'])

    def is_visible(self, request=None):
        return self.visible

    def get_url(self, request=None, view_name=None):
        return absolute_url(self, request, view_name)


@adapter_config(context=IAssociationItem,
                provides=IWorkflowPublicationInfo)
def association_item_publication_info(context):
    """Association item publication info"""
    if not context.visible:
        return HiddenContentPublicationInfo()
    return None


@adapter_config(context=IAssociationItem,
                provides=IFormContextPermissionChecker)
@adapter_config(context=IAssociationContainer,
                provides=IFormContextPermissionChecker)
class MenuPermissionChecker(ContextAdapter):
    """Menu permission checker"""

    @property
    def edit_permission(self):
        parent = get_parent(self.context, IAssociationContainerTarget)
        if parent is not None:
            return IFormContextPermissionChecker(parent).edit_permission


@subscriber(IObjectAddedEvent, context_selector=IAssociationItem)
def handle_added_association(event):
    """Handle added association item"""
    content = get_parent(event.object, IAssociationContainerTarget)
    if content is not None:
        get_current_registry().notify(ObjectModifiedEvent(content))


@subscriber(IObjectModifiedEvent, context_selector=IAssociationItem)
def handle_modified_association(event):
    """Handle modified association item"""
    content = get_parent(event.object, IAssociationContainerTarget)
    if content is not None:
        get_current_registry().notify(ObjectModifiedEvent(content))


@subscriber(IObjectRemovedEvent, context_selector=IAssociationItem)
def handle_removed_association(event):
    """Handle removed association item"""
    content = get_parent(event.object, IAssociationContainerTarget)
    if content is not None:
        get_current_registry().notify(ObjectModifiedEvent(content))


@adapter_config(name='associations',
                required=(IAssociationContainerTarget, IPyAMSLayer),
                provides=IJSONConverter)
class JSONAssociationContainerConverter(JSONBaseConverter):
    """JSON association container converter"""

    is_inner = True

    @property
    def conversion_target(self):
        return IAssociationContainer(self.context, None)

    def convert_content(self, params):
        result = super().convert_content(params)
        self.get_list_attribute(IAssociationContainer(self.context).get_visible_items(),
                                params, result, name='items')
        return result.get('items', [])
