(function ($, globals) {

	'use strict';

	var ams = globals.MyAMS;

	globals.ONF = {

		/**
		 * Search engines management
		 */
		search: {

			clearSelection: function (evt) {
				var select = $(evt.currentTarget).data('select2');
				if (select) {
					select.val(null);
				}
			},

			updateSearch: function (evt) {
				var form = $(evt.currentTarget).parents('form');
				if (!form.exists()) {
					form = $('form[id="recherche-results"]');
				}
				if (form.exists()) {
					$('input[name="start"]', form).val(0);
					$('input[name="map_center"]', form).val('');
					$('.opsmap__area_filter').addClass('hidden');
					if (form.data('onf-submit-on-change') === true) {
						var value = $(evt.currentTarget).val();
						if (value || (form.data('onf-submit-on-clear') === true)) {
							form.submit();
						}
					}
				}
			},

			updateSort: function () {
				var form = $('form[id="recherche-results"]');
				if (form.exists()) {
					var index = $(this).data('ams-sort-index');
					$('input[name="order_by"]', form).val(index);
					form.submit();
				}
			},

			updatePageLength: function () {
				var form = $('form[id="recherche-results"]');
				if (form.exists()) {
					var length = $(this).val();
					$('input[name="start"]', form).val(0);
					$('input[name="length"]', form).val(length);
					form.submit();
				}
			},

			previousPage: function () {
				var form = $('form[id="recherche-results"]');
				if (form.exists()) {
					var current = $(this).parents('ol.pagination').data('ams-current-page');
					var length = $('input[name="length"]', form).val();
					$('input[name="start"]', form).val(length * (current - 2));
					form.submit();
				}
			},

			nextPage: function () {
				var form = $('form[id="recherche-results"]');
				if (form.exists()) {
					var current = $(this).parents('ol.pagination').data('ams-current-page');
					var length = $('input[name="length"]', form).val();
					$('input[name="start"]', form).val(length * current);
					form.submit();
				}
			},

			gotoPage: function () {
				var form = $('form[id="recherche-results"]');
				if (form.exists()) {
					var target = parseInt($(this).text());
					var length = $('input[name="length"]', form).val();
					$('input[name="start"]', form).val(length * (target - 1));
					form.submit();
				}
			},

			toggleFacetVisibility: function () {
				var isExpanded = $(this).attr('aria-expanded') === 'true';
				var data = $(this).data();
				$(this).text(isExpanded ? data.amsLessLabel : data.amsMoreLabel);
				$(this).blur();
			},

			toggleFilter: function () {
				let $button = $(this);
				let isPressed = $button.attr('aria-pressed') === 'true';
				$button.attr('aria-pressed', !isPressed);
				if (isPressed) {
					$button.removeClass('pressed');
				} else {
					$button.addClass('pressed');
				}
				ONF.search.updateUrlWithCheckedFilters();
			},

			redirectToUrl: function () {
				const $select = $(this);
				const selectedUrl = $select.val();
				console.log("select", selectedUrl)
				window.location.href = selectedUrl;
			},

			updateUrlWithCheckedFilters: function () {
				var urlParams = new URLSearchParams(window.location.search);
				var filterTypes = new Set($('input[name="filter"], select[name="filter"], button[name="filter"]').map(function () {
					return $(this).data('filter-type');
				}).get());
				filterTypes.forEach(type => urlParams.delete(type));
				if (!urlParams.has('start')) {
					urlParams.append('start', '0');
				}
				$('input[name="filter"]:checked, select[name="filter"]').each(function () {
					var filterType = $(this).data('filter-type');
					var value = $(this).val();
					// Append each selected filter value under its respective filter type
					if (value) {
						urlParams.append(filterType, value);
					}
				});

				$('button[name="filter"]').each(function () {
					var filterType = $(this).data('filter-type');
					var value = $(this).val();
					var isPressed = $(this).attr('aria-pressed') === 'true';

					if (isPressed && value) {
						urlParams.append(filterType, value);
					}
				});
				// Build the new URL with updated parameters
				window.location.href = `${window.location.protocol}//${window.location.host}${window.location.pathname}?${urlParams.toString()}`;
			}
		},

		/**
		 * Communal directory management
		 */
		directory :{

			updateSearch: function(evt) {
				var form = $(evt.currentTarget).parents('form');
				if (!form.exists()) {
					form = $('form[id="recherche-results"]');
				}
				if (form.exists()) {
					if (form.data('onf-submit-on-change') === true) {
						var value = $(evt.currentTarget).val();
						if (value || (form.data('onf-submit-on-clear') === true)) {
							form.submit();
						}
					}
				}
			},

			fetchAndDisplay: async function(evt) {

				const createCard = (agent, label) => {
					const card = $('<div />')
						.addClass('contact-card');
					const cardInner = $('<div />')
						.addClass('contact-card__inner')
						.appendTo(card);
					const cardContent = $('<div />')
						.addClass('contact-card__content')
						.appendTo(cardInner);
					const agentName = $('<p />')
						.appendTo(cardContent);
					$('<strong />').text(agent.agent_name)
						.appendTo(agentName);
					if (label) {
						$('<br />').appendTo(agentName);
						$('<span />').text(label)
							.appendTo(agentName);
					}
					if (agent.agent_phones) {
						const agentPhone = $('<p />').appendTo(cardContent);
						$('<strong />')
							.text('Tél. ')
							.appendTo(agentPhone);
						$.each(agent.agent_phones, (idx, phone) => {
							if (idx > 0) {
								$('<span />')
									.text (' - ')
									.appendTo(agentPhone);
							}
							$('<a />')
								.attr('href', `tel:+33${phone}`)
								.text(phone)
								.appendTo(agentPhone);
						});
					}
					if (agent.agent_email) {
						const agentEmail = $('<p />').appendTo(cardContent);
						$('<a />').attr('href', `mailto:${agent.agent_email}`)
							.addClass('btn-contact')
							.text('Contacter')
							.appendTo(agentEmail);
					} else {
						$('<p />').text('Adresse e-mail non disponible.')
							.appendTo(cardContent);
					}
					return card;
				};
				
				const createFrame = (title, message, cssClass='') => {
					const frame = $('<div />')
						.addClass(`framed ${cssClass}`);
					if (title) {
						$('<h3 />').appendTo(frame);
					}
					if (message) {
						$('<p />').html(message)
							.appendTo(frame);
					}
					return frame;
				};

				evt.preventDefault();

				const inseeCode = $('input[name="query"]').val();
				let result = null;
				// let communeLabel = null;

				try {
					// const inseeUrl = `/api/onf/rest/insee/${inseeCode}`;
					// const inseeResponse = await fetch(inseeUrl);
					//
					// if (!inseeResponse.ok) {
					// 	throw new Error(`Error fetching INSEE data: ${inseeResponse.statusText}`);
					// }
					// const inseeData = await inseeResponse.json();
					// const inseeCommune = inseeData.commune;
					// if (inseeCommune) {
					// 	communeLabel = `${inseeCommune.artmaj || ''}${inseeCommune.artmaj ? ' ' : ''}${inseeCommune?.label}`;

					const dirUrl = `/api/onf/rest/directory/${inseeCode}`;
					const dirResponse = await fetch(dirUrl);

					if (!dirResponse.ok) {
						throw new Error(`Error fetching directory data: ${dirResponse.statusText}`);
					}
					result = await dirResponse.json();
					// } else {
					// 	result = null;
					// }

				} catch (error) {
					console.error(error.message);
					return;
				}

				const container = $('#contact-info');
				container.empty();

				if (!result || (result.status !== 'success')) {
					$('<h3 />').text("Une erreur s'est produite, veuillez essayer avec une autre commune.")
						.appendTo(container);
					return;
				}
				if (result.manager?.agent_id && (result.manager?.agent_id !== result.contact?.agent_id)) {
					createCard(result.manager, result.manager.intitule_poste).appendTo(container);
				}
				if (result.contact?.agent_id) {
					createCard(result.contact, result.manager?.agent_id === result.contact.agent_id ? result.manager.intitule_poste : "Technicien forestier territorial").appendTo(container);
				} else {
					createFrame(null, `Le nom et les coordonnées du technicien forestier territorial pour votre commune s'afficheront prochainement ici. ${result.manager?.agent_id ? "<br /><br />Dans l'attente, n'hésitez pas à contacter le responsable de l'unité territoriale." : ''}`, 'm-5')
						.appendTo(container);
				}
			}
		},

		/**
		 * Forms management
		 */
		form: {

			autoSubmit: function (evt) {
				$(evt.target).parents('form').submit();
			},

			resetFilters: function (evt) {
				var form = $("#recherche-filters");
				$('input[type="text"]', form).val('');
				$('input[name="filter"]', form).prop('checked', false);
				$('select[name="filter"]', form).prop('selectedIndex', 0);
				if ($.fn.select2) {
					$('.select2', form).select2('val', null);
				}

				$('button[name="filter"]').each(function () {
					$(this).attr('aria-pressed', false);
				});

				ONF.search.updateUrlWithCheckedFilters();
			},

			resetAndSubmit: function (evt) {
				var form = $(evt.target).parents('form');
				$('input[type="text"]', form).each(function (idx, elt) {
					$(elt).val('');
				});
				$('.select2', form).each(function (idx, elt) {
					$(elt).select2('val', null);
				});
				form.submit();
			},

			submitHandler: function (form, event) {

				function doSubmit(form) {
					var button = $('button[type="submit"]', form),
						name = button.attr('name'),
						input = $('input[name="' + name + '"]', form);
					if (!input.exists()) {
						$('<input />').attr('type', 'hidden')
							.attr('name', name)
							.attr('value', button.attr('value'))
							.appendTo(form);
					}
					form.submit();
				}

				if (window.grecaptcha) {  // check if recaptcha was loaded
					var captcha_key = $(form).data('ams-form-captcha-key');
					grecaptcha.execute(captcha_key, { action: 'form_submit' }).then(function (token) {
						$('.state-error', form).removeClass('state-error');
						$('input[name="g-recaptcha-response"]', form).val(token);
						doSubmit(form);
					});
				} else {
					doSubmit(form);
				}
			}
		},

		/**
		 * Live forests management
		 */
		live: {

			updateContentType: function (evt) {
				var button = $(evt.target),
					contentType = button.attr('aria-checked') === 'false' ? 'activity' : 'forest',
					form = $('form[id="recherche-results"]'),
					userSearch = $('input[name="user_search"]', form).val(),
					display = $('input[name="display"]', form).val(),
					displayType = display.substring(1, 2) === 'l' ? 'l' : 'm';
				window.location = `${form.attr('action')}?content_type=${contentType}&` +
					`display=${contentType.substring(0, 1)}${displayType}&start=0&user_search=${userSearch}`;
			},

			showActiveMarker: function (evt) {
				var target = $('.spot', evt.currentTarget).data('targetmarker'),
					map = $('.ams-map'),
					marker = map.data('markers')[target];
				if (marker !== undefined) {
					marker.fire('mouseover', { scroll: false });
				}
			},

			hideActiveMarker: function (evt) {
				var target = $('.spot', evt.currentTarget).data('targetmarker'),
					map = $('.ams-map'),
					marker = map.data('markers')[target];
				if (marker !== undefined) {
					marker.fire('mouseout');
				}
			},

			enableAreaFilter: function (evt) {
				var searchForm = $('[id="recherche-results"]'),
					center = $('input[name="map_center"]', searchForm);
				if (center.exists()) {
					$('.opsmap__area_filter').removeClass('hidden');
				}
			},

			applyAreaFilter: function (evt) {
				var searchForm = $('[id="recherche-results"]'),
					center = $('input[name="map_center"]', searchForm);
				if (center.exists()) {
					var map = $('.ams-map').data('leafmap'),
						geo_center = map.getCenter();
					$('input[name="start"]', searchForm).val(0);
					center.val(`${geo_center.lng} ${geo_center.lat}`);
					searchForm.submit();
				}
			}
		},

		/**
		 * Near-me management
		 */
		pdcm: {

			openPopup: function (evt) {
				const
					body = $('body'),
					source = $(evt.currentTarget),
					target = source.data('onf-pdcm-target'),
					popupOverlay = $('<div>').addClass('pdcm-overlay');
				if (target) {
					$('<iframe>').addClass('pdcm-frame')
							.attr('id', 'pdcm-frame')
							.attr('src', target)
							.appendTo(popupOverlay);
					popupOverlay.appendTo(body);
					body.addClass('with-pdcm-overlay');
				}
			},

			closePopup: function(evt) {
				const
					body = $('body', window.parent.document),
					overlay = $('.pdcm-overlay', body);
				overlay.remove();
				body.removeClass('with-pdcm-overlay');
			},

			openHelp: function(evt) {
				$('.pdcm-help').addClass('is-open');
			},

			closeHelp: function(evt) {
				$('.pdcm-help').removeClass('is-open');
			},

			enableAreaFilter: function (evt) {
				var searchForm = $('form[id="recherche-filters"]'),
					location = $('input[name="user_location"]', searchForm);
				if (location.exists()) {
					$('.opsmap__area_filter').removeClass('hidden');
				}
			},

			applyAreaFilter: function (evt) {
				var searchForm = $('form[id="recherche-filters"]'),
					location = $('input[name="user_location"]', searchForm);
				if (location.exists()) {
					var map = $('.ams-map').data('leafmap'),
						geo_center = map.getCenter();
					location.val(`${geo_center.lng} ${geo_center.lat}`);
					searchForm.submit();
				}
			}
		},

		/**
		 * Zabuto calendar plugin for hunting calendars
		 */
		calendar: {

			init: () => {
				$('.calendar').each((idx, elt) => {
					$(elt).on('zabuto:calendar:init', (evt) => {
						setTimeout(ONF.calendar.selectCurrentDate, 500);
					});
					$(elt).on('zabuto:calendar:day', (evt) => {
						const layers = $('.map').data('leafmap.layers');
						if (layers) {
							layers['artemis.contrats'].setStyle({
								color: '#961917',
								fillColor: '#961917'
							});
							layers['artemis.forets'].setStyle({
								color: '#961917',
								fillColor: '#961917'
							});
							if (evt.eventdata) {
								const
									id_contrats = new Array(),
									id_forets = new Array(),
									eventData = evt.eventdata;
								for (const event of (eventData.events || [])) {
									id_contrats.push(`'${event.id_contrat}'`);
									if (!event.has_perimetres) {
										const forestIds = $(elt).data('ams-forest-ids');
										id_forets.push(`'${forestIds}'`);
									}
								}
								layers['artemis.contrats'].setWhere('id_nat_contrat in (' + id_contrats.join(',') + ')');
								if (id_forets.length > 0) {
									layers['artemis.forets'].setWhere('id_nat_frt in (' + id_forets.join(',') + ')');
								} else {
									layers['artemis.forets'].setWhere('0=1');
								}
							} else {
								layers['artemis.contrats'].setWhere('0=1');
								layers['artemis.forets'].setWhere('0=1');
							}
						}
						// buttons switch
						$('.zabuto-calendar td').removeClass('selected');
						$(evt.element).addClass('selected');
					});
					const calendar = $(elt).zabuto_calendar({
						language: 'fr',
						week_starts: 'monday',
						events: $(elt).data('ams-events'),
						first_date: new Date(),
						last_date: new Date($(elt).data('ams-last-date'))
					});
				});
			},

			selectCurrentDate: () => {
				$('.zabuto-calendar__day--today').click();
			}
		}
	};

	/**
	 * Update main menus status
	 */
	$('.main-nav__link', '#menu-principal').each(function () {
		if (window.location.href.startsWith($(this).attr('data-ams-url') || '##')) {
			$(this).parents('li').first().addClass('main-nav__item--current');
		}
	});

	/**
	 * Native sharing handler
	 */
	if (navigator.share) {
		const regularShareWrapper = $('#regularShare');
		const nativeShareWrapper = $('#nativeShare');
		const shareBtn = $('#nativeShareButton');
		const shareResult = $('#nativeShareResult');

		const shareData = {
			title: $('title', 'head').text(),
			text: $('meta[name="description"]', 'head').attr('content'),
			url: $('meta[rel="canonical"]', 'head').attr('href')
		}

		shareBtn.on('click', async () => {
			try {
				await navigator.share(shareData);
				shareResult.text("Partage effectué !")
			} catch (err) {
				shareResult.text("'Une erreur s'est produite : " + err);
			}
		});

		regularShareWrapper.css('display', 'none');
		nativeShareWrapper.css('display', 'block');
	}

	/**
	 * Add wrapper to tables
	 */
	$('table', '.edito').wrap('<div class="table-wrapper"></div>');

	/**
	 * Update all external links target
	 */
	var base = window.location.protocol + '//' + window.location.host;
	$('a[href]').each(function () {
		var href = $(this).attr('href');
		if (!(href.startsWith('/') ||
			href.startsWith('#') ||
			href.startsWith('javascript:') ||
			href.startsWith('mailto:') ||
			href.startsWith(base))) {
			$(this).attr('target', '_blank');
		}
	});

	/**
	 * Update SVG images
	 */
	var svgs = $('.svg-container');
	if (svgs.length > 0) {
		svgs.each(function () {
			var container = $(this);
			var svg = $('svg', container),
				width = svg.attr('width'),
				height = svg.attr('height');
			if (width && height) {
				svg.get(0).setAttribute('viewBox',
					'0 0 ' + Math.round(parseFloat(width)) + ' ' +
					Math.round(parseFloat(height)));
			}
			svg.attr('width', '100%')
				.attr('height', 'auto')
				.removeAttr('preserveAspectRatio');
		})
	}

	/**
	 * Home page video player button
	 */
	var videoBg = $('#homebannerVideoBg');
	if (videoBg.length > 0) {
		var button = $('#videoBgButton');
		var buttonText = $('#videoBgButtonTxt');

		button.on('click', function (e) {
			if (buttonText.hasClass('playing')) {
				videoBg.get(0).pause();
				button.removeClass('homebanner__videoBtn--paused');
				buttonText.removeClass('playing')
					.text(buttonText.data('onf-play-text'));
			} else {
				videoBg.get(0).play();
				button.addClass('homebanner__videoBtn--paused');
				buttonText.addClass('playing')
					.text(buttonText.data('onf-stop-text'));
			}
		});
	}


	// Initialize Masonry
	var msnry;
	var grid = $('.masonry-layout > .portlets');
	if (grid.exists()) {
		ams.require(['error', 'ajax'], function () {
			ams.ajax.check([$.fn.masonry, $.fn.imagesLoaded], [
				'/--static--/onf_website/plugins/masonry.pkgd.min.js',
				'/--static--/onf_website/plugins/imagesloaded.pkgd.min.js'
			], function () {

				var masonryGrid = function () {
					var w = Math.max(
						document.documentElement.clientWidth,
						window.innerWidth || 0
					);
					if (w >= 1035 && !msnry) {
						msnry = grid.masonry({
							itemSelector: '.portlet',
							columnWidth: '.grid-sizer',
							percentPosition: true,
							horizontalOrder: false
						})
					} else if (w < 1035 && msnry) {
						msnry = Masonry.data(grid);
						msnry.destroy();
						msnry = null;
					}
					// Reload Facebook
					if (typeof FB !== 'undefined') {
						FB.XFBML.parse();
					}
				}

				grid.imagesLoaded(masonryGrid);
				window.addEventListener('resize', masonryGrid);
			});
		});
	}


	// Videos sources updates
	//
	// Update videos sources based on current screen resolution
	function resetVideosSources() {
		var videos = $('video');
		if (videos.length > 0) {
			var width;
			if (window.matchMedia === undefined) {
				width = (window.innerWidth > 0) ? window.innerWidth : screen.width
			}
			videos.each(function (idx, elt) {
				var video = $(elt),
					position = elt.currentTime || 0,
					sources = video.data('ams-sources'),
					loaded = {};
				if (sources) {
					video.empty();
					for (var idxSrc in sources) {
						if (!sources.hasOwnProperty(idxSrc)) {
							continue;
						}
						var source = sources[idxSrc],
							type = source.type,
							condition;
						if (loaded[type] !== undefined) {
							continue;
						}
						if (window.matchMedia === undefined) {
							condition = (source.min < width);
							if (condition && source.max) {
								condition = (width <= source.max);
							}
						} else {
							var min = source.min ? '(min-device-width: ' + source.min + 'px)' : '',
								max = source.max ? '(max-device-width: ' + source.max + 'px)' : '',
								query = min + ((min && max) ? ' and ' : '') + max;
							condition = window.matchMedia(query).matches;
						}
						if (condition) {
							$('<source>').attr('type', type)
								.attr('src', source.src)
								.appendTo(video);
							loaded[type] = true;
						}
					}
					elt.currentTime = position;
				}
			});
		}
	}
	resetVideosSources();
	window.addEventListener('resize', resetVideosSources);


	// Initialize main events handlers
	ams.initHandlers(document);


	// Download custom MyAMS modules to handle forms
	ams.require(['plugins-loader'], function () {
		ams.plugins.initData(document);

		// Create rellax extension
		if ($('.rellax', document).exists()) {
			var rellax = new Rellax('.rellax', {
				breakpoints: [720, 990, 1290]
			});
		}
		if ($.fn.tilt) {
			$('[data-tilt]').tilt();
		}

		// Initiate gallery grids
		if (window.MagicGrid) {
			$('.gallery-grid').each((idx, elt) => {
				var magicGrid = new MagicGrid({
					container: elt,
					animate: true,
					gutter: 0,
					static: true,
					useMin: true
				});
				magicGrid.listen();
			});
		}

		// Initialize BaguetteBox
		if (window.baguetteBox) {
			$('.gallery').each((idx, elt) => {
				var gallery = $(elt),
					galleryId = `#${gallery.attr('id')}`;
				var box = baguetteBox.run(galleryId, {
					captions: function (element) {
						var img;
						element = $(element);
						if (element.hasClass('zoom')) {
							img = $('img', element.siblings('picture'));
						} else {
							img = $('img', element);
						}
						var title = img.data('title') || img.attr('alt') || '',
							copyright = img.data('copyright') || '';
						return '<span class="d-block px-3">' + title + ((title && copyright) ? ' - ' : '') + (copyright ? ('<span class="legend__copyright">' + copyright + '</span>') : '') + '</span>';
					}
				});
				$('a.zoom', gallery).on('click', (evt) => {
					baguetteBox.show(0, box[0]);
					return false;
				});
			});
		}

		// Init forms
		var forms = $('form.ams-form', document);
		var required = ['plugins'];
		var html = $('html');
		var lang = html.attr('lang') || html.attr('xml:lang');
		if (lang && !lang.startsWith('en')) {
			ams.lang = lang;
			required.push(ams.baseURL + 'i18n/myams_' + lang.substr(0, 2) + ams.devext + '.js');
		}
		if (forms.exists()) {
			ams.require(['error', 'ajax', 'form', 'events', 'helpers', 'i18n'], function () {
				var required = ['plugins'];
				var html = $('html');
				var lang = html.attr('lang') || html.attr('xml:lang');
				if (lang && !lang.startsWith('en')) {
					ams.lang = lang;
					required.push(ams.baseURL + 'i18n/myams_' + lang.substr(0, 2) + ams.devext + '.js');
				}
				ams.require(required, function () {
					ams.getCSS('/--static--/pyams_default_theme/css/select2.css', 'pyams_select2', function () {
						ams.form.init(document);
						ams.plugins.init(document);
						ams.events.init(document);
					});
				});
			});
		} else {
			var plugins = $('[data-ams-plugins]', document);
			if (plugins.exists()) {
				ams.require(['error', 'ajax', 'events', 'helpers', 'i18n'], function () {
					ams.require(required, function () {
						ams.plugins.init(document);
						ams.events.init(document);
					});
				});
			}
		}
	});

	// Download custom MyAMS modules to handle maps
	var maps = $('div.ams-map', document);
	if (maps.exists()) {
		ams.require([
			'error', 'ajax'
		], function () {
			ams.require([
				window.location.protocol + '//' + window.location.host + '/--static--/pyams_gis/js/pyams_gis.js'
			], function () {
				PyAMS_GIS.init(maps, {}, null, function () {
					PyAMS_GIS.markers.init(maps);
					// init events handlers
					setTimeout(function () {
						maps.each((idx, map) => {
							var data = $(map).data(),
								handlers = data.mapEventsHandlers;
							if (handlers) {
								for (var event in handlers) {
									if (handlers.hasOwnProperty(event)) {
										data.leafmap.on(event, MyAMS.getFunctionByName(handlers[event]));
									}
								}
							}
						});
					}, 1000);
				});
			});
		});
	}

// let isActive = false;
// let formCenter = [47.0901987, 0];
// let lastFocusedElement;
//
// function getUrlParameter(name) {
//     name = name.replace(/[\[]/, '\\[').replace(/[\]]/, '\\]');
//     var regex = new RegExp('[\\?&]' + name + '=([^&#]*)');
//     var results = regex.exec(location.search);
//     return results === null ? '' : decodeURIComponent(results[1].replace(/\+/g, ' '));
// }
//
// function setUrlParameter(key, value) {
//     const url = new URL(window.location.href);
//     url.searchParams.set(key, value);
//     window.history.pushState({}, '', url);
// }
//
// function removeUrlParameter(key) {
//     const url = new URL(window.location.href);
//     url.searchParams.delete(key);
//     window.history.pushState({}, '', url);
// }
//
// function resetState(){
// 	$('.step3').hide();
// 	$('.step2').show();
// 	$('.step2 .gotoStep2').hide();
// 	$('.step2 .gotoStep3').show();
// 	$('.pres-de-chez-moi-map').removeClass("pres-de-chez-moi-map--step3")
// }
//
// function toggleModal(show) {
//     isActive = show !== undefined ? show : !isActive;
//     if (isActive) {
//         lastFocusedElement = document.activeElement;
//         $('#presDeChezMoiModal').attr('aria-hidden', 'false').addClass('active');
//         $('#closeModal').focus();
//         trapFocus($('#presDeChezMoiModal'));
//         setUrlParameter('modal', 'open');
//     } else {
//         $('#presDeChezMoiModal').attr('aria-hidden', 'true').removeClass('active');
//         if (lastFocusedElement) {
//             lastFocusedElement.focus();
//         }
//         resetState()
//         removeUrlParameter('modal');
//     }
// }
//
// function checkUrlState() {
//     const modalState = getUrlParameter('modal');
//     if (modalState === 'open') {
//         toggleModal(true);
//     } else {
//         toggleModal(false);
//     }
// }
//
// function trapFocus(element) {
//     const focusableElements = element.find('button, [href], input, select, textarea, [tabindex]:not([tabindex="-1"])');
//     const firstFocusableElement = focusableElements[0];
//     const lastFocusableElement = focusableElements[focusableElements.length - 1];
//
//     element.on('keydown', function(e) {
//         if (e.key === 'Tab') {
//             if (e.shiftKey) {
//                 if (document.activeElement === firstFocusableElement) {
//                     e.preventDefault();
//                     lastFocusableElement.focus();
//                 }
//             } else {
//                 if (document.activeElement === lastFocusableElement) {
//                     e.preventDefault();
//                     firstFocusableElement.focus();
//                 }
//             }
//         }
//     });
// }
//
// function gotoStep(stepVal) {
//     switch (stepVal) {
//         case 1:
//             $('.step3').hide();
//             $('.step2').show();
//             break;
//         case 2:
//             $('.step3').hide();
//             $('.step2').show();
// 			$('.step2 .gotoStep2').hide();
// 			$('.step2 .gotoStep3').show();
// 			$('.pres-de-chez-moi-map').removeClass("pres-de-chez-moi-map--step3")
//             break;
//         case 3:
//             $('.step3').show();
// 			$('.step2 .gotoStep3').hide();
//             $('.step2 .gotoStep2').show();
// 			$('.pres-de-chez-moi-map').addClass("pres-de-chez-moi-map--step3")
//             break;
//         default:
//             $('.step3').hide();
//             $('.step2').show();
//     }
// }
//
// checkUrlState();
//
// $(document).ready(function() {
//     $('#toggleActive').click(() => toggleModal(true));
//     $('#closeModal').click(() => toggleModal(false));
//     $('#gotoStep3').click(() => gotoStep(3));
//     $('#gotoStep2').click(() => gotoStep(2));
//     $('#closeStep3').click(() => gotoStep(2));
//
//     $('#resetForm').click(function() {
//         console.log('Reset!');
//         gotoStep(1);
//     });
//
//     $('#openMore, #openMore2').click(function() {
//         $(this).hide().next('ul').removeClass('hidden');
//     });
//
//     $('#searchForm, #mapSearchForm').submit(function(event) {
//         event.preventDefault();
//         console.log(this.id + ' submitted!');
//     });
//
//     $('#toponymeField').on('input', function() {
//         const query = $(this).val();
//         if (query.length > 4) {
//             $.get('https://nominatim.openstreetmap.org/search', {
//                 q: query + ' france',
//                 format: 'jsonv2'
//             }, function(data) {
//                 const results = data.map(el => `<li><a href="javascript:;" data-lat="${el.lat}" data-lon="${el.lon}">${el.display_name}</a></li>`).join('');
//                 $('#pdcmResults').html(results).addClass('active');
//             });
//         }
//     });
//
//     $('#pdcmResults').on('click', 'a', function() {
//         formCenter = [$(this).data('lat'), $(this).data('lon')];
//         $('#toponymeField').val($(this).text());
//         $('#pdcmResults').removeClass('active');
//         console.log(formCenter);
//     });
//
// 	$('#open-filter').on("click", function(){
// 		resetState()
// 	});
//
//     // Close modal with Escape key
//     $(document).on('keydown', function(e) {
//         if (e.key === 'Escape' && isActive) {
//             toggleModal(false);
//         }
//     });
//
//     // Close modal when clicking outside
//     $('#presDeChezMoiModal').on('click', function(e) {
//         if (e.target === this) {
//             toggleModal(false);
//         }
//     });
// });

})(jQuery, this);
