#
# Copyright (c) 2015-2021 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""ONF_website.shared.view.skin.map module

This module defines a custom renderer which can be used to render view items as
map markers.
"""

import json

from zope.interface import Interface
from zope.schema.fieldproperty import FieldProperty

from onf_website.component.location.skin.map import BaseMapMarkersRendererMixin
from onf_website.shared.view.skin.interfaces import IONFViewItemsPortletMapRendererSettings
from pyams_content.shared.view.portlet import IViewItemsPortletSettings
from pyams_default_theme.shared.view.portlet import BaseViewItemsPortletRenderer
from pyams_file.property import FileProperty
from pyams_gis.configuration import MapConfiguration
from pyams_gis.interfaces.configuration import IMapConfiguration
from pyams_gis.interfaces.utility import IMapManager
from pyams_portal.interfaces import IPortalContext, IPortletRenderer
from pyams_skin.layer import IPyAMSLayer, IPyAMSUserLayer
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config
from pyams_utils.factory import factory_config
from pyams_utils.registry import get_utility


__docformat__ = 'restructuredtext'

from onf_website import _


@factory_config(provided=IONFViewItemsPortletMapRendererSettings)
class ONFViewItemsPortletMapRendererSettings(MapConfiguration):
    """View items portlet renderer settings"""

    _use_default_map_configuration = FieldProperty(
        IONFViewItemsPortletMapRendererSettings['use_default_map_configuration'])

    map_height = FieldProperty(IONFViewItemsPortletMapRendererSettings['map_height'])
    display_illustrations = FieldProperty(
        IONFViewItemsPortletMapRendererSettings['display_illustrations'])
    marker_image = FileProperty(IONFViewItemsPortletMapRendererSettings['marker_image'])
    display_clusters = FieldProperty(
        IONFViewItemsPortletMapRendererSettings['display_clusters'])

    @property
    def use_default_map_configuration(self):
        return self._use_default_map_configuration

    @use_default_map_configuration.setter
    def use_default_map_configuration(self, value):
        self._use_default_map_configuration = value

    @property
    def no_use_default_map_configuration(self):
        return not bool(self.use_default_map_configuration)

    @no_use_default_map_configuration.setter
    def no_use_default_map_configuration(self, value):
        self.use_default_map_configuration = not bool(value)

    @property
    def configuration(self):
        if self.use_default_map_configuration:
            manager = get_utility(IMapManager)
            return IMapConfiguration(manager).get_configuration()
        return self.get_configuration()


@adapter_config(name='ONF::view.map',
                context=(IPortalContext, IPyAMSLayer, Interface, IViewItemsPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/map.pt', layer=IPyAMSUserLayer)
class ONFViewItemsPortletMapRenderer(BaseMapMarkersRendererMixin, BaseViewItemsPortletRenderer):
    """View items portlet map renderer"""

    label = _("ONF: map view with markers")
    weight = 30

    settings_interface = IONFViewItemsPortletMapRendererSettings

    def get_markers(self, results, marker_image=None):
        return super().get_markers(results, self.renderer_settings.marker_image)
