#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from persistent import Persistent
from zope.container.contained import Contained
from zope.interface import Interface
from zope.intid import IIntIds
from zope.schema.fieldproperty import FieldProperty

from onf_website.shared.common.skin.card import SharedContentCard
from onf_website.shared.common.skin.interfaces import IContentCard
from onf_website.shared.site.skin.interfaces import \
    ISiteContainerSummaryPortletAccordionRendererSettings
from onf_website.skin.public.interfaces import IBodyClass
from onf_website.skin.public.layer import IONFBaseLayer
from pyams_content.shared.site.interfaces import ISiteContainer, ISiteFolder, ISiteManager
from pyams_content.shared.site.portlet import ISiteContainerSummaryPortletSettings
from pyams_default_theme.features.search.portlet import HIDDEN_HEADER_DISPLAY, \
    START_HEADER_DISPLAY
from pyams_default_theme.interfaces import IContentSummaryInfo
from pyams_default_theme.shared.site.portlet import SiteContainerSummaryPortletDefaultRenderer
from pyams_i18n.interfaces import II18n
from pyams_portal.interfaces import IPortalContext, IPortletRenderer
from pyams_portal.portlet import PortletRenderer
from pyams_skin.interfaces.viewlet import IBreadcrumbs
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.viewlet.breadcrumb import BreadcrumbsAdapter
from pyams_template.template import override_template, template_config
from pyams_utils.adapter import adapter_config
from pyams_utils.factory import factory_config
from pyams_utils.registry import get_utility
from pyams_utils.text import get_text_start


__docformat__ = 'restructuredtext'

from onf_website import _


@adapter_config(context=(ISiteContainer, IONFBaseLayer, Interface),
                provides=IBreadcrumbs)
class SiteContainerBreadcrumbsAdapter(BreadcrumbsAdapter):
    """Site container breadcrumbs adapter

    Use default breadcrumbs adapter.
    """


@adapter_config(context=(ISiteManager, IONFBaseLayer),
                provides=IBodyClass)
def site_manager_body_class_adapter(context, request):
    return 'front'


@adapter_config(context=(ISiteManager, IONFBaseLayer, Interface),
                provides=IContentCard)
@adapter_config(context=(ISiteFolder, IONFBaseLayer, Interface),
                provides=IContentCard)
class SiteManagerContentCard(SharedContentCard):
    """Site manager content card"""

    show_tag = False


override_template(context=SiteContainerSummaryPortletDefaultRenderer, layer=IONFBaseLayer,
                  template='templates/site-summary.pt')


#
# Custom site container renderer using accordion
#

@factory_config(ISiteContainerSummaryPortletAccordionRendererSettings)
class SiteContainerSummaryPortletAccordionRendererSettings(Persistent, Contained):
    """Site container summary portlet accordion renderer settings"""

    folders = FieldProperty(ISiteContainerSummaryPortletAccordionRendererSettings['folders'])
    opened_folders = FieldProperty(
        ISiteContainerSummaryPortletAccordionRendererSettings['opened_folders'])
    inner_css_class = FieldProperty(
        ISiteContainerSummaryPortletAccordionRendererSettings['inner_css_class'])
    header_display_mode = FieldProperty(
        ISiteContainerSummaryPortletAccordionRendererSettings['header_display_mode'])
    max_header_length = FieldProperty(
        ISiteContainerSummaryPortletAccordionRendererSettings['max_header_length'])


@adapter_config(name='summary-accordion',
                context=(IPortalContext, IPyAMSLayer, Interface,
                         ISiteContainerSummaryPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/site-accordion.pt', layer=IPyAMSLayer)
class SiteContainerSummaryPortletAccordionRenderer(PortletRenderer):
    """Site container summary portlet accordion renderer"""

    label = _("ONF: Site container summary with accordion")
    weight = 4

    settings_interface = ISiteContainerSummaryPortletAccordionRendererSettings

    def get_folders(self, container=None):
        if container is None:
            container = ISiteContainer(self.context, None)
        if container is None:
            return
        registry = self.request.registry
        settings = self.renderer_settings
        intids = get_utility(IIntIds)
        for item in container.get_visible_items(self.request):
            if not ISiteFolder.providedBy(item):
                continue
            item_id = intids.queryId(item)
            if (not settings.folders) or (item_id in settings.folders):
                info = registry.queryMultiAdapter((item, self.request), IContentSummaryInfo)
                if info is not None:
                    yield info, item_id in (settings.opened_folders or ())

    def get_visible_items(self, parent):
        container = ISiteContainer(parent, None)
        if container is not None:
            registry = self.request.registry
            yield from filter(lambda x: x is not None,
                              map(lambda x: registry.queryMultiAdapter((x, self.request),
                                                                       IContentSummaryInfo),
                                  parent.get_visible_items(self.request)))

    def get_header(self, item):
        settings = self.renderer_settings
        display_mode = settings.header_display_mode
        if display_mode == HIDDEN_HEADER_DISPLAY:
            return ''
        header = II18n(item).query_attribute('header', request=self.request)
        if display_mode == START_HEADER_DISPLAY:
            header = get_text_start(header, settings.max_header_length)
        return header
