#
# Copyright (c) 2015-2022 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

__docformat__ = 'restructuredtext'

from z3c.form import field
from zope.intid import IIntIds

from onf_website import _
from onf_website.shared.planning.interfaces import IPlanningSearchFolder
from onf_website.shared.planning.search import PlanningSearchFolder
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION, MANAGE_SITE_PERMISSION
from pyams_content.shared.common.zmi.properties import SharedContentPropertiesEditForm
from pyams_content.shared.site.interfaces import ISiteContainer
from pyams_content.shared.site.zmi import SiteManagerFoldersSelectorFieldWidget
from pyams_content.shared.site.zmi.folder import ISiteFolderAddFormFields
from pyams_content.zmi.interfaces import ISiteTreeTable
from pyams_form.form import AJAXAddForm, ajax_config
from pyams_form.interfaces.form import IInnerSubForm
from pyams_i18n.interfaces import II18n, INegotiator
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.interfaces.viewlet import IToolbarAddingMenu
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.viewlet.toolbar import ToolbarMenuItem
from pyams_utils.adapter import NullAdapter, adapter_config
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION
from pyams_utils.registry import get_utility
from pyams_utils.unicode import translate_string
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogAddForm
from pyams_zmi.interfaces.menu import IPropertiesMenu
from pyams_zmi.layer import IAdminLayer


@viewlet_config(name='add-planning-search-folder.menu',
                context=ISiteContainer, layer=IAdminLayer,
                view=ISiteTreeTable, manager=IToolbarAddingMenu,
                permission=MANAGE_SITE_PERMISSION, weight=95)
class PlanningSearchFolderAddMenu(ToolbarMenuItem):
    """Search folder add menu"""

    label = _("Add planning search folder")
    label_css_class = 'fa fa-fw fa-search'
    url = 'add-planning-search-folder.html'
    modal_target = True


@pagelet_config(name='add-planning-search-folder.html',
                context=ISiteContainer, layer=IPyAMSLayer,
                permission=MANAGE_SITE_PERMISSION)
@ajax_config(name='add-planning-search-folder.json',
             context=ISiteContainer, layer=IPyAMSLayer, base=AJAXAddForm)
class PlanningSearchFolderAddForm(AdminDialogAddForm):
    """Search folder add form"""

    @property
    def title(self):
        return II18n(self.context).query_attribute('title', request=self.request)

    legend = _("Add planning search folder")
    icon_css_class = 'fa fa-fw fa-search'

    fields = field.Fields(ISiteFolderAddFormFields)
    fields['parent'].widgetFactory = SiteManagerFoldersSelectorFieldWidget

    edit_permission = MANAGE_SITE_PERMISSION

    def updateWidgets(self, prefix=None):
        super(PlanningSearchFolderAddForm, self).updateWidgets(prefix)
        if 'parent' in self.widgets:
            self.widgets['parent'].permission = MANAGE_SITE_PERMISSION

    def create(self, data):
        return PlanningSearchFolder()

    def update_content(self, content, data):
        data = data.get(self, data)
        # initialize
        content.title = data['title']
        content.short_name = data['title']
        content.notepad = data['notepad']
        intids = get_utility(IIntIds)
        parent = intids.queryObject(data.get('parent'))
        if parent is not None:
            negotiator = get_utility(INegotiator)
            title = II18n(content).get_attribute('title', lang=negotiator.server_language)
            name = translate_string(title, force_lower=True, spaces='-')
            if name in parent:
                index = 1
                new_name = '{name}-{index:02}'.format(name=name, index=index)
                while new_name in parent:
                    index += 1
                    new_name = '{name}-{index:02}'.format(name=name, index=index)
                name = new_name
            parent[name] = content

    def add(self, content):
        pass

    def nextURL(self):
        return absolute_url(self.context, self.request, 'admin#site-tree.html')

    def get_ajax_output(self, changes):
        return {
            'status': 'reload'
        }


@pagelet_config(name='properties.html',
                context=IPlanningSearchFolder, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@ajax_config(name='properties.json',
             context=IPlanningSearchFolder, layer=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION)
class PlanningSearchFolderPropertiesEditForm(SharedContentPropertiesEditForm):
    """Planning search folder properties edit form"""

    def updateWidgets(self, prefix=None):
        # don't override base widgets update method
        super(SharedContentPropertiesEditForm, self).updateWidgets(prefix)


@adapter_config(name='view-settings',
                context=(IPlanningSearchFolder, IPyAMSLayer, SharedContentPropertiesEditForm),
                provides=IInnerSubForm)
class PlanningSearchFolderSettingsEditSubform(NullAdapter):
    """Planning search folder hidden settings edit sub-form extension"""


@viewlet_config(name='hearing.menu',
                context=IPlanningSearchFolder, layer=IAdminLayer,
                manager=IPropertiesMenu)
class PlanningSearchFolderHearingMenu(NullAdapter):
    """Planning search folder hidden hearing menu"""


@viewlet_config(name='location.menu',
                context=IPlanningSearchFolder, layer=IAdminLayer,
                manager=IPropertiesMenu, permission=VIEW_SYSTEM_PERMISSION)
class PlanningSearchFolderLocationMenu(NullAdapter):
    """Planning search folder hidden location menu"""


@viewlet_config(name='tags.menu',
                context=IPlanningSearchFolder, layer=IAdminLayer,
                manager=IPropertiesMenu, permission=VIEW_SYSTEM_PERMISSION)
class PlanningSearchFolderTagsMenu(NullAdapter):
    """Planning search folder hidden tags menu"""


@viewlet_config(name='themes.menu',
                context=IPlanningSearchFolder, layer=IAdminLayer,
                manager=IPropertiesMenu, permission=VIEW_SYSTEM_PERMISSION)
class PlanningSearchFolderThemesMenu(NullAdapter):
    """Planning search folder hidden themes menu"""


@viewlet_config(name='collections.menu',
                context=IPlanningSearchFolder, layer=IAdminLayer,
                manager=IPropertiesMenu, permission=VIEW_SYSTEM_PERMISSION)
class PlanningSearchFolderCollectionsMenu(NullAdapter):
    """Planning search folder hidden collections menu"""
