
"""ONF_website.shared.planning.manager module

"""

from pyramid.events import subscriber
from pyramid.threadlocal import get_current_registry
from zope.component.interfaces import ISite
from zope.interface import implementer
from zope.lifecycleevent import ObjectAddedEvent
from zope.lifecycleevent.interfaces import IObjectAddedEvent
from zope.schema.fieldproperty import FieldProperty

from onf_website.features.zfiles import IZFilesClientTarget
from onf_website.shared.forest.interfaces import DOM_ILLUSTRATION_KEY
from onf_website.shared.planning import Planning
from onf_website.shared.planning.interfaces import IPlanningManager, IPlanningManagerFactory, \
    PLANNING_CONTENT_TYPE
from pyams_content.component.illustration import IIllustration, IIllustrationTarget
from pyams_content.component.paragraph import IParagraphFactorySettings
from pyams_content.component.theme import ITagsTarget
from pyams_content.component.theme import IThemesManagerTarget
from pyams_content.reference.pictograms.interfaces import IPictogramManagerTarget
from pyams_content.shared.common import ISharedSite
from pyams_content.shared.common.interfaces import ISharedContentFactory
from pyams_content.shared.common.manager import SharedTool
from pyams_sequence.reference import get_reference_target
from pyams_utils.adapter import adapter_config, get_annotation_adapter
from pyams_utils.registry import get_current_registry, query_utility, utility_config
from pyams_utils.traversing import get_parent


__docformat__ = 'restructuredtext'


@implementer(IPlanningManager, IParagraphFactorySettings, IIllustrationTarget,
             IThemesManagerTarget, IPictogramManagerTarget,
             ITagsTarget, IZFilesClientTarget)
class PlanningManager(SharedTool):
    """Planning manager class"""

    shared_content_type = PLANNING_CONTENT_TYPE

    allowed_paragraphs = FieldProperty(IParagraphFactorySettings['allowed_paragraphs'])
    auto_created_paragraphs = FieldProperty(IParagraphFactorySettings['auto_created_paragraphs'])

    default_owner = FieldProperty(IPlanningManager['default_owner'])
    default_header = FieldProperty(IPlanningManager['default_header'])
    default_hearing_source = FieldProperty(IPlanningManager['default_hearing_source'])
    default_source_folder = FieldProperty(IPlanningManager['default_source_folder'])

    msg_expired = FieldProperty(IPlanningManager['msg_expired'])
    msg_expired_3y = FieldProperty(IPlanningManager['msg_expired_3y'])
    msg_no_doc = FieldProperty(IPlanningManager['msg_no_doc'])
    msg_missing_docs = FieldProperty(IPlanningManager['msg_missing_docs'])

    def get_default_source(self):
        """Get default hearing source"""
        target = get_reference_target(self.default_source_folder)
        if target is None:
            target = query_utility(ISharedSite, name=self.default_hearing_source)
        return target

    templates = FieldProperty(IPlanningManager['templates'])


@utility_config(provides=IPlanningManagerFactory)
class PlanningManagerFactory:
    """Planning manager factory"""

    def __new__(cls):
        return PlanningManager


@adapter_config(context=IPlanningManager, provides=ISharedContentFactory)
def planning_manager_content_factory(context):
    return Planning


@subscriber(IObjectAddedEvent, context_selector=IPlanningManager)
def handle_added_planning_manager(event):
    """Register planning manager when added"""
    site = get_parent(event.newParent, ISite)
    registry = site.getSiteManager()
    if registry is not None:
        registry.registerUtility(event.object, IPlanningManager)


@adapter_config(name='dom',
                context=IPlanningManager, provides=IIllustration)
def planning_manager_dom_illustration_factory(context):
    """Planning manager DOM illustration factory"""

    def illustration_callback(illustration):
        get_current_registry().notify(ObjectAddedEvent(illustration, context,
                                                       illustration.__name__))

    return get_annotation_adapter(context, DOM_ILLUSTRATION_KEY, IIllustration,
                                  name='++illustration++dom',
                                  callback=illustration_callback)
