#
# Copyright (c) 2015-2021 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

from zope.interface import Interface

from onf_website.reference.hunting.model import GestionDate, SaisonChasse
from onf_website.shared.forest import IForestManager
from onf_website.shared.hunting import FakeHuntingCalendar, IHuntingCalendarInfo, IHuntingManager, IWfHuntingCalendar, \
    MissingHuntingCalendar
from onf_website.skin.public.layer import IONFBaseLayer
from pyams_default_theme.shared.common.interfaces import ISharedContentHeadViewletManager
from pyams_i18n.interfaces import II18n
from pyams_portal.interfaces import IPortalPage, IPortalTemplate
from pyams_portal.page import PortalPage
from pyams_skin.interfaces.metas import IHTMLContentMetas
from pyams_skin.layer import IPyAMSUserLayer
from pyams_skin.metas import HTMLTagMeta
from pyams_template.template import template_config
from pyams_utils.adapter import ContextAdapter, ContextRequestAdapter, ContextRequestViewAdapter, adapter_config
from pyams_utils.interfaces.url import ICanonicalURL, IRelativeURL
from pyams_utils.registry import get_utility, query_utility
from pyams_utils.unicode import translate_string
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import Viewlet, viewlet_config

__docformat__ = 'restructuredtext'

from onf_website import _


@viewlet_config(name='hunting-head',
                context=IWfHuntingCalendar, layer=IONFBaseLayer, view=Interface,
                manager=ISharedContentHeadViewletManager, weight=1)
@template_config(template='templates/head-specificities.pt', layer=IONFBaseLayer)
class HuntingCalendarHeadViewlet(Viewlet):
    """Hunting calendar head viewlet"""

    @property
    def info(self):
        if isinstance(self.context, FakeHuntingCalendar):
            season = SaisonChasse.get_current_or_next()
            if season is not None:
                translate = self.request.localizer.translate
                manager = get_utility(IHuntingManager)
                return translate(_("{season_prefix} {season_label}")).format(
                        season_prefix=II18n(manager).query_attribute('season_prefix',
                                                                     request=self.request),
                        season_label=season.saison)
        info = IHuntingCalendarInfo(self.context)
        return II18n(info).query_attribute('season', request=self.request)


@adapter_config(required=FakeHuntingCalendar,
                provides=IPortalPage)
class FakeHuntingCalendarPortalPage(ContextAdapter, PortalPage):
    """Fake forest portal page"""

    can_inherit = False
    use_local_template = True

    @property
    def local_template(self):
        manager = get_utility(IHuntingManager)
        if manager is None:
            return None
        template = manager.default_template
        if template is not None:
            return query_utility(IPortalTemplate, name=template)
        return IPortalPage(manager).template


@adapter_config(required=MissingHuntingCalendar,
                provides=IPortalPage)
class MissingHuntingCalendarPortalPage(ContextAdapter, PortalPage):
    """Missing forest portal page"""

    can_inherit = False
    use_local_template = True

    @property
    def local_template(self):
        manager = get_utility(IHuntingManager)
        template = manager.missing_info_template
        if template is not None:
            return query_utility(IPortalTemplate, name=template)
        return IPortalPage(manager).template


@adapter_config(name='title',
                context=(FakeHuntingCalendar, IPyAMSUserLayer, Interface),
                provides=IHTMLContentMetas)
class FakeHuntingCalendarTitleMetasAdapter(ContextRequestViewAdapter):
    """Title metas adapter"""

    order = 1

    def get_metas(self):
        """Get HTML metas headers"""
        yield HTMLTagMeta('title', self.context.title)


@adapter_config(context=(FakeHuntingCalendar, IPyAMSUserLayer),
                provides=ICanonicalURL)
class FakeHuntingCalendarCanonicalURL(ContextRequestAdapter):
    """Fake forest canonical URL adapter"""

    def get_url(self, view_name=None, query=None):
        context = None
        manager = query_utility(IForestManager)
        if manager is not None:
            context = manager.get_default_source()
        if context is None:
            return '{}/+cc/{}::{}.html'.format(absolute_url(self.request.root, self.request),
                                               ''.join(self.context.forest_ids).lower(),
                                               translate_string(self.context.title, spaces='-'))
        return '{}/++cc++{}::{}.html'.format(absolute_url(context, self.request),
                                             ''.join(self.context.forest_ids).lower(),
                                             translate_string(self.context.title, spaces='-'))


@adapter_config(context=(FakeHuntingCalendar, IPyAMSUserLayer),
                provides=IRelativeURL)
class FakeHuntingCalendarRelativeURL(ContextRequestAdapter):
    """Fake forest relative URL adapter"""

    def get_url(self, display_context=None, view_name=None, query=None):
        translate = self.request.localizer.translate
        return '{}/++cc++{}::{}.html'.format(absolute_url(display_context, self.request),
                                             ''.join(self.context.forest_ids).lower(),
                                             translate_string(self.context.title, spaces='-'))
