#
# Copyright (c) 2015-2021 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

from persistent import Persistent
from pyramid.decorator import reify
from zope.container.contained import Contained
from zope.interface import implementer, provider
from zope.schema.fieldproperty import FieldProperty

from onf_website.component.hearing import HearingInfo, IHearingInfo, IHearingTarget
from onf_website.component.location import ILocationInfo, ILocationTarget, LocationInfo
from onf_website.reference.forest.model.foret import Foret, InformationForet, \
    PARENT_SESSION as RDF_SESSION, ProprietaireForet
from onf_website.shared.hunting.interfaces import HUNTING_CONTENT_NAME, HUNTING_CONTENT_TYPE, \
    HUNTING_INFO_ANNOTATIONS_KEY, IHuntingCalendar, IHuntingCalendarInfo, IHuntingManager, IWfHuntingCalendar, \
    IWfHuntingCalendarFactory
from pyams_alchemy.engine import get_user_session
from pyams_content.component.illustration import IIllustrationTarget, ILinkIllustrationTarget
from pyams_content.component.paragraph import IParagraphContainerTarget
from pyams_content.component.theme import ITagsInfo, ITagsTarget, IThemesTarget
from pyams_content.features.preview.interfaces import IPreviewTarget
from pyams_content.features.review import IReviewTarget
from pyams_content.shared.common import IWfSharedContentFactory, SharedContent, WfSharedContent, \
    register_content_type
from pyams_i18n.interfaces import II18n
from pyams_utils.adapter import ContextAdapter, adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config
from pyams_utils.interfaces import ICacheKeyValue
from pyams_utils.interfaces.url import DISPLAY_CONTEXT
from pyams_utils.registry import get_utility, query_utility
from pyams_utils.request import check_request, query_request
from pyams_utils.timezone import tztime
from pyams_workflow.content import WorkflowContentPublicationInfo
from pyams_workflow.interfaces import IWorkflowPublicationInfo

__docformat__ = 'restructuredtext'

from onf_website import _


HUNTING_REFS_KEY = 'onf_website.forest.refs'
HUNTING_REFS_MARKER = object()


def initlow(value):
    """Return value with first letter in lowercase"""
    if not value:
        return ''
    return '{}{}'.format(value[0].lower(), value[1:])
    
    
class BaseHuntingCalendarMixin:
    """Base hunting calendar mixin class"""

    content_type = HUNTING_CONTENT_TYPE
    content_name = HUNTING_CONTENT_NAME

    
@implementer(IWfHuntingCalendar, ILocationTarget, IHearingTarget)
class FakeHuntingCalendar(BaseHuntingCalendarMixin):
    """Fake hunting calendar"""
    
    forest_ids = None
    
    def __init__(self, forest_ids=None):
        super().__init__()
        if forest_ids:
            if isinstance(forest_ids, str):
                forest_ids = [forest_ids]
            self.forest_ids = forest_ids

    @property
    def __parent__(self):
        manager = get_utility(IHuntingManager)
        return manager.get_default_source()
        
    @property
    def __name__(self):
        return '++cc++{}'.format(''.join(self.forest_ids))
    
    @reify
    def title(self):
        manager = get_utility(IHuntingManager)
        request = check_request()
        prefix = II18n(manager).query_attribute('title_prefix', request=request)
        translate = request.localizer.translate
        return translate(_("{prefix} {label}")).format(
                prefix=prefix,
                label = ''.join(set(
                    initlow((info.libelle if info is not None else None) or foret.libelle_usage)
                    for foret, prop, info in self.forest_refs
                )))

    description = None
    baseline_draft = None
    baseline_principal = None
    baseline_timestamp = None
    baseline_published = None

    @reify
    def header(self):
        manager = get_utility(IHuntingManager)
        return II18n(manager).query_attribute('default_header')

    def get_forest_refs(self):
        session = get_user_session(RDF_SESSION)
        return session.query(Foret, ProprietaireForet, InformationForet) \
            .join(ProprietaireForet,
                  Foret.id_nat_frt == ProprietaireForet.id_nat_frt) \
            .outerjoin(InformationForet,
                       Foret.id_nat_frt == InformationForet.id_nat_frt) \
            .filter(Foret.id_nat_frt.in_(self.forest_ids)) \
            .order_by(ProprietaireForet.pourcentage_part, ProprietaireForet.categorie)

    @property
    def forest_refs(self):
        request = query_request()
        is_display_context = (request is not None) and \
                             (request.annotations.get(DISPLAY_CONTEXT) is self)
        refs = HUNTING_REFS_MARKER
        if is_display_context:
            refs = request.annotations.get(HUNTING_REFS_KEY, HUNTING_REFS_MARKER)
        if refs is HUNTING_REFS_MARKER:
            refs = self.get_forest_refs().all()
            if is_display_context:
                request.annotations[HUNTING_REFS_KEY] = refs
        return refs


class MissingHuntingCalendar(FakeHuntingCalendar):
    """Missing hunting calendar"""
    

@adapter_config(required=FakeHuntingCalendar,
                provides=ICacheKeyValue)
def fake_hunting_calendar_cache_key(context):
    """Fake hunting calendar cache key adapter"""
    return 'CC::{}'.format(','.join(context.forest_ids))

    
@factory_config(IWfHuntingCalendar)
@implementer(IWfHuntingCalendar, IIllustrationTarget, ILinkIllustrationTarget,
             IParagraphContainerTarget, ILocationTarget, IHearingTarget, ITagsTarget,
             IThemesTarget, IPreviewTarget, IReviewTarget)
class WfHuntingCalendar(BaseHuntingCalendarMixin, WfSharedContent):
    """Hunting calendar class"""

    references = FieldProperty(IWfHuntingCalendar['references'])
    
    @property
    def forest_ids(self):
        location = ILocationInfo(self)
        return location.forests or ()


register_content_type(WfHuntingCalendar)


@factory_config(IHuntingCalendarInfo)
class HuntingCalendarInfo(Persistent, Contained):
    """Hunting calendar persistent information"""

    season = FieldProperty(IHuntingCalendarInfo['season'])


@adapter_config(context=IWfHuntingCalendar, provides=IHuntingCalendarInfo)
def hunting_calendar_info_factory(context):
    """Hunting calendar info factory"""
    return get_annotation_adapter(context, HUNTING_INFO_ANNOTATIONS_KEY, IHuntingCalendarInfo,
                                  name='++info++')


@provider(IWfHuntingCalendarFactory)
@implementer(IHuntingCalendar)
class HuntingCalendar(SharedContent):
    """Workflow managed activity"""


@adapter_config(context=IWfHuntingCalendarFactory, provides=IWfSharedContentFactory)
def hunting_calendar_content_factory(context):
    return WfHuntingCalendar


#
# Custom hunting calendar location adapter
#

@adapter_config(required=FakeHuntingCalendar,
                provides=ILocationInfo)
class FakeHuntingCalendarLocation(ContextAdapter, LocationInfo):
    """Fake hunting calendar location info"""

    @property
    def _forests(self):
        return self.context.forest_ids
    
    @property
    def forests(self):
        return self._forests
    
    @property
    def locations(self):
        return self.get_gps_locations()


#
# Custom forest hearing management
#

@adapter_config(required=FakeHuntingCalendar,
                provides=IHearingInfo)
class FakeHuntingCalendarHearingInfo(ContextAdapter, HearingInfo):
    """Fake hunting calendar hearing info"""

    @property
    def _forests(self):
        return ILocationInfo(self.context).forests

    @property
    def forests(self):
        return self._forests

    @property
    def _source(self):
        manager = query_utility(IHuntingManager)
        if manager is not None:
            return manager.default_hearing_source

    @property
    def source(self):
        return self._source

    @property
    def source_folder(self):
        manager = query_utility(IHuntingManager)
        if manager is not None:
            return manager.default_source_folder


#
# Fake hunting calendar workflow
#

@adapter_config(required=FakeHuntingCalendar,
                provides=IWorkflowPublicationInfo)
class FakeHuntingCalendarWorkflowPublicationInfo(ContextAdapter, WorkflowContentPublicationInfo):
    """Fake hunting calendar workflow publication info"""

    @reify
    def publication_effective_date(self):
        return min(
            tztime(foret.date_debut_validite)
            for foret, proprietaire, info_foret in self.context.forest_refs
        )

    @reify
    def publication_expiration_date(self):
        return max(
            tztime(foret.date_fin_validite)
            for foret, proprietaire, info_foret in self.context.forest_refs
        )

    @property
    def visible_publication_date(self):
        return None


#
# Fake hunting calendar tags
#

@adapter_config(required=FakeHuntingCalendar,
                provides=ITagsInfo)
def fake_hunting_calendar_tags(context):
    """Fake hunting calendar tags"""
    manager = get_utility(IHuntingManager)
    return ITagsInfo(manager)
