#
# Copyright (c) 2015-2021 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

from uuid import uuid4

from hypatia.catalog import CatalogQuery
from hypatia.interfaces import ICatalog
from hypatia.query import Eq
from pyramid.httpexceptions import HTTPNotFound
from pyramid.response import Response
from pyramid.view import view_config
from z3c.form.button import Button, Buttons, handler
from z3c.form.field import Fields
from z3c.form.interfaces import DISPLAY_MODE
from zope.dublincore.interfaces import IZopeDublinCore
from zope.interface import Interface, alsoProvides, implementer
from zope.lifecycleevent import ObjectCreatedEvent, ObjectModifiedEvent
from zope.location import locate

from onf_website.reference.forest.schema import ForestField
from onf_website.shared.forest import FOREST_CONTENT_TYPE, FakeForest, IForestManager, \
    IWfForest
from onf_website.shared.forest.interfaces import MANAGE_FOREST_PERMISSION
from onf_website.skin.zmi import onf_website
from pyams_catalog.query import CatalogResultSet
from pyams_content.interfaces import COMMENT_CONTENT_PERMISSION
from pyams_content.shared.common import IWfSharedContentRoles
from pyams_content.workflow import DRAFT, PROPOSED
from pyams_content.zmi import pyams_content
from pyams_form.form import AJAXAddForm, InnerAddForm, InnerEditForm, ajax_config
from pyams_form.help import FormHelp
from pyams_form.interfaces.form import IFormContextPermissionChecker, IFormHelp, IWidgetForm, \
    IWidgetsPrefixViewletsManager, IWidgetsSuffixViewletsManager
from pyams_form.schema import ResetButton
from pyams_i18n.interfaces import II18n
from pyams_i18n.schema import I18nHTMLField
from pyams_pagelet.pagelet import pagelet_config
from pyams_security.schema import Principal
from pyams_security.utility import get_principal
from pyams_skin.interfaces import IInnerPage
from pyams_skin.interfaces.tinymce import ITinyMCEConfiguration
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.skin import apply_skin
from pyams_utils.adapter import ContextRequestAdapter, ContextRequestViewAdapter, adapter_config
from pyams_utils.date import format_datetime
from pyams_utils.fanstatic import get_resource_path
from pyams_utils.html import html_to_text
from pyams_utils.interfaces.data import IObjectData
from pyams_utils.list import boolean_iter
from pyams_utils.registry import get_utility
from pyams_utils.traversing import get_parent
from pyams_utils.url import absolute_url, generate_url
from pyams_viewlet.viewlet import Viewlet, viewlet_config
from pyams_workflow.interfaces import IWorkflowInfo, IWorkflowState, IWorkflowVersions
from pyams_zmi.form import AdminAddForm
from pyams_zmi.layer import IAdminLayer
from pyams_zmi.skin import AdminSkin


__docformat__ = 'restructuredtext'

from onf_website import _


class IForestUserBaselineInputInnerForm(Interface):
    """Forest user input inner form marker interface"""


class IWfForestUserBaselineInput(Interface):
    """User forest baseline input form interface"""

    forest_id = ForestField(title=_("Forest national ID"),
                            description=_("You can select a forest to display it's current "
                                          "essential text"),
                            required=True)

    baseline_draft = I18nHTMLField(title=_("My proposed baseline"),
                                   description=_("Proposed baseline text for this forest"),
                                   required=False)

    baseline_principal = Principal(title=_("Last proposal"),
                                   description=_("Principal who proposed this baseline"),
                                   required=False)


class IWfForestUserBaselineInputButtons(Interface):
    """User forest baseline input form buttons interface"""

    reset = ResetButton(name='reset', title=_("Reset"))
    publish = Button(name='publish', title=_("Save and request publication"))
    save = Button(name='save', title=_("Save my presentation"))


@pagelet_config(name='forest-user-baseline.html',
                context=IForestManager, layer=IPyAMSLayer,
                permission=MANAGE_FOREST_PERMISSION)
@ajax_config(name='forest-user-baseline.json',
             context=IForestManager, layer=IPyAMSLayer,
             permission=MANAGE_FOREST_PERMISSION,
             base=AJAXAddForm)
@implementer(IWidgetForm, IInnerPage, IObjectData)
class ForestUserBaselineInputForm(AdminAddForm):
    """Forest user baseline input form"""

    legend = _("What is the forest you would like to present?")
    prefix = 'form.'

    @property
    def fields(self):
        fields = Fields(IWfForestUserBaselineInput)
        if 'form.widgets.forest_id' not in self.request.form:
            fields = fields.select('forest_id')
        return fields

    buttons = Buttons(IWfForestUserBaselineInputButtons)
    handleActions = True

    edit_permission = MANAGE_FOREST_PERMISSION

    object_data = {
        'ams-plugins': 'onf_website',
        'ams-plugin-onf_website-src': get_resource_path(onf_website),
        'ams-reset-handler': 'ONF_website.forest.resetBaselineForm'
    }

    def updateWidgets(self, prefix=None):
        super().updateWidgets(prefix)
        forest_id = self.widgets.get('forest_id')
        if forest_id is not None:
            forest_id.object_data = {
                'ams-change-handler': 'MyAMS.helpers.select2ChangeHelper',
                'ams-change-stop-propagation': 'true',
                'ams-select2-change-on-reset': 'false',
                'ams-select2-helper-type': 'html',
                'ams-select2-helper-url': absolute_url(self.context, self.request,
                                                       'get-forest-baseline.html'),
                'ams-select2-helper-argument': 'forest_id',
                'ams-select2-helper-target': '#forest-user-baseline-helper'
            }
            alsoProvides(forest_id, IObjectData)

    def updateActions(self):
        super().updateActions()
        publish = self.actions.get('publish')
        if publish is not None:
            publish.addClass('btn-primary')
        save = self.actions.get('save')
        if save is not None:
            save.addClass('btn-primary')

    @handler(buttons['reset'])
    def handleReset(self, action):
        return

    @handler(buttons['publish'])
    def handlePublish(self, action):
        forest = self.handleSave(self, action)
        state = IWorkflowState(forest)
        if state.state != PROPOSED:
            info = IWorkflowInfo(forest)
            contributors = forest.contributors
            forest.contributors = contributors | {self.request.principal.id}
            info.fire_transition_toward(PROPOSED, check_security=False)
            forest.contributors = contributors
            dc = IZopeDublinCore(forest)
            forest.baseline_timestamp = dc.modified
        return forest

    @handler(buttons['save'])
    def handleSave(self, action):
        data, errors = self.extractData()
        forest_id = data.get('forest_id')
        if forest_id is None:
            raise HTTPNotFound
        catalog = get_utility(ICatalog)
        params = Eq(catalog['content_type'], FOREST_CONTENT_TYPE) & \
            Eq(catalog['forests'], forest_id)
        has_results, results = boolean_iter(CatalogResultSet(CatalogQuery(catalog).query(params)))
        if not has_results:
            fake = FakeForest()
            fake.forest_ids = [forest_id]
            if not fake.forest_refs:
                raise HTTPNotFound
            content = self.context.shared_content_factory()
            self.request.registry.notify(ObjectCreatedEvent(content))
            uuid = str(uuid4())
            self.context[uuid] = content
            forest = self.context.shared_content_factory.content_class()
            locate(forest, content)
            self.request.registry.notify(ObjectCreatedEvent(forest))
            forest.title = {'fr': fake.title}
            forest.short_name = forest.title.copy()
            forest.content_url = generate_url(fake.title)
            forest.forest_ids = fake.forest_ids
            forest.creator = self.request.principal.id
            owner = self.request.principal.id
            if owner in self.context.contributors:
                forest.owner = owner
            else:
                forest.owner = self.context.default_owner
            if not self.request.has_permission(COMMENT_CONTENT_PERMISSION, context=forest):
                IWfSharedContentRoles(forest).readers = {self.request.principal.id}
            IWorkflowVersions(content).add_version(forest, None)
            translate = self.request.localizer.translate
            IWorkflowInfo(forest).fire_transition('init',
                                                  comment=translate(_("Draft created from text "
                                                                      "presentation module")))
        else:
            forest = next(results)
            versions = IWorkflowVersions(forest)
            if versions.has_version({DRAFT, PROPOSED}):
                forest = versions.get_versions({DRAFT, PROPOSED}, sort=True)[-1]
            if not self.request.has_permission(COMMENT_CONTENT_PERMISSION, context=forest):
                IWfSharedContentRoles(forest).readers |= {self.request.principal.id}
        forest.baseline_draft = data.get('baseline_draft')
        forest.baseline_principal = self.request.principal.id
        self.request.registry.notify(ObjectModifiedEvent(forest))
        dc = IZopeDublinCore(forest)
        forest.baseline_timestamp = dc.modified
        return forest

    def add(self, obj):
        pass

    def nextURL(self):
        pass

    def get_ajax_output(self, changes):
        translate = self.request.localizer.translate
        message = None
        for action in self.actions.executedActions:
            if action.field is IWfForestUserBaselineInputButtons['save']:
                message = translate(_("Your proposal has been saved. You didn't requested its "
                                      "publication, but you can find it to complete or update "
                                      "just by using this form and selecting your forest "
                                      "again..."))
            elif action.field is IWfForestUserBaselineInputButtons['publish']:
                message = translate(_("Your proposal has been saved and the editorial team has "
                                      "been notified!"))
        result = {'status': 'reload'}
        if message:
            result['messagebox'] = {
                'status': 'success',
                'title': translate(_("Input saved!")),
                'content': message,
                'icon': 'fa fa-fw fa-user',
                'timeout': 10000
            }
        return result


@viewlet_config(name='forest-user-baseline',
                context=IForestManager, layer=IAdminLayer, view=ForestUserBaselineInputForm,
                manager=IWidgetsSuffixViewletsManager, weight=10)
class ForestUserBaselineInputFormSuffix(Viewlet):
    """Forest user input form suffix"""

    def render(self):
        return '<div id="forest-user-baseline-helper"></div>' \
               '<div id="forest-user-baseline-feedback"></div>'


@implementer(IForestUserBaselineInputInnerForm)
class ForestUserBaselineInputInnerAddForm(InnerAddForm):
    """Forest user baseline input inner add form"""

    prefix = 'form.'
    legend = _("Presentation text")
    fields = Fields(IWfForestUserBaselineInput).select('baseline_draft')

    edit_permission = MANAGE_FOREST_PERMISSION


@implementer(IForestUserBaselineInputInnerForm)
class ForestUserBaselineInputInnerEditForm(InnerEditForm):
    """Forest user baseline input inner edit form"""

    legend = _("Presentation text")

    @property
    def fields(self):
        fields = Fields(IWfForest).select('baseline_draft', 'baseline_principal_status',
                                          'baseline_published')
        if not self.context.baseline_published:
            fields = fields.omit('baseline_published')
        return fields

    edit_permission = MANAGE_FOREST_PERMISSION

    def updateWidgets(self, prefix=None):
        super().updateWidgets(prefix)
        translate = self.request.localizer.translate
        baseline = self.widgets.get('baseline_draft')
        if baseline is not None:
            value = baseline.value or {}
            published = self.context.baseline_published or {}
            if not value:
                value = published.copy()
            else:
                for lang, lang_baseline in published.items():
                    if not value.get(lang):
                        value[lang] = lang_baseline
            baseline.value = value
            draft = II18n(self.context).query_attribute('baseline_draft', request=self.request)
            current_length = len(html_to_text(draft or ''))
            if current_length:
                baseline.after_widget_notice = \
                    translate(_('<span class="text-info">Current text '
                                'length: {} characters</span>')).format(current_length)
                manager = get_parent(self.context, IForestManager)
                max_length = manager.baseline_max_length
                if current_length > max_length:
                    baseline.after_widget_notice += \
                        translate(_('<strong class="text-danger padding-left-5">/ +{:n}</strong>')) \
                            .format(current_length - max_length)
        principal = self.widgets.get('baseline_principal_status')
        if principal is not None:
            baseline_draft = II18n(self.context).query_attribute('baseline_draft',
                                                                 request=self.request)
            if baseline_draft:
                principal.value = translate(_("{date} by {principal}")).format(
                    date=format_datetime(self.context.baseline_timestamp),
                    principal=get_principal(self.request, self.context.baseline_principal).title)
            else:
                principal.value = translate(_("No text proposal was found."))
        published = self.widgets.get('baseline_published')
        if published is not None:
            published.set_mode(DISPLAY_MODE)


@adapter_config(context=(IWfForest, IAdminLayer, IForestUserBaselineInputInnerForm),
                provides=IFormHelp)
class ForestUserBaselineInputFormHelp(FormHelp):
    """Forest user baseline input form help"""

    def __new__(cls, context, request, view):
        manager = get_utility(IForestManager)
        if not manager.baseline_help:
            return None
        return FormHelp.__new__(cls)

    @property
    def message(self):
        manager = get_utility(IForestManager)
        return manager.baseline_help

    message_format = 'rest'


@adapter_config(context=(IForestUserBaselineInputInnerForm, Interface),
                provides=ITinyMCEConfiguration)
class ForestUserBaselineInputInnerFormEditorConfiguration(ContextRequestAdapter):
    """Forest user baseline input inner form editor configuration"""

    configuration = {
        'ams-plugins': 'pyams_content',
        'ams-plugin-pyams_content-src': get_resource_path(pyams_content),
        'ams-plugin-pyams_content-async': 'false',
        'ams-tinymce-init-callback': 'PyAMS_content.TinyMCE.initEditor',
        'ams-tinymce-menubar': False,
        'ams-tinymce-plugins': ['paste', 'lists', 'charmap'],
        'ams-tinymce-toolbar': 'undo redo | pastetext | bold italic | bullist numlist',
        'ams-tinymce-toolbar1': False,
        'ams-tinymce-toolbar2': False,
        'ams-tinymce-height': 300
    }


@adapter_config(context=(IWfForest, IAdminLayer, IForestUserBaselineInputInnerForm),
                provides=IFormContextPermissionChecker)
class ForestUserBaselineInputInnerEditFormPermissionChecker(ContextRequestViewAdapter):
    """Forest user baseline input inner edit form context permission checker"""

    edit_permission = MANAGE_FOREST_PERMISSION


@view_config(name='get-forest-baseline.html',
             context=IForestManager, request_type=IPyAMSLayer,
             permission=MANAGE_FOREST_PERMISSION, xhr=True)
def get_forest_user_baseline_input_form(request):
    """Forest user baseline input form"""
    apply_skin(request, AdminSkin)
    forest_id = request.params.get('forest_id')
    if forest_id is None:
        raise HTTPNotFound
    catalog = get_utility(ICatalog)
    params = Eq(catalog['content_type'], FOREST_CONTENT_TYPE) & \
        Eq(catalog['forests'], forest_id)
    has_results, results = boolean_iter(CatalogResultSet(CatalogQuery(catalog).query(params)))
    if has_results:
        forest = next(results)
        versions = IWorkflowVersions(forest)
        if versions.has_version({DRAFT, PROPOSED}):
            forest = versions.get_versions({DRAFT, PROPOSED}, sort=True)[-1]
        else:
            version = versions.get_last_versions()[-1]
            info = IWorkflowInfo(version)
            contributors = (version.contributors or set()).copy()
            version.contributors = contributors | {request.principal.id}
            translate = request.localizer.translate
            forest = info.fire_transition_toward(DRAFT, check_security=False,
                                                 comment=translate(_("New draft created from "
                                                                     "text presentation module")))
            forest.contributors = version.contributors = contributors
        form = ForestUserBaselineInputInnerEditForm(forest, request)
    else:
        forest = FakeForest()
        forest.forest_ids = [forest_id]
        form = ForestUserBaselineInputInnerAddForm(request.context, request)
    form.update()
    return Response(form.render())
