#
# Copyright (c) 2015-2021 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""ONF_website.shared.forest module

"""

from hypatia.catalog import CatalogQuery
from hypatia.interfaces import ICatalog
from hypatia.query import Eq
from pyramid.decorator import reify
from pyramid.events import subscriber
from zope.dublincore.interfaces import IZopeDublinCore
from zope.interface import implementer, provider
from zope.lifecycleevent.interfaces import IObjectAddedEvent
from zope.schema.fieldproperty import FieldProperty

from onf_website.component.hearing import HearingInfo
from onf_website.component.hearing.interfaces import IHearingInfo, IHearingTarget
from onf_website.component.location import LocationInfo
from onf_website.component.location.interfaces import ILocationInfo, ILocationTarget
from onf_website.reference.forest.model.foret import Foret, InformationForet, \
    PARENT_SESSION as RDF_SESSION, ProprietaireForet
from onf_website.reference.forest.model.interfaces.foret import IForet
from onf_website.reference.planning.model.planning import InformationPlanning, PlanningData, \
    PlanningForet
from onf_website.shared.forest.interfaces import FOREST_CONTENT_NAME, FOREST_CONTENT_TYPE, \
    IForest, IForestManager, IWfForest, IWfForestFactory, IWfRealForest
from pyams_alchemy.engine import get_user_session
from pyams_catalog.query import CatalogResultSet
from pyams_content.component.gallery import IGalleryTarget
from pyams_content.component.illustration import IIllustration, IIllustrationTarget, ILinkIllustration, \
    ILinkIllustrationTarget
from pyams_content.component.paragraph import IParagraphContainerTarget
from pyams_content.component.theme import ITagsTarget, IThemesTarget
from pyams_content.features.preview.interfaces import IPreviewTarget
from pyams_content.features.review import IReviewTarget
from pyams_content.shared.common import IWfSharedContentFactory, SharedContent, WfSharedContent, \
    register_content_type
from pyams_portal.interfaces import IPortalPage
from pyams_utils.adapter import ContextAdapter, adapter_config
from pyams_utils.interfaces import ICacheKeyValue
from pyams_utils.interfaces.url import DISPLAY_CONTEXT
from pyams_utils.list import boolean_iter
from pyams_utils.registry import get_utility, query_utility
from pyams_utils.request import query_request
from pyams_utils.timezone import tztime
from pyams_workflow.content import WorkflowContentPublicationInfo
from pyams_workflow.interfaces import IWorkflow, IWorkflowPublicationInfo, IWorkflowVersionTransitionEvent, \
    IWorkflowVersions


__docformat__ = 'restructuredtext'


FOREST_REFS_KEY = 'onf_website.forest.refs'
PLANNING_REFS_KEY = 'onf_website.planning.refs'

FOREST_REFS_MARKER = object()
PLANNING_REFS_MARKER = object()


class BaseForestMixin:
    """Base forest mixin class"""

    content_type = FOREST_CONTENT_TYPE
    content_name = FOREST_CONTENT_NAME

    references = FieldProperty(IWfForest['references'])
    forest_ids = FieldProperty(IWfForest['forest_ids'])

    def get_forest_refs(self):
        session = get_user_session(RDF_SESSION)
        return session.query(Foret, ProprietaireForet, InformationForet) \
            .join(ProprietaireForet,
                  Foret.id_nat_frt == ProprietaireForet.id_nat_frt) \
            .outerjoin(InformationForet,
                       Foret.id_nat_frt == InformationForet.id_nat_frt) \
            .filter(Foret.id_nat_frt.in_(self.forest_ids)) \
            .order_by(ProprietaireForet.pourcentage_part, ProprietaireForet.categorie)

    @property
    def forest_refs(self):
        request = query_request()
        is_display_context = (request is not None) and \
                             (request.annotations.get(DISPLAY_CONTEXT) is self)
        refs = FOREST_REFS_MARKER
        if is_display_context:
            refs = request.annotations.get(FOREST_REFS_KEY, FOREST_REFS_MARKER)
        if refs is FOREST_REFS_MARKER:
            refs = self.get_forest_refs().all()
            if is_display_context:
                request.annotations[FOREST_REFS_KEY] = refs
        return refs

    def get_planning_refs(self):
        session = get_user_session(RDF_SESSION)
        return session.query(PlanningData, InformationPlanning) \
            .join(PlanningForet,
                  PlanningData.id_nat_amgt == PlanningForet.id_nat_amgt) \
            .outerjoin(InformationPlanning,
                       PlanningData.id_nat_amgt == InformationPlanning.id_nat_amgt) \
            .filter(PlanningForet.id_nat_foret.in_(self.forest_ids)) \
            .order_by(PlanningData.date_debut_applicabilite)

    @property
    def planning_refs(self):
        request = query_request()
        is_display_context = (request is not None) and \
                             (request.annotations.get(DISPLAY_CONTEXT) is self)
        refs = PLANNING_REFS_MARKER
        if is_display_context:
            refs = request.annotations.get(PLANNING_REFS_KEY, PLANNING_REFS_MARKER)
        if refs is PLANNING_REFS_MARKER:
            refs = self.get_planning_refs().all()
            if is_display_context:
                request.annotations[PLANNING_REFS_KEY] = refs
        return refs

    def get_departments(self):
        try:
            return ', '.join(sorted(set(('{} ({})'.format(foret.departement_insee.nccenr,
                                                          foret.departement_insee.dep)
                                         for foret, proprietaire, info in self.forest_refs))))
        except AttributeError:
            return '--'

    def get_communes(self):
        try:
            return ', '.join(sorted(set((commune.commune.nccenr
                                         for foret, proprietaire, info in self.forest_refs
                                         for commune in foret.communes))))
        except AttributeError:
            return '--'


@implementer(IWfForest, ILocationTarget, IHearingTarget)
class FakeForest(BaseForestMixin):
    """Fake forest class"""

    def __init__(self, forest_ids=None):
        super().__init__()
        if forest_ids:
            if isinstance(forest_ids, str):
                forest_ids = [forest_ids]
            self.forest_ids = forest_ids

    @reify
    def __parent__(self):
        manager = query_utility(IForestManager)
        if manager is not None:
            return manager.get_default_source()

    @property
    def __name__(self):
        return '++frt++{}'.format(''.join(self.forest_ids))

    @property
    def title(self):
        return ''.join(set(
            (info.libelle if info is not None else None) or foret.libelle_usage
            for foret, prop, info in self.forest_refs
        ))

    description = None
    baseline_draft = None
    baseline_principal = None
    baseline_timestamp = None
    baseline_published = None

    @property
    def header(self):
        foret, proprietaire, info_foret = self.forest_refs[0]
        return info_foret.header if info_foret is not None else None


@adapter_config(context=FakeForest, provides=ICacheKeyValue)
def fake_forest_cache_key(context):
    """Fake forest cache key adapter"""
    return 'FRT::{}'.format(','.join(context.forest_ids))


@implementer(IWfForest, IWfRealForest, IGalleryTarget, ILocationTarget, IHearingTarget,
             IIllustrationTarget, ILinkIllustrationTarget, IParagraphContainerTarget,
             ITagsTarget, IThemesTarget,
             IPreviewTarget, IReviewTarget)
class WfForest(BaseForestMixin, WfSharedContent):
    """Base forest class"""

    baseline_draft = FieldProperty(IWfForest['baseline_draft'])
    baseline_principal = FieldProperty(IWfForest['baseline_principal'])
    baseline_timestamp = FieldProperty(IWfForest['baseline_timestamp'])
    baseline_published = FieldProperty(IWfForest['baseline_published'])
    gallery_comment = FieldProperty(IWfForest['gallery_comment'])


register_content_type(WfForest)


@provider(IWfForestFactory)
@implementer(IForest)
class Forest(SharedContent):
    """Workflow managed forest class"""


@adapter_config(context=IWfForestFactory, provides=IWfSharedContentFactory)
def forest_content_factory(context):
    return WfForest


@subscriber(IObjectAddedEvent, context_selector=IWfRealForest)
def real_forest_creation_handler(event):
    """Forest creation handler"""
    forest = event.object
    foret, proprietaire, info_foret = forest.get_forest_refs().first()
    if ((not forest.header) or (not forest.header.get('fr'))) and (info_foret is not None):
        forest.header = {'fr': info_foret.header}
    if proprietaire is not None:
        templates = get_utility(IForestManager).templates
        owner_template = templates.get(proprietaire.categorie)
        if owner_template is not None:
            page = IPortalPage(forest)
            page.inherit_parent = False
            page.shared_template = owner_template


@subscriber(IWorkflowVersionTransitionEvent, context_selector=IWfRealForest)
def real_forest_transition_handler(event):
    """Forest workflow transition handler"""
    if event.destination != event.workflow.initial_state:
        return
    forest = event.object
    if forest.baseline_draft != forest.baseline_published:
        forest.baseline_draft = forest.baseline_published
        forest.baseline_principal = event.principal.id
        dc = IZopeDublinCore(forest)
        forest.baseline_timestamp = dc.modified
    forest.gallery_comment = None


#
# Custom forest location management
#

@adapter_config(context=FakeForest, provides=ILocationInfo)
@adapter_config(context=WfForest, provides=ILocationInfo)
class ForestLocationInfo(ContextAdapter, LocationInfo):
    """Forest location info"""

    @property
    def _forests(self):
        return self.context.forest_ids

    @property
    def forests(self):
        return self._forests

    @property
    def locations(self):
        return self.get_gps_locations()


#
# Custom forest hearing management
#

@adapter_config(context=FakeForest, provides=IHearingInfo)
@adapter_config(context=WfForest, provides=IHearingInfo)
class ForestHearingInfo(ContextAdapter, HearingInfo):
    """Forest hearing info"""

    @property
    def _forests(self):
        return ILocationInfo(self.context).forests

    @property
    def forests(self):
        return self._forests

    @property
    def _source(self):
        manager = query_utility(IForestManager)
        if manager is not None:
            return manager.default_hearing_source

    @property
    def source(self):
        return self._source

    @property
    def source_folder(self):
        manager = query_utility(IForestManager)
        if manager is not None:
            return manager.default_source_folder


@adapter_config(context=FakeForest, provides=IWorkflowPublicationInfo)
class ForestWorkflowPublicationInfo(ContextAdapter, WorkflowContentPublicationInfo):
    """Forest workflow publication info"""

    @reify
    def publication_effective_date(self):
        return min(
            tztime(foret.date_debut_validite)
            for foret, proprietaire, info_foret in self.context.forest_refs
        )

    @reify
    def publication_expiration_date(self):
        return max(
            tztime(foret.date_fin_validite)
            for foret, proprietaire, info_foret in self.context.forest_refs
        )

    @property
    def visible_publication_date(self):
        return None


@adapter_config(required=IForet,
                provides=ILinkIllustration)
def forest_illustration(context):
    """Forest illustration"""
    catalog = get_utility(ICatalog)
    params = Eq(catalog['content_type'], FOREST_CONTENT_TYPE) & \
        Eq(catalog['forests'], context.id_nat_frt)
    has_forests, forests = boolean_iter(CatalogResultSet(CatalogQuery(catalog).query(params)))
    if has_forests:
        forest = next(forests)
        workflow = IWorkflow(forest, None)
        if workflow is None:
            return None
        versions = IWorkflowVersions(forest, None)
        if versions is None:
            return None
        forest = None
        versions = versions.get_versions(workflow.visible_states, sort=True)
        if versions:
            forest = versions[-1]
        if forest is not None:
            illustration = ILinkIllustration(forest, None)
            if (illustration is not None) and illustration.has_data():
                return illustration
            illustration = IIllustration(forest, None)
            if (illustration is not None) and illustration.has_data():
                return illustration
    return None
