#
# Copyright (c) 2008-2019 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from zope.container.constraints import contains
from zope.container.interfaces import IContainer
from zope.interface import Interface
from zope.location.interfaces import IContained
from zope.schema import Bool, Choice, Set, URI, TextLine

from onf_website.reference.insee.schema import CommunesListField
from onf_website.reference.orga.schema import StructuresListField
from onf_website.shared.contact.interfaces.theme import CONTACT_THEMES_VOCABULARY
from pyams_i18n.schema import I18nTextLineField
from pyams_utils.schema import MailAddressField

from onf_website import _


class IContactAssignment(Interface):
    """Contact assignments interfaces"""

    active = Bool(title=_("Active assignment?"),
                  required=True,
                  default=True)

    theme = Choice(title=_("Contact theme"),
                   description=_("Theme matching this assignment"),
                   vocabulary=CONTACT_THEMES_VOCABULARY,
                   required=True)

    cities = CommunesListField(title=_("Cities"),
                               required=False)

    departments = Set(title=_("Departments"),
                      value_type=Choice(vocabulary='ONF regions with departments'),
                      required=False)

    structures = StructuresListField(title=_("Structures"),
                                     required=False)

    target_url = URI(title=_("Target URI"),
                     description=_("This URI will be displayed in contact form if this "
                                   "assignment is matching location and theme"),
                     required=False)

    target_url_label = I18nTextLineField(title=_("Target URI label"),
                                         description=_("This is the label which will be associated "
                                                       "with the given URI"),
                                         required=False)

    mail_address = MailAddressField(title=_("Mail address"),
                                    description=_("Alternate email address for this assignment, "
                                                  "without '<' or '>'"),
                                    required=False)

    phone_number = TextLine(title=_("Phone number"),
                            description=_('Phone number in international format.\n E.g. +33 ....'),
                            required=False)

    def get_theme(self):
        """Get contact theme associated with this assignment"""

    def is_matching(self, insee_code):
        """Returns a boolean value to check if given INSEE code is matching any assignment"""


CONTACT_ASSIGNMENTS_ANNOTATION_KEY = 'onf_website.contact.assignments'


class IContactAssignmentsContainer(IContainer, IContained):
    """Contact assignments manager interface"""

    contains(IContactAssignment)

    def get_active_assignments(self, theme=None):
        """Get currently active assignments"""

    def get_matching_assignment(self, theme, insee_code, dept_code):
        """Returns any assignment matching given themes and postal codes, or None

        :param str theme: input theme; can also be an :ref:`IContactTheme` object
        :param str insee_code: INSEE code of commune for which we are looking for an assignment
        :param str dept_code: INSEE code of department for which we are looking for an assignment
        :return IContactAssignment: any assignment matching given arguments
        """
