#
# Copyright (c) 2008-2019 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from persistent import Persistent
from zope.container.contained import Contained
from zope.container.folder import Folder
from zope.interface import implementer
from zope.schema.fieldproperty import FieldProperty
from zope.traversing.interfaces import ITraversable

from onf_website.reference.insee.model import Departement
from onf_website.reference.location.interfaces.region import IRegion
from onf_website.reference.orga.model import Structure
from onf_website.shared.contact import CONTACT_ASSIGNMENTS_ANNOTATION_KEY, IContactAssignment, \
    IContactAssignmentsContainer
from onf_website.shared.contact.interfaces import IContactManager, IWfContact
from onf_website.shared.contact.interfaces.theme import IContactTheme, IContactThemeManager
from pyams_form.interfaces.form import IFormContextPermissionChecker
from pyams_utils.adapter import ContextAdapter, adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config
from pyams_utils.registry import query_utility
from pyams_utils.traversing import get_parent


@implementer(IContactAssignment)
class ContactAssignment(Persistent, Contained):
    """Contact affectation persistent class"""

    active = FieldProperty(IContactAssignment['active'])

    theme = FieldProperty(IContactAssignment['theme'])
    cities = FieldProperty(IContactAssignment['cities'])
    departments = FieldProperty(IContactAssignment['departments'])
    structures = FieldProperty(IContactAssignment['structures'])
    target_url = FieldProperty(IContactAssignment['target_url'])
    target_url_label = FieldProperty(IContactAssignment['target_url_label'])
    mail_address = FieldProperty(IContactAssignment['mail_address'])
    phone_number = FieldProperty(IContactAssignment['phone_number'])

    def get_theme(self):
        if self.theme is not None:
            contacts = query_utility(IContactManager)
            if contacts is not None:
                manager = IContactThemeManager(contacts)
                return manager.get(self.theme)

    def get_departments(self):
        departments = self.departments or ()
        if departments:
            for value in departments:
                if IRegion.providedBy(value):
                    for dept in value.departments or ():
                        yield dept.insee_code
                else:
                    yield value.insee_code

    def is_matching(self, insee_code, dept_code):
        if not self.active:
            # ignore inactive assignments
            return False
        if not (self.cities or self.departments or self.structures):
            # default assignment for this theme
            return True
        if not (insee_code or dept_code):
            # can't match without INSEE code
            return False
        if self.cities and (insee_code in self.cities):
            # exact commune match
            return True
        if self.departments:
            # check for departments match
            dept_codes = set()
            if dept_code:
                dept_codes.add(dept_code)
            if insee_code:
                if insee_code.startswith('97'):
                    dept_codes.add(insee_code[0:3])
                else:
                    dept_codes.add(insee_code[0:2])
            depts = set((dept.dep for dept in Departement.find_by_insee_code(dept_codes)))
            if depts & set(self.get_departments()):
                return True
        if self.structures:
            # check for structures match
            structures = set(self.structures)
            unit = Structure.find_by_insee_code(insee_code).first()
            if unit is not None:
                unit_code = {unit.code}
                if unit_code & structures:
                    return True
                else:
                    for parent in unit.get_parents((Structure.code, Structure.code_sign)):
                        if parent.code_sign:
                            parent_code = {parent.code}
                            if parent_code & structures:
                                return True
        return False


@adapter_config(context=IContactAssignment, provides=IFormContextPermissionChecker)
def contact_assignment_permission_checker_factory(context):
    """Contact assignment permission checker"""
    contact = get_parent(context, IWfContact)
    return IFormContextPermissionChecker(contact)


@factory_config(IContactAssignmentsContainer)
class ContactAssignmentsContainer(Folder):
    """Contact assignments container"""

    def get_active_assignments(self, theme=None):
        if theme is None:
            return filter(lambda x: IContactAssignment(x).active, self.values())
        if IContactTheme.providedBy(theme):
            theme = theme.__name__
        result = self.get(theme)
        if result and result.active:
            return result
        return None

    def get_matching_assignment(self, theme, insee_code, dept_code):
        if IContactTheme.providedBy(theme):
            theme = theme.__name__
        assignment = self.get(theme)
        if (assignment is None) or not assignment.active:
            return None
        elif assignment.is_matching(insee_code, dept_code):
            return assignment
        return None


@adapter_config(context=IWfContact, provides=IContactAssignmentsContainer)
def contact_assignments_factory(context):
    """Contact assignments factory"""
    return get_annotation_adapter(context, CONTACT_ASSIGNMENTS_ANNOTATION_KEY,
                                  IContactAssignmentsContainer, name='++assign++')


@adapter_config(name='assign', context=IWfContact, provides=ITraversable)
class ContactAssignmentsTraverser(ContextAdapter):
    """Contact assignments traverser"""

    def traverse(self, name, furtherpath=None):
        return self.context.assignments
