#
# Copyright (c) 2015-2021 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

from z3c.form.browser.radio import RadioFieldWidget
from zope.interface import Interface

from onf_website.shared.activity import IWfActivity
from onf_website.shared.activity.interfaces import IActivityManager
from pyams_content.component.gallery.zmi import GalleryMenu
from pyams_content.interfaces import CREATE_CONTENT_PERMISSION, MANAGE_CONTENT_PERMISSION
from pyams_content.shared.common.zmi import SharedContentAJAXAddForm
from pyams_content.shared.common.zmi.types import TypedSharedContentAddForm, \
    TypedSharedContentTypeFieldsEditForm
from pyams_form.form import ajax_config
from pyams_form.interfaces.form import IWidgetsSuffixViewletsManager
from pyams_i18n.interfaces import II18n
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.interfaces import IContentTitle
from pyams_skin.interfaces.viewlet import IMenuHeader, IWidgetTitleViewletManager
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.viewlet.menu import MenuItem
from pyams_skin.viewlet.toolbar import ToolbarAction
from pyams_template.template import template_config
from pyams_utils.adapter import ContextRequestAdapter, ContextRequestViewAdapter, adapter_config
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION
from pyams_viewlet.viewlet import Viewlet, viewlet_config
from pyams_zmi.interfaces.menu import IContentManagementMenu, IPropertiesMenu
from pyams_zmi.layer import IAdminLayer


__docformat__ = 'restructuredtext'

from onf_website import _


@adapter_config(context=(IWfActivity, IContentManagementMenu),
                provides=IMenuHeader)
class ActivityContentMenuHeader(ContextRequestAdapter):
    """Activity content menu header adapter"""

    header = _("This activity")


@adapter_config(context=(IWfActivity, IPyAMSLayer, Interface),
                provides=IContentTitle)
class ActivityTitleAdapter(ContextRequestViewAdapter):
    """Activity title adapter"""

    @property
    def title(self):
        translate = self.request.localizer.translate
        return translate(_("Activity « {title} »")).format(
            title=II18n(self.context).query_attribute('title', request=self.request))


@viewlet_config(name='add-shared-content.action',
                context=IActivityManager, layer=IAdminLayer, view=Interface,
                manager=IWidgetTitleViewletManager,
                permission=CREATE_CONTENT_PERMISSION, weight=1)
class ActivityAddAction(ToolbarAction):
    """Activity adding action"""

    label = _("Add activity")
    label_css_class = 'fa fa-fw fa-plus'
    url = 'add-shared-content.html'
    modal_target = True


@pagelet_config(name='add-shared-content.html',
                context=IActivityManager, layer=IPyAMSLayer,
                permission=CREATE_CONTENT_PERMISSION)
@ajax_config(name='add-shared-content.json',
             context=IActivityManager, request_type=IPyAMSLayer,
             permission=CREATE_CONTENT_PERMISSION, base=SharedContentAJAXAddForm)
class ActivityAddForm(TypedSharedContentAddForm):
    """Activity add form"""

    legend = _("Add activity")


#
# Activity info edit form
#

@viewlet_config(name='custom-properties.menu',
                context=IWfActivity, layer=IPyAMSLayer,
                manager=IPropertiesMenu,
                permission=MANAGE_CONTENT_PERMISSION, weight=20)
class ActivityTypePropertiesMenu(MenuItem):
    """Activity type properties menu"""

    label = _("Specific properties")
    icon_class = 'fa-paperclip'
    url = '#type-properties.html'


@pagelet_config(name='type-properties.html',
                context=IWfActivity, layer=IPyAMSLayer,
                permission=MANAGE_CONTENT_PERMISSION)
@ajax_config(name='type-properties.json',
             context=IWfActivity, layer=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION)
class ActivityTypePropertiesEditForm(TypedSharedContentTypeFieldsEditForm):
    """Activity type properties edit form"""

    @property
    def fields(self):
        fields = super().fields
        if 'markup' in fields:
            fields['markup'].widgetFactory = RadioFieldWidget
        return fields


@viewlet_config(name='type-properties.suffix',
                context=IWfActivity, layer=IAdminLayer, view=ActivityTypePropertiesEditForm,
                manager=IWidgetsSuffixViewletsManager, weight=10)
@template_config(template='templates/type-properties-help.pt')
class ActivityTypePropertiesEditFormHelp(Viewlet):
    """Activity type properties edit form help"""


#
# Medias gallery
#

@viewlet_config(name='gallery.menu',
                context=IWfActivity, layer=IPyAMSLayer,
                manager=IPropertiesMenu, permission=VIEW_SYSTEM_PERMISSION, weight=15)
class ActivityGalleryMenu(GalleryMenu):
    """Activity gallery menu item"""

    label = _("Presentation gallery...")
