# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from pyramid.decorator import reify
from pyramid.httpexceptions import HTTPNotFound
from zope.interface import implementer

from onf_website.reference.universe import IUniverse
from onf_website.skin.public.interfaces import IInnerView
from onf_website.skin.public.layer import IONFBaseLayer
from pyams_content.features.search.portlet import ISearchResultsFinder
from pyams_pagelet.pagelet import pagelet_config
from pyams_portal.interfaces import IPortalPage, IPortalPortletsConfiguration, IPortalTemplate, PORTAL_PAGE_KEY
from pyams_portal.page import PortalPage
from pyams_portal.views import PortalContextIndexPage
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.skin import apply_skin
from pyams_utils.adapter import ContextRequestAdapter, adapter_config, get_annotation_adapter
from pyams_utils.registry import query_utility


class UniversePortalPage(PortalPage):
    """Base universe portal page"""

    can_inherit = False
    use_local_template = False


class UniverseIndexPage(PortalContextIndexPage):
    """Universe list view"""

    @reify
    def portlet_configuration(self):
        return IPortalPortletsConfiguration(self.page.template)


#
# Map views
#

class UniverseMapPortalPage(UniversePortalPage):
    """Universe portal page"""

    @property
    def template(self):
        return query_utility(IPortalTemplate, name=self.__parent__.map_template)


@adapter_config(name='map-view',
                required=IUniverse,
                provides=IPortalPage)
def universe_map_portal_page(context):
    """Universe map portal page"""
    return get_annotation_adapter(context, f'{PORTAL_PAGE_KEY}::map', UniverseMapPortalPage)


@pagelet_config(name='map-view.html',
                context=IUniverse, layer=IPyAMSLayer)
class UniverseMapView(UniverseIndexPage):
    """Universe map view"""

    @reify
    def page(self):
        registry = self.request.registry
        return registry.queryAdapter(self.context, IPortalPage, name='map-view')


@pagelet_config(name='inner-map-view.html',
                context=IUniverse, layer=IPyAMSLayer)
@implementer(IInnerView)
class InnerUniverseMapView(UniverseMapView):
    """Inner universe map view"""

    def update(self):
        universe = IUniverse(self.context)
        if not universe.visible:
            raise HTTPNotFound()
        apply_skin(self.request, universe.skin)
        super().update()


#
# List views
#

class UniverseListPortalPage(UniversePortalPage):
    """Universe list portal page"""

    @property
    def template(self):
        return query_utility(IPortalTemplate, name=self.__parent__.list_template)


@adapter_config(name='list-view',
                required=IUniverse,
                provides=IPortalPage)
def universe_list_portal_page(context):
    """Universe list portal page"""
    return get_annotation_adapter(context, f'{PORTAL_PAGE_KEY}::list', UniverseListPortalPage)


@pagelet_config(name='list-view.html',
                context=IUniverse, layer=IPyAMSLayer)
class UniverseListView(UniverseIndexPage):
    """Universe list view"""

    @reify
    def page(self):
        registry = self.request.registry
        return registry.queryAdapter(self.context, IPortalPage, name='list-view')
    

@pagelet_config(name='inner-list-view.html',
                context=IUniverse, layer=IPyAMSLayer)
@implementer(IInnerView)
class InnerUniverseListView(UniverseListView):
    """Inner universe list view"""

    def update(self):
        universe = IUniverse(self.context)
        if not universe.visible:
            raise HTTPNotFound()
        apply_skin(self.request, universe.skin)
        super().update()


#
# Results finder
#

@adapter_config(required=(IUniverse, IONFBaseLayer),
                provides=ISearchResultsFinder)
class UniverseSearchResultsFinder(ContextRequestAdapter):
    """Universe search results finder"""
    
    def get_results(self, start=0, length=None, limit=None,
                    ignore_cache=False, get_count=False,
                    aggregates=None, settings=None, **kwargs):
        if self.request.view_name.endswith('map-view.html'):
            return self.context.get_map_items(self.request, aggregates, **kwargs)
        elif self.request.view_name.endswith('list-view.html'):
            return self.context.get_list_items(self.request, aggregates, **kwargs)
