#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from persistent import Persistent
from zope.container.contained import Contained
from zope.interface import Interface
from zope.schema.fieldproperty import FieldProperty

from onf_website.component.location import ILocationInfo
from onf_website.reference.universe.portlet import INearmePortletSettings
from onf_website.reference.universe.portlet.skin.interfaces import INearmePortletBaseRendererSettings, \
    INearmePortletLinkRendererSettings, INearmePortletPopupRendererSettings
from onf_website.shared.site import ISiteManagerUniverseInfo
from pyams_content.shared.site.interfaces import ISiteManager
from pyams_portal.interfaces import IPortalContext, IPortletRenderer
from pyams_portal.portlet import PortletRenderer
from pyams_skin.layer import IPyAMSLayer
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config
from pyams_utils.factory import factory_config
from pyams_utils.traversing import get_parent
from pyams_utils.url import get_display_context

__docformat__ = 'restructuredtext'

from onf_website import _


class NearmePortletBaseRendererSettings(Persistent, Contained):
    """Near me portlet base renderer settings"""

    nearme_mode = FieldProperty(INearmePortletBaseRendererSettings['nearme_mode'])


class NearmePortletBaseRenderer(PortletRenderer):
    """Nearme portlet base renderer"""

    @property
    def target(self):
        site = None
        context = get_display_context(self.request)
        if context is not None:
            site = get_parent(context, ISiteManager)
        if site is None:
            site = self.request.root
        universe_info = ISiteManagerUniverseInfo(site, None)
        if universe_info is not None:
            universe = universe_info.universe
            return universe if (universe is not None) and universe.visible else None
        return None


@factory_config(INearmePortletLinkRendererSettings)
class NearmePortletLinkRendererSettings(NearmePortletBaseRendererSettings):
    """Nearme portlet link renderer settings"""


@adapter_config(required=(IPortalContext, IPyAMSLayer, Interface, INearmePortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/nearme-link.pt', layer=IPyAMSLayer)
class NearmePortletLinkRenderer(NearmePortletBaseRenderer):
    """Nearme portlet link renderer"""

    label = _("Left column near-me link (default)")
    weight = 1

    settings_interface = INearmePortletLinkRendererSettings

    @property
    def location_params(self):
        location = ILocationInfo(self.request.context, None)
        if location is None:
            return None
        return '&'.join(location.get_search_params())


@factory_config(INearmePortletPopupRendererSettings)
class NearmePortletPopupRendererSettings(NearmePortletBaseRendererSettings):
    """Nearme portlet popup link renderer settings"""


@adapter_config(name='nearme-popup',
                required=(IPortalContext, IPyAMSLayer, Interface, INearmePortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/nearme-popup.pt', layer=IPyAMSLayer)
class NearmePortletPopupRenderer(NearmePortletBaseRenderer):
    """Nearme portlet popup renderer"""

    label = _("Main page popup link")
    weight = 10

    settings_interface = INearmePortletPopupRendererSettings
