#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from pyramid.events import subscriber
from zope.component.interfaces import ISite
from zope.interface import implementer
from zope.lifecycleevent.interfaces import IObjectAddedEvent
from zope.schema.fieldproperty import FieldProperty
from zope.schema.vocabulary import SimpleTerm, SimpleVocabulary

from onf_website.component.hearing import IHearingTarget
from onf_website.component.hearing.interfaces import IHearingManagerTarget
from onf_website.reference.universe.interfaces import IUniverse, IUniverseTable
from pyams_content.reference import ReferenceInfo
from pyams_content.shared.view.interfaces import IViewsManager
from pyams_content_es.shared.view.merge import get_merged_results
from pyams_i18n.interfaces import II18n, II18nManager
from pyams_sequence.reference import get_sequence_target
from pyams_utils.container import BTreeOrderedContainer
from pyams_utils.factory import factory_config
from pyams_utils.registry import get_utility, query_utility
from pyams_utils.request import check_request
from pyams_utils.traversing import get_parent
from pyams_utils.vocabulary import vocabulary_config
from pyams_workflow.interfaces import IWorkflow

__docformat__ = 'restructuredtext'


@factory_config(IUniverseTable)
@implementer(II18nManager, IHearingManagerTarget)
class UniverseTable(BTreeOrderedContainer):
    """Universe table"""

    title = FieldProperty(IUniverseTable['title'])
    short_name = FieldProperty(IUniverseTable['short_name'])
    dropdown_menu_title = FieldProperty(IUniverseTable['dropdown_menu_title'])
    inactive_info_title = FieldProperty(IUniverseTable['inactive_info_title'])
    inactive_info_text = FieldProperty(IUniverseTable['inactive_info_text'])
    online_help_title = FieldProperty(IUniverseTable['online_help_title'])
    online_help_text = FieldProperty(IUniverseTable['online_help_text'])

    languages = FieldProperty(II18nManager['languages'])

    def get_visible_items(self):
        yield from filter(lambda x: x.visible, self.values())


@subscriber(IObjectAddedEvent, context_selector=IUniverseTable)
def handle_added_universe_table(event):
    """Handle new universe table"""
    site = get_parent(event.object, ISite)
    registry = site.getSiteManager()
    if registry is not None:
        registry.registerUtility(event.object, IUniverseTable)


@factory_config(IUniverse)
@implementer(IHearingTarget)
class Universe(ReferenceInfo):
    """Universe record"""

    visible = FieldProperty(IUniverse['visible'])
    title = FieldProperty(IUniverse["title"])
    description = FieldProperty(IUniverse["description"])
    skin = FieldProperty(IUniverse['skin'])
    map_template = FieldProperty(IUniverse['map_template'])
    map_views = FieldProperty(IUniverse['map_views'])
    map_limit = FieldProperty(IUniverse['map_limit'])
    list_template = FieldProperty(IUniverse['list_template'])
    list_views = FieldProperty(IUniverse['list_views'])
    list_limit = FieldProperty(IUniverse['list_limit'])
    contact_target = FieldProperty(IUniverse["contact_target"])

    def _get_views(self, views):
        views_manager = get_utility(IViewsManager)
        workflow = IWorkflow(views_manager)
        for oid in views or ():
            view = get_sequence_target(oid, state=workflow.visible_states)
            if view is not None:
                yield view
    
    def get_map_views(self):
        yield from self._get_views(self.map_views)

    def get_list_views(self):
        yield from self._get_views(self.list_views)

    def get_map_items(self, request=None, aggregates=None, limit=None, **kwargs):
        if request is None:
            request = check_request()
        return get_merged_results(self.get_map_views(), context=self, request=request, get_user_params=True,
                                  aggregates=aggregates, limit=limit or self.map_limit or 999, **kwargs)

    def get_list_items(self, request=None, aggregates=None, limit=None, **kwargs):
        if request is None:
            request = check_request()
        return get_merged_results(self.get_list_views(), context=self, request=request, get_user_params=True,
                                  aggregates=aggregates,  limit=limit or self.list_limit or 999, **kwargs)


@vocabulary_config(name='ONF universe')
class UniverseVocabulary(SimpleVocabulary):
    """Universes vocabulary"""

    def __init__(self, context=None):
        table = query_utility(IUniverseTable)
        if table is not None:
            request = check_request()
            terms = [
                SimpleTerm(v,
                           token=v.__name__,
                           title=II18n(v).query_attribute('title', request=request))
                for v in table.values()
            ]
        else:
            terms = []
        super().__init__(terms)
