
__docformat__ = 'restructuredtext'

from persistent import Persistent
from pyramid.events import subscriber
from sqlalchemy.sql import and_, func, or_
from zope.component.interfaces import ISite
from zope.container.contained import Contained
from zope.interface import implementer
from zope.lifecycleevent.interfaces import IObjectAddedEvent
from zope.schema.fieldproperty import FieldProperty

from onf_website.reference.planning.interfaces import IPlanningTable
from onf_website.reference.planning.model import PARENT_SESSION as RDF_SESSION
from onf_website.reference.planning.model.planning import InformationPlanning, PlanningData, \
    PlanningForet
from pyams_alchemy.engine import get_user_session
from pyams_i18n.interfaces import II18nManager
from pyams_utils.traversing import get_parent


@implementer(IPlanningTable, II18nManager)
class PlanningTable(Persistent, Contained):
    """Planning table manager"""

    title = FieldProperty(IPlanningTable['title'])
    short_name = FieldProperty(IPlanningTable['short_name'])

    languages = FieldProperty(II18nManager['languages'])

    planning_displays = FieldProperty(IPlanningTable['planning_displays'])

    @property
    def visible_owners_ids(self):
        return [
            key
            for (key, value) in self.planning_displays.items()
            if value
        ]

    def get_published_plannings(self):
        session = get_user_session(RDF_SESSION)
        latest_by_forest = session.query(
                PlanningForet.id_nat_foret,
                func.max(PlanningData.date_debut_applicabilite).label('date_debut_applicabilite')
            ) \
            .outerjoin(PlanningForet,
                       PlanningForet.id_nat_amgt == PlanningData.id_nat_amgt) \
            .filter(PlanningData.code_categorie_proprietaire.in_(self.visible_owners_ids)) \
            .group_by(
                PlanningForet.id_nat_foret
            ) \
            .subquery()
        yield from session.query(PlanningData, InformationPlanning) \
            .outerjoin(PlanningForet,
                       PlanningForet.id_nat_amgt == PlanningData.id_nat_amgt) \
            .outerjoin(InformationPlanning,
                       PlanningData.id_nat_amgt == InformationPlanning.id_nat_amgt) \
            .join(latest_by_forest,
                  and_(
                      PlanningData.date_debut_applicabilite == latest_by_forest.c.date_debut_applicabilite,
                      PlanningForet.id_nat_foret == latest_by_forest.c.id_nat_foret
                  )) \
            .filter(and_(PlanningData.code_categorie_proprietaire.in_(self.visible_owners_ids),
                         or_(InformationPlanning.visible == None,
                             InformationPlanning.visible == True)))


@subscriber(IObjectAddedEvent, context_selector=IPlanningTable)
def handle_added_planning_table(event):
    """Handle new planning table"""
    site = get_parent(event.object, ISite)
    registry = site.getSiteManager()
    if registry is not None:
        registry.registerUtility(event.object, IPlanningTable)
