#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'


import json

from z3c.form.browser.widget import HTMLInputWidget
from z3c.form.converter import BaseDataConverter
from z3c.form.interfaces import IDataConverter, IFieldWidget
from z3c.form.widget import FieldWidget, Widget
from zope.interface import implementer_only
from zope.schema.fieldproperty import FieldProperty

from onf_website.reference.orga.model import Structure
from onf_website.reference.orga.schema import IStructureField, IStructuresListField
from onf_website.reference.orga.zmi.interfaces import IStructureWidget, IStructuresListWidget
from pyams_form.interfaces.form import IFormLayer
from pyams_form.widget import widgettemplate_config
from pyams_utils.adapter import adapter_config


#
# Structure field
#

@adapter_config(context=(IStructureField, IStructureWidget),
                provides=IDataConverter)
class StructureDataConverter(BaseDataConverter):
    """Structure data converter"""

    def toWidgetValue(self, value):
        if not value:
            return None
        return Structure.get(value).first()

    def toFieldValue(self, value):
        return value


@widgettemplate_config(mode='input',
                       template='templates/structure-input.pt', layer=IFormLayer)
@widgettemplate_config(mode='display',
                       template='templates/structure-display.pt', layer=IFormLayer)
@implementer_only(IStructureWidget)
class StructureWidget(HTMLInputWidget, Widget):
    """Structure widget"""

    structure_type = FieldProperty(IStructureWidget['structure_type'])
    category_code = FieldProperty(IStructuresListWidget['category_code'])

    @property
    def query_params(self):
        return json.dumps({
            'structure_type': self.structure_type,
            'category_code': self.category_code
        })

    @property
    def value_map(self):
        if not self.value:
            return ''
        return json.dumps({self.value.token: self.value.title})


@adapter_config(context=(IStructureField, IFormLayer),
                provides=IFieldWidget)
def StructureFieldWidget(field, request):
    """Structure field widget factory"""
    return FieldWidget(field, StructureWidget(request))


#
# Structures list field
#

@adapter_config(context=(IStructuresListField, IStructuresListWidget), provides=IDataConverter)
class StructuresListDataConverter(BaseDataConverter):
    """Structures list data converter"""

    def toWidgetValue(self, value):
        if not value:
            return ()
        return Structure.get(value).all()

    def toFieldValue(self, value):
        if not value:
            return []
        return value.split('|')


@widgettemplate_config(mode='input',
                       template='templates/structures-list-input.pt', layer=IFormLayer)
@widgettemplate_config(mode='display',
                       template='templates/structures-list-display.pt', layer=IFormLayer)
@implementer_only(IStructuresListWidget)
class StructuresListWidget(HTMLInputWidget, Widget):
    """Structures list widget"""

    structure_type = FieldProperty(IStructuresListWidget['structure_type'])
    category_code = FieldProperty(IStructuresListWidget['category_code'])

    @property
    def query_params(self):
        return json.dumps({
            'structure_type': self.structure_type,
            'category_code': self.category_code
        })

    @property
    def values(self):
        return '|'.join((structure.token for structure in self.value or ()))

    @property
    def values_map(self):
        result = {}
        [result.update({value.token: value.title}) for value in self.value or ()]
        return json.dumps(result)


@adapter_config(context=(IStructuresListField, IFormLayer),
                provides=IFieldWidget)
def StructuresListFieldWidget(field, request):
    """Structures list field widget factory"""
    return FieldWidget(field, StructuresListWidget(request))
