#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

import re

from geoalchemy2 import Geometry
from sqlalchemy import Column, Integer, Unicode
from sqlalchemy.ext.declarative import declared_attr
from sqlalchemy.orm import joinedload, relation
from sqlalchemy.schema import ForeignKey, ForeignKeyConstraint, UniqueConstraint
from sqlalchemy.sql.expression import and_, or_
from zope.interface import implementer, provider
from zope.schema.interfaces import ITitledTokenizedTerm

from onf_website.reference.insee.model.interfaces import IArrondissement, ICanton, ICodeLibelle, ICodePostal, ICommune, \
    ICommuneModel, IDepartement, IDepartementModel, IPays, IRegion
from pyams_alchemy import Base
from pyams_alchemy.engine import get_user_session
from pyams_alchemy.mixin import DynamicSchemaMixin
from pyams_utils.unicode import translate_string

__docformat__ = 'restructuredtext'


PARENT_SCHEMA = 'insee'
PARENT_SESSION = 'INSEE'


# Table 'INSEE_CODE_CDC'

@implementer(ICodeLibelle)
class CodeCDC(DynamicSchemaMixin, Base):
    """Content class for 'CODE_CDC' records"""

    __tablename__ = 'insee_code_cdc'
    __schema__ = PARENT_SCHEMA

    id = Column('id', Integer, primary_key=True)
    code = Column('code', Unicode(1), unique=True)
    libelle = Column('libelle', Unicode(50))


# Table 'INSEE_CODE_TNCC'

@implementer(ICodeLibelle)
class CodeTNCC(DynamicSchemaMixin, Base):
    """Content class for 'CODE_TNCC' records"""

    __tablename__ = 'insee_code_tncc'
    __schema__ = PARENT_SCHEMA

    id = Column('id', Integer, primary_key=True)
    code = Column('code', Unicode(1), unique=True)
    libelle = Column('libelle', Unicode(50))


# Table 'INSEE_CODE_TYPCT'

@implementer(ICodeLibelle)
class CodeTypCT(DynamicSchemaMixin, Base):
    """Content class for 'CODE_TYPCT' records"""

    __tablename__ = 'insee_code_typct'
    __schema__ = PARENT_SCHEMA

    id = Column('id', Integer, primary_key=True)
    code = Column('code', Unicode(1), unique=True)
    libelle = Column('libelle', Unicode(100))


# Table 'INSEE_CODE_CHEFLIEU'    

@implementer(ICodeLibelle)
class CodeChefLieu(DynamicSchemaMixin, Base):
    """Content class for 'CODE_CHEF_LIEU' records"""

    __tablename__ = 'insee_code_cheflieu'
    __schema__ = PARENT_SCHEMA

    id = Column('id', Integer, primary_key=True)
    code = Column('code', Unicode(1), unique=True)
    libelle = Column('libelle', Unicode(50))


# Table 'PAYS'

@implementer(IPays, ITitledTokenizedTerm)
class Pays(DynamicSchemaMixin, Base):
    """Content class for 'PAYS' records"""

    __tablename__ = 'insee_pays'
    __schema__ = PARENT_SCHEMA

    id = Column('id', Integer, primary_key=True)
    cog = Column('cog', Unicode(5), unique=True)
    actual = Column('actual', Unicode(1))
    capay = Column('capay', Unicode(5))
    crpay = Column('crpay', Unicode(5))
    ani = Column('ani', Unicode(4))
    libcog = Column('libcog', Unicode(70))
    libenr = Column('libenr', Unicode(70))
    ancnom = Column('ancnom', Unicode(20))
    code_iso = Column('code_iso', Unicode(2))

    @property
    def value(self):
        return self.cog

    @property
    def token(self):
        return self.cog

    @property
    def shortname(self):
        return self.libcog

    @property
    def title(self):
        return self.shortname

    @property
    def longname(self):
        return self.title

    @classmethod
    def get_session(cls, session=PARENT_SESSION):
        if isinstance(session, str):
            session = get_user_session(session)
        return session

    @classmethod
    def get(cls, pays, actual=True, session=PARENT_SESSION):
        if isinstance(pays, (list, tuple, set)):
            params = [Pays.cog.in_(pays)]
        else:
            params = [Pays.cog == pays]
        if actual:
            params.append(Pays.actual == u'1')
        session = Pays.get_session(session)
        return session.query(Pays).filter(and_(*params))

    @classmethod
    def find(cls, query, actual=True, session=PARENT_SESSION):
        if isinstance(query, dict):
            pays = query.get('pays')
            label = query.get('label')
        else:
            query = query.strip()
            try:
                _check = int(query)
            except:
                pays = None
                label = query
            else:
                pays = query
                label = None
        session = Pays.get_session(session)
        if pays:
            return Pays.find_by_insee_code(pays, actual, session)
        elif label:
            return Pays.find_by_label(label, actual, session)

    @classmethod
    def find_by_insee_code(cls, pays, actual=True, session=PARENT_SESSION):
        if isinstance(pays, (list, tuple, set)):
            params = Pays.cog.in_(pays)
        else:
            params = Pays.cog == pays
        if actual:
            params = and_(params, Pays.actual == u'1')
        session = Pays.get_session(session)
        return session.query(Pays).filter(params)

    @classmethod
    def find_by_label(cls, label, actual=True, session=PARENT_SESSION):
        session = Pays.get_session(session)
        base_label = translate_string(label, escape_slashes=False, force_lower=False).upper()
        upcase_label = label.upper()
        params = or_(Pays.libenr.like('%%%s%%' % base_label),
                     Pays.libenr.like('%%%s%%' % upcase_label),
                     Pays.libcog.like('%%%s%%' % base_label),
                     Pays.libcog.like('%%%s%%' % upcase_label))
        if actual:
            params = and_(params, Pays.actual == u'1')
        return session.query(Pays).filter(params)


# Table 'REGION'

@implementer(IRegion, ITitledTokenizedTerm)
class Region(DynamicSchemaMixin, Base):
    """Content class for 'REGION' records"""

    __tablename__ = 'insee_region'
    __schema__ = PARENT_SCHEMA

    id = Column('id', Integer, primary_key=True)
    region = Column('region', Unicode(2), unique=True)
    cheflieu = Column('cheflieu', Unicode(5))
    tncc = Column('tncc', Unicode, ForeignKey(CodeTNCC.code))
    ncc = Column('ncc', Unicode)
    nccenr = Column('nccenr', Unicode)

    @property
    def value(self):
        return self.region

    @property
    def token(self):
        return self.region

    @property
    def shortname(self):
        return self.nccenr

    @property
    def title(self):
        return self.shortname

    @property
    def longname(self):
        return self.title

    @classmethod
    def get_session(cls, session=PARENT_SESSION):
        if isinstance(session, str):
            session = get_user_session(session)
        return session

    @classmethod
    def get(cls, reg, session=PARENT_SESSION):
        return Region.find_by_insee_code(reg, session)

    @classmethod
    def find(cls, query, session=PARENT_SESSION):
        if isinstance(query, dict):
            reg = query.get('reg')
            label = query.get('label')
        else:
            query = query.strip()
            try:
                _check = int(query)
            except:
                reg = None
                label = query
            else:
                reg = query
                label = None
        session = Region.get_session(session)
        if reg:
            return Region.find_by_insee_code(reg, session)
        elif label:
            return Region.find_by_label(label, session)

    @classmethod
    def find_by_insee_code(cls, reg, session=PARENT_SESSION):
        if isinstance(reg, (list, tuple, set)):
            params = Region.region.in_(reg)
        else:
            params = Region.region == reg
        session = Region.get_session(session)
        return session.query(Region).filter(params)

    @classmethod
    def find_by_label(cls, label, session=PARENT_SESSION):
        session = Region.get_session(session)
        label = translate_string(label, escape_slashes=False, force_lower=False).upper()
        return session.query(Region).filter(Region.ncc.like('%%%s%%' % label))


# Table 'DEPARTEMENT'

REG_DEPT_CODE = re.compile(r'\d{2,3}')


@provider(IDepartementModel)
@implementer(IDepartement, ITitledTokenizedTerm)
class Departement(DynamicSchemaMixin, Base):
    """Content class for 'DEPARTEMENT' records"""

    __tablename__ = 'insee_departement'
    __schema__ = PARENT_SCHEMA
    __entity__ = 'DPT'

    id = Column('id', Integer, primary_key=True)
    region = Column('region', Unicode(2), ForeignKey(Region.region))
    dep = Column('dep', Unicode(3), unique=True)
    cheflieu = Column('cheflieu', Unicode(5))
    tncc = Column('tncc', Unicode(1), ForeignKey(CodeTNCC.code))
    ncc = Column('ncc', Unicode(70))
    nccenr = Column('nccenr', Unicode(70))

    @property
    def value(self):
        return self.dep

    @property
    def token(self):
        return self.dep

    @property
    def shortname(self):
        return self.nccenr

    @property
    def title(self):
        return u'%s (%s)' % (self.nccenr, self.dep)

    @property
    def longname(self):
        return self.title

    @classmethod
    def get_session(cls, session=PARENT_SESSION):
        if isinstance(session, str):
            session = get_user_session(session)
        return session

    @classmethod
    def get(cls, dep, session=PARENT_SESSION):
        return Departement.find_by_insee_code(dep, session)

    @classmethod
    def find(cls, query, session=PARENT_SESSION):
        if isinstance(query, dict):
            dep = query.get('dep')
            label = query.get('label')
        else:
            query = query.strip()
            try:
                _check = int(query)
            except:
                dep = None
                label = query
            else:
                dep = query
                label = None
        session = Departement.get_session(session)
        if dep:
            return Departement.find_by_insee_code(dep, session)
        elif label:
            return Departement.find_by_label(label, session)

    @classmethod
    def find_by_insee_code(cls, dep, session=PARENT_SESSION):
        if isinstance(dep, (list, tuple, set)):
            params = Departement.dep.in_(dep)
        else:
            params = Departement.dep == dep
        session = Departement.get_session(session)
        return session.query(Departement).filter(params)

    @classmethod
    def find_by_label(cls, label, session=PARENT_SESSION):
        session = Departement.get_session(session)
        label = translate_string(label, escape_slashes=False, force_lower=False).upper()
        return session.query(Departement).filter(Departement.ncc.like('%%%s%%' % label))

    @classmethod
    def search(cls, form):
        return ()


class DepartementGeom(DynamicSchemaMixin, Base):
    """Content class for departments with geometry"""

    __tablename__ = 'insee_departement_geom'
    __schema__ = PARENT_SCHEMA

    gid = Column(Integer, primary_key=True)
    code = Column('insee_dept', Unicode(3), ForeignKey(Departement.dep), unique=True)
    geom = Column(Geometry('MULTIPOLYGON', 4326))


# Table 'ARRONDISSEMENTS'

@implementer(IArrondissement)
class Arrondissement(DynamicSchemaMixin, Base):
    """Content class for 'ARRONDISSEMENT' records"""

    __tablename__ = 'insee_arrondissement'
    __schema__ = PARENT_SCHEMA

    @declared_attr
    def __table_args__(cls):
        return (
            UniqueConstraint('dep', 'ar'),
            cls.get_schema()
        )

    id = Column('id', Integer, primary_key=True)
    region = Column('region', Unicode(2), ForeignKey(Region.region))
    dep = Column('dep', Unicode(3), ForeignKey(Departement.dep))
    ar = Column('ar', Unicode(1))
    cheflieu = Column('cheflieu', Unicode(5))
    tncc = Column('tncc', Unicode(1), ForeignKey(CodeTNCC.code))
    artmaj = Column('artmaj', Unicode(5))
    ncc = Column('ncc', Unicode(70))
    artmin = Column('artmin', Unicode(5))
    nccenr = Column('nccenr', Unicode(70))


# Table 'CANTONS'

@implementer(ICanton)
class Canton(DynamicSchemaMixin, Base):
    """Content class for 'CANTON records"""

    __tablename__ = 'insee_canton'
    __schema__ = PARENT_SCHEMA

    @declared_attr
    def __table_args__(cls):
        return (
            UniqueConstraint('dep', 'ar', 'canton'),
            ForeignKeyConstraint(['dep', 'ar'], [Arrondissement.dep, Arrondissement.ar]),
            cls.get_schema()
        )

    id = Column('id', Integer, primary_key=True)
    region = Column('region', Unicode(2), ForeignKey(Region.region))
    dep = Column('dep', Unicode(3), ForeignKey(Departement.dep))
    ar = Column('ar', Unicode(2))
    canton = Column('canton', Unicode(2))
    typct = Column('typct', Unicode(1), ForeignKey(CodeTypCT.code))
    cheflieu = Column('cheflieu', Unicode(5))
    tncc = Column('tncc', Unicode(1), ForeignKey(CodeTNCC.code))
    artmaj = Column('artmaj', Unicode(5))
    ncc = Column('ncc', Unicode(70))
    artmin = Column('artmin', Unicode(5))
    nccenr = Column('nccenr', Unicode(70))


# Table 'COMMUNE'

_prefix = ("aux", "l'", "l", "le", "la", "les", "los")
_check = re.compile("([^\'\s]*)[\s\']*(.*)")

REG_COMMUNE_CODE = re.compile(r'\d[A-Z\d]\d{3}')


def _check_label(label):
    label = label.lower()
    match = _check.match(label)
    if match:
        groups = match.groups()
        if groups[0] in _prefix:
            label = groups[1].replace(' ', '-')
        else:
            label = label.replace(' ', '-')
    else:
        label = label.replace(' ', '-')
    return label


@provider(ICommuneModel)
@implementer(ICommune, ITitledTokenizedTerm)
class Commune(DynamicSchemaMixin, Base):
    """Content class for 'COMMUNE' records"""

    __tablename__ = 'insee_commune'
    __schema__ = PARENT_SCHEMA
    __entity__ = 'COM'

    @declared_attr
    def __table_args__(cls):
        return (
            UniqueConstraint('dep', 'com'),
            ForeignKeyConstraint(['dep', 'ar'], [Arrondissement.dep, Arrondissement.ar]),
            cls.get_schema()
        )

    id = Column(Integer, primary_key=True)
    cdc = Column(Unicode(1), ForeignKey(CodeCDC.code))
    cheflieu = Column(Unicode(1), ForeignKey(CodeChefLieu.code))
    reg = Column(Unicode(2), ForeignKey(Region.region))
    dep = Column(Unicode(3), ForeignKey(Departement.dep))
    com = Column(Unicode(3))
    ar = Column(Unicode(1))
    ct = Column(Unicode(2))
    tncc = Column(Unicode(1), ForeignKey(CodeTNCC.code))
    artmaj = Column(Unicode(5))
    ncc = Column(Unicode(70))
    artmin = Column(Unicode(5))
    nccenr = Column(Unicode(70))
    code = Column(Unicode(5))

    @property
    def value(self):
        return self.code

    @property
    def token(self):
        return self.code

    @property
    def shortname(self):
        return u'%s %s' % (self.artmin or '', self.nccenr)

    @property
    def title(self):
        return u'%s %s (%s)' % (self.artmin or '', self.nccenr, self.dep)

    @property
    def title_enr(self):
        if self.artmin:
            return '{} {}'.format(self.artmin[1:-1], self.nccenr)
        else:
            return self.nccenr

    @property
    def longname(self):
        return u'%s %s (%s - %s)' % (self.artmin or '', self.nccenr, self.dep, self.ref_departement.nccenr)

    @classmethod
    def get_session(cls, session=PARENT_SESSION):
        if isinstance(session, str):
            session = get_user_session(session)
        return session

    @classmethod
    def get(cls, insee_code, session=PARENT_SESSION):
        return Commune.find_by_insee_code(insee_code, session)

    @classmethod
    def find(cls, query, session=PARENT_SESSION, get_codes=False):
        if isinstance(query, dict):
            insee = query.get('insee')
            postal_code = query.get('postal_code')
            label = query.get('label')
        else:
            query = query.strip()
            try:
                _check = int(query)
            except:
                insee = None
                postal_code = None
                label = query
            else:
                insee = None
                postal_code = query
                label = None
        session = Commune.get_session(session)
        if insee:
            return Commune.find_by_insee_code(insee, session, get_codes)
        elif postal_code:
            return Commune.find_by_postal_code(postal_code, session, get_codes)
        elif label:
            return Commune.find_by_label(label, session, get_codes)

    @classmethod
    def find_by_insee_code(cls, insee_code, session=PARENT_SESSION, get_codes=False):
        if isinstance(insee_code, (list, tuple, set)):
            params = Commune.code.in_(insee_code)
        else:
            params = Commune.code == insee_code
        session = Commune.get_session(session)
        query = session.query(Commune)
        if get_codes:
            query = query.options(joinedload(Commune.ref_codespostaux))
        return query.filter(params)

    @classmethod
    def find_by_postal_code(cls, postal_code, session=PARENT_SESSION, get_codes=False):
        if isinstance(postal_code, (list, tuple, set)):
            params = CodePostal.code_postal.in_(postal_code)
        else:
            params = CodePostal.code_postal == postal_code
        session = Commune.get_session(session)
        query = session.query(Commune) \
                       .join(CodePostal,
                             Commune.dep + Commune.com == CodePostal.code_insee)
        if get_codes:
            query = query.options(joinedload(Commune.ref_codespostaux))
        return query.filter(params)

    @classmethod
    def find_by_label(cls, label, session=PARENT_SESSION, get_codes=False):
        session = Commune.get_session(session)
        label = _check_label(translate_string(label,
                                              keep_chars="_-.'",
                                              escape_slashes=False,
                                              force_lower=False)).upper()
        query = session.query(Commune).filter(Commune.ncc.like('%%%s%%' % label))
        if get_codes:
            query = query.options(joinedload(Commune.ref_codespostaux))
        return query

    @classmethod
    def search(cls, form):
        return ()


class CommuneGeom(DynamicSchemaMixin, Base):
    """Content class for communes with geometry"""

    __tablename__ = 'insee_commune_geom'
    __schema__ = PARENT_SCHEMA

    gid = Column(Integer, primary_key=True)
    id = Column(Unicode(24))
    code = Column('insee_com', Unicode(5), ForeignKey(Commune.code))
    geom = Column(Geometry('MULTIPOLYGON', 4326))


# Table 'HEXAPOSTE' 

@implementer(ICodePostal)
class CodePostal(DynamicSchemaMixin, Base):
    """Content class for 'HEXAPOSTE' records"""

    __tablename__ = 'hexaposte'
    __schema__ = PARENT_SCHEMA

    id = Column('id', Integer, primary_key=True)
    code_insee = Column('codeinsee', Unicode(5))
    reserve = Column('reserve', Unicode(1))
    libelle = Column('libelle', Unicode(38))
    code_postal = Column('codepostal', Unicode(5))
    distributeur = Column('distributeur', Unicode(1))
    libelle_cedex = Column('libellecedex', Unicode(26))
    filler = Column('filler', Unicode(5))


# Foreign keys

Region.ref_cheflieu = relation(Commune,
                               primaryjoin=Region.cheflieu==Commune.code,
                               foreign_keys=Commune.code,
                               viewonly=True,
                               uselist=False)
Region.ref_tncc = relation(CodeTNCC)

Departement.ref_region = relation(Region, backref='ref_departements')
Departement.ref_cheflieu = relation(Commune,
                                    primaryjoin=Departement.cheflieu==Commune.code,
                                    foreign_keys=Commune.code,
                                    viewonly=True,
                                    uselist=False)
Departement.ref_tncc = relation(CodeTNCC)

Departement.ref_geom = relation(DepartementGeom,
                                primaryjoin=Departement.dep==DepartementGeom.code,
                                uselist=False)

Arrondissement.ref_region = relation(Region)
Arrondissement.ref_departement = relation(Departement)
Arrondissement.ref_cheflieu = relation(Commune,
                                       primaryjoin=Arrondissement.cheflieu==Commune.code,
                                       foreign_keys=Commune.code,
                                       viewonly=True,
                                       uselist=False)
Arrondissement.ref_tncc = relation(CodeTNCC)

Canton.ref_region = relation(Region)
Canton.ref_departement = relation(Departement)
Canton.ref_arrondissement = relation(Arrondissement,
                                     primaryjoin=and_(Canton.dep == Arrondissement.dep,
                                                      Canton.ar == Arrondissement.ar),
                                     foreign_keys=[Arrondissement.dep, Arrondissement.ar])
Canton.ref_typect = relation(CodeTypCT)
Canton.ref_cheflieu = relation(Commune,
                               primaryjoin=Canton.cheflieu==Commune.code,
                               foreign_keys=Commune.code,
                               viewonly=True,
                               uselist=False)
Canton.ref_tncc = relation(CodeTNCC)

Commune.ref_cdc = relation(CodeCDC)
Commune.ref_codecheflieu = relation(CodeChefLieu)
Commune.ref_region = relation(Region)
Commune.ref_departement = relation(Departement)
Commune.ref_arrondissement = relation(Arrondissement,
                                      primaryjoin=and_(Commune.dep == Arrondissement.dep,
                                                       Commune.ar == Arrondissement.ar),
                                      foreign_keys=[Arrondissement.dep, Arrondissement.ar])
Commune.ref_tncc = relation(CodeTNCC)
Commune.ref_codespostaux = relation(CodePostal,
                                    primaryjoin=CodePostal.code_insee==Commune.code,
                                    foreign_keys=CodePostal.code_insee,
                                    order_by=CodePostal.code_postal,
                                    viewonly=True)

Commune.ref_geom = relation(CommuneGeom,
                            primaryjoin=Commune.code==CommuneGeom.code,
                            uselist=False)

CodePostal.ref_communes = relation(Commune,
                                   primaryjoin=CodePostal.code_insee==Commune.code,
                                   foreign_keys=Commune.code,
                                   viewonly=True)
