# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#
from datetime import datetime, timezone

from elasticsearch_dsl import Q, Search
from pyramid.view import view_config
from pyramid_es import get_client

from onf_website.reference.forest import IForestTable
from onf_website.reference.hunting.model import SaisonChasse
from onf_website.shared.hunting import HUNTING_CONTENT_TYPE, IHuntingManager
from onf_website.shared.hunting.interfaces import IHuntingCalendarSearchFolder
from pyams_default_theme.features.sitemap import ISitemapExtension
from pyams_i18n.interfaces import II18n
from pyams_skin.layer import IPyAMSUserLayer
from pyams_utils.adapter import ContextRequestAdapter, adapter_config
from pyams_utils.interfaces.site import ISiteRoot
from pyams_utils.registry import get_utility, query_utility

__docformat__ = 'restructuredtext'

from pyams_workflow.interfaces import IWorkflow


@adapter_config(name='hunting-calendars',
                context=(ISiteRoot, IPyAMSUserLayer),
                provides=ISitemapExtension)
class SiteRoothuntingCalendarsSitemap(ContextRequestAdapter):
    """Site root hunting calendars sitemap extension"""
    
    def __new__(cls, context, request):
        table = query_utility(IForestTable)
        if table is None:
            return None
        manager = query_utility(IHuntingManager)
        if manager is None:
            return None
        source = manager.get_default_source()
        if not IHuntingCalendarSearchFolder.providedBy(source):
            return None
        return ContextRequestAdapter.__new__(cls)
    
    @property
    def source(self):
        manager = get_utility(IHuntingManager)
        return manager.get_default_source()
    
    
@view_config(name='sitemap.xml',
             context=IHuntingCalendarSearchFolder,
             request_type=IPyAMSUserLayer,
             renderer='templates/hunting-sitemap.pt')
class HuntingCalendarSitemapView:
    """Hunting calendar sitemap view"""
    
    def __init__(self, request):
        self.request = request
    
    @property
    def title_prefix(self):
        manager = get_utility(IHuntingManager)
        return II18n(manager).query_attribute('title_prefix', request=self.request)
    
    @staticmethod
    def get_published_hunting_calendars():
        table = get_utility(IForestTable)
        yield from table.get_published_hunting_calendars()
        
    def get_contents(self):
        """Get indexed contents with location forest"""
        saison = SaisonChasse.get_current_or_next()
        if saison is None:
            date_saison = datetime.now(timezone.utc).isoformat()
        else:
            date_saison = saison.date_debut_saison.isoformat()
        wf_params = []
        for workflow in self.request.registry.getAllUtilitiesRegisteredFor(IWorkflow):
            wf_params.extend(workflow.visible_states)
        params = \
            Q('term',
              **{'content_type': HUNTING_CONTENT_TYPE}) & \
            Q('terms',
              **{'workflow.status': wf_params}) & \
            Q('range',
              **{'workflow.effective_date': {'lte': 'now/m'}}) & \
            (Q('bool',
               must=Q('range',
                      **{'workflow.push_end_date': {'gte': 'now/m'}})) |
             Q('bool',
               must_not=Q('exists',
                          **{'field': 'workflow.push_end_date'}))) & \
            Q('bool',
              must=Q('exists',
                     **{'field': 'location.forests'}))
        es_client = get_client(self.request)
        contents = {}
        for content in Search(using=es_client.es, index=es_client.index) \
                .params(request_timeout=60) \
                .query(params) \
                .source([
                    'location.forests',
                    'workflow.modified_date'
                ]):
            for forest in content.location.forests or ():
                if forest in contents:
                    contents[forest] = max(contents[forest],
                                           content.workflow.modified_date,
                                           date_saison)
                else:
                    contents[forest] = max(content.workflow.modified_date,
                                           date_saison)
        return contents
    
    def __call__(self):
        self.request.response.content_type = 'text/xml'
        self.contents = self.get_contents()
        return {}
