# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from datetime import date

from sqlalchemy import Boolean, Column, Date, Integer, Numeric, TIMESTAMP as Timestamp, VARCHAR as Varchar
from sqlalchemy.orm import joinedload, relation
from sqlalchemy.schema import ForeignKey
from sqlalchemy.sql import and_, or_

from onf_website.reference.forest.model.foret import Foret
from pyams_alchemy import Base
from pyams_alchemy.engine import get_user_session
from pyams_alchemy.mixin import DynamicSchemaMixin

__docformat__ = 'restructuredtext'


PARENT_SCHEMA = 'artemis'
PARENT_SESSION = 'ARTEMIS'


class ContratChasse(DynamicSchemaMixin, Base):
    """Contrat-Chasse model class"""
    
    __tablename__ = 'contrat_chasse'
    __schema__ = PARENT_SCHEMA
    
    id = Column(Integer, primary_key=True)
    id_nat_contrat = Column(Varchar(8), unique=True)
    code_chasse_type = Column(Integer)
    code_contrat_statut = Column(Integer)
    structure_gestionnaire = Column(Varchar(6))
    
    date_debut_validite = Column(Date)
    date_fin_validite = Column(Date)


class LotChasse(DynamicSchemaMixin, Base):
    """Lot-Chasse model class"""
    
    __tablename__ = 'lot_chasse'
    __schema__ = PARENT_SCHEMA
    
    id = Column(Integer, primary_key=True)
    id_nat_lot = Column(Varchar(8), unique=True)
    code = Column(Varchar(50))
    code_foret_majoritaire = Column('foret_majoritaire', Varchar(7))
    id_local = Column(Varchar(20))
    numero = Column(Varchar(20))
    libelle_long = Column(Varchar(150))
    superficie = Column('superficie_declaree', Numeric(38, 8))
    date_debut_validite = Column(Date)
    date_fin_validite = Column(Date)
    code_lot_statut = Column(Integer)
    
    @classmethod
    def find_lots(cls, forest_ids, session=PARENT_SESSION):
        session = get_user_session(session)
        if isinstance(forest_ids, str):
            forest_ids = [forest_ids]
        return session.query(LotChasse) \
            .filter(LotChasse.code_foret_majoritaire.in_(forest_ids))


class LotAttribution(DynamicSchemaMixin, Base):
    """Lot-Attribution model class"""
    
    __tablename__ = 'lot_attribution'
    __schema__ = PARENT_SCHEMA
    
    id = Column(Integer, primary_key=True)
    id_nat_contrat = Column(Varchar(8),
                            ForeignKey(ContratChasse.id_nat_contrat))
    id_nat_lot = Column(Varchar(8),
                        ForeignKey(LotChasse.id_nat_lot))
    date_debut_validite = Column(Date)
    date_fin_validite = Column(Date)
    
    contrat = relation(ContratChasse, backref='attributions')
    lot = relation(LotChasse, backref='attributions')
    
    
class LotForet(DynamicSchemaMixin, Base):
    """Lot-Foret model class"""
    
    __tablename__ = 'lot_foret'
    __schema__ = PARENT_SCHEMA
    
    objectid = Column(Integer, primary_key=True)
    id_nat_lot = Column(Varchar, ForeignKey(LotChasse.id_nat_lot))
    id_nat_frt = Column(Varchar(7), ForeignKey(Foret.id_nat_frt))
    superficie = Column('superficie_declaree', Numeric(38, 8))
    date_debut_validite = Column(Timestamp)
    date_fin_validite = Column(Timestamp)

    lot = relation(LotChasse, backref='lots_forets')
    foret = relation(Foret, backref='lots_chasse')


class SaisonChasse(DynamicSchemaMixin, Base):
    """Hunting season model class"""
    
    __tablename__ = 'ref_chasse_saison'
    __schema__ = PARENT_SCHEMA
    
    id = Column(Integer, primary_key=True)
    saison = Column(Varchar(11))
    code = Column(Varchar(4))
    date_debut_validite = Column(Date)
    date_fin_validite = Column(Date)
    date_debut_saison = Column(Date)
    date_fin_saison = Column(Date)
    
    @classmethod
    def get_current_or_next(cls, session=PARENT_SESSION):
        session = get_user_session(session)
        today = date.today()
        return session.query(SaisonChasse) \
            .filter(SaisonChasse.date_fin_saison >= today) \
            .order_by(SaisonChasse.date_debut_validite) \
            .first()


class GestionDate(DynamicSchemaMixin, Base):
    """Lot hunting date model class"""

    __tablename__ = 'gestion_date'
    __schema__ = PARENT_SCHEMA

    id = Column(Integer, primary_key=True)
    id_date_portail = Column(Integer)
    id_nat_contrat = Column(Varchar(10), ForeignKey(ContratChasse.id_nat_contrat))
    id_saison = Column(Integer, ForeignKey(SaisonChasse.id))
    date_chasse = Column(Date)
    code_mode_chasse = Column('code_chasse_mode', Varchar(3))
    valide = Column(Varchar(1))
    flag_actif = Column(Integer)

    contrat = relation(ContratChasse, backref='dates_chasse')
    saison = relation(SaisonChasse, backref='dates_chasse')

    @classmethod
    def get_dates(cls, forest_ids, current_season=True, valid_only=False, session=PARENT_SESSION):
        today = date.today()
        if isinstance(forest_ids, str):
            forest_ids = [forest_ids]
        params = [
            ContratChasse.code_chasse_type == 1,  # chasse à tir collectif
            ContratChasse.code_contrat_statut == 2,  # contrat validé
            or_(ContratChasse.date_fin_validite == None,
                ContratChasse.date_fin_validite > today),
            LotChasse.code_lot_statut == 2,  # lot validé
            or_(LotChasse.code_foret_majoritaire.in_(forest_ids),
                LotForet.id_nat_frt.in_(forest_ids)),
            or_(LotChasse.date_fin_validite == None,
                LotChasse.date_fin_validite > today),
            GestionDate.valide.in_(('2',) if valid_only else ('1', '2')),
            GestionDate.code_mode_chasse == 'COL'
        ]
        if current_season:
            saison = SaisonChasse.get_current_or_next(session)
            if saison is not None:
                params.append(GestionDate.id_saison == saison.id)
            else:
                params.append(GestionDate.id_saison == None)
        session = get_user_session(session)
        query = session.query(GestionDate) \
            .options(joinedload(GestionDate.contrat)) \
            .join(ContratChasse, ContratChasse.id_nat_contrat == GestionDate.id_nat_contrat) \
            .outerjoin(LotAttribution, LotAttribution.id_nat_contrat == ContratChasse.id_nat_contrat) \
            .outerjoin(LotChasse, LotChasse.id_nat_lot == LotAttribution.id_nat_lot) \
            .outerjoin(LotForet, LotForet.id_nat_lot == LotChasse.id_nat_lot)
        return query.filter(and_(*params))
    
    @classmethod
    def get_unvalidated_dates(cls, forest_ids, current_season=True, session=PARENT_SESSION):
        today = date.today()
        if isinstance(forest_ids, str):
            forest_ids = [forest_ids]
        params = [
            ContratChasse.code_chasse_type == 1,  # chasse à tir collectif
            ContratChasse.code_contrat_statut == 2,  # contrat validé
            or_(ContratChasse.date_fin_validite == None,
                ContratChasse.date_fin_validite > today),
            LotChasse.code_lot_statut == 2,  # lot validé
            or_(LotChasse.code_foret_majoritaire.in_(forest_ids),
                LotForet.id_nat_frt.in_(forest_ids)),
            or_(LotChasse.date_fin_validite == None,
                LotChasse.date_fin_validite > today),
            GestionDate.valide == '1',
            GestionDate.code_mode_chasse == 'COL'
        ]
        if current_season:
            saison = SaisonChasse.get_current_or_next(session)
            if saison is not None:
                params.append(GestionDate.id_saison == saison.id)
            else:
                params.append(GestionDate.id_saison == None)
        session = get_user_session(session)
        query = session.query(GestionDate) \
            .options(joinedload(GestionDate.contrat)) \
            .join(ContratChasse, ContratChasse.id_nat_contrat == GestionDate.id_nat_contrat) \
            .outerjoin(LotAttribution, LotAttribution.id_nat_contrat == ContratChasse.id_nat_contrat) \
            .outerjoin(LotChasse, LotChasse.id_nat_lot == LotAttribution.id_nat_lot) \
            .outerjoin(LotForet, LotForet.id_nat_lot == LotChasse.id_nat_lot)
        return query.filter(and_(*params))


class LotPerimetre(DynamicSchemaMixin, Base):
    """Lot-Perimeter model class"""
    
    __tablename__ = 'lot_perimetre'
    __schema__ = PARENT_SCHEMA
    
    objectid = Column(Integer, primary_key=True)
    # shape = Column(SdeGeometry('MULTIPOLYGON', RGF93))
    id_nat_lot = Column(Varchar, ForeignKey(LotChasse.id_nat_lot))
    code = Column(Varchar)
    date_debut_validite = Column(Timestamp)
    date_fin_validite = Column(Timestamp)
    actif = Column(Boolean)
    
    lot = relation(LotChasse, backref='perimetres')
