#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from pyramid.decorator import reify
from pyramid.httpexceptions import HTTPNotFound
from pyramid.url import resource_url
from pyramid.view import view_config
from sqlalchemy.sql import func, or_
from z3c.form.field import Fields
from z3c.form.interfaces import DISPLAY_MODE, HIDDEN_MODE, INPUT_MODE
from z3c.table.interfaces import IColumn, IValues
from zope.interface import Interface, implementer
from zope.schema import Choice, TextLine

from onf_website.reference.forest import IForestTable
from onf_website.reference.forest.interfaces import FOREST_OWNERS_CATEGORIES
from onf_website.reference.forest.model.code import Code
from onf_website.reference.forest.model.foret import Foret, InformationForet, \
    PARENT_SESSION as FORET_SESSION, ProprietaireForet
from onf_website.reference.forest.model.interfaces.foret import IForet, IInformationForet, \
    StatutForetException
from onf_website.reference.forest.schema import ForestsListField
from onf_website.reference.insee.schema import DepartmentField
from onf_website.reference.orga.schema import StructureField
from pyams_alchemy.engine import get_user_session
from pyams_content.interfaces import MANAGE_SITE_ROOT_PERMISSION
from pyams_content.reference.zmi.table import ReferenceTableHeaderAdapter
from pyams_form.form import ajax_config
from pyams_form.search import SearchForm, SearchResultsView, SearchView
from pyams_pagelet.interfaces import PageletCreatedEvent
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.interfaces import IContentSearch, IInnerPage, IPageHeader
from pyams_skin.interfaces.container import ITableElementEditor
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.skin import apply_skin
from pyams_skin.table import AttributeSwitcherColumn, ImageColumn, NameColumn, \
    VisibilitySwitcherColumn
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION
from pyams_utils.unicode import translate_string
from pyams_utils.url import absolute_url
from pyams_zmi.form import AdminDialogEditForm
from pyams_zmi.layer import IAdminLayer
from pyams_zmi.view import AdminView


__docformat__ = 'restructuredtext'

from onf_website import _


class IForestSearchFields(Interface):
    """Forest search fields"""

    label = TextLine(title=_("Free search"),
                     description=_("Search forest label for this text"),
                     required=False)

    forest_ids = ForestsListField(title=_("Forest Ids"),
                                  description=_("List of searched forests"),
                                  required=False)

    department = DepartmentField(title=_("Department"),
                                 description=_("Department of searched forests"),
                                 required=False)

    structure = StructureField(title=_("Management structure"),
                               description=_("Human resources attachment structure"),
                               required=False)

    owners_category = Choice(title=_("Owner category"),
                             description=_("Owner category selection"),
                             required=False,
                             vocabulary=FOREST_OWNERS_CATEGORIES)


@implementer(IInnerPage)
class ForestTableSearchForm(SearchForm):
    """Forest table search form"""

    legend = _("Forest search")

    fields = Fields(IForestSearchFields)
    sort_results = True

    prefix = 'search_form.'
    ajax_handler = 'forest-search-results.html'

    def __init__(self, context, request):
        super().__init__(context, request)
        request.registry.notify(PageletCreatedEvent(self))
        apply_skin(request, 'PyAMS admin skin')

    def updateWidgets(self, prefix=None):
        super().updateWidgets(prefix)
        structure = self.widgets.get('structure')
        if structure is not None:
            structure.category_code = 'UT'


@adapter_config(context=(IForestTable, IPyAMSLayer, ForestTableSearchForm),
                provides=IContentSearch)
class ForestTableSearchFormResultsAdapter(ContextRequestViewAdapter):
    """Forest table search form results adapter"""

    def get_search_results(self, data):
        """Search results getter"""
        if not data:
            return
        session = get_user_session(FORET_SESSION)
        filters = []
        label = data.get('label')
        if label:
            translated_label = translate_string(label, force_lower=False)
            filters.append(or_(func.unaccent(func.lower(Foret.libelle_usage)).contains(translated_label.lower()),
                               Foret.libelle_recherche.contains(translated_label.upper())))
        forest_ids = data.get('forest_ids')
        if forest_ids:
            filters.append(Foret.id_nat_frt.in_(forest_ids))
        department = data.get('department')
        if department:
            filters.append(Foret.departement == department)
        structure = data.get('structure')
        if structure:
            filters.append(Foret.code_structure_rh_gestionnaire == structure)
        category = data.get('owners_category')
        if category:
            filters.append(ProprietaireForet.categorie == category)
        if not filters:
            return
        query = session.query(Foret, InformationForet, Code) \
            .join(ProprietaireForet,
                  Foret.id_nat_frt == ProprietaireForet.id_nat_frt) \
            .join(Code,
                  (ProprietaireForet.categorie == Code.code) &
                  (Code.unite == 'CATEGORIE_PROPRIETAIRE')) \
            .outerjoin(InformationForet,
                       Foret.id_nat_frt == InformationForet.id_nat_frt)
        yield from query.filter(*filters).limit(999)


@pagelet_config(name='contents.html',
                context=IForestTable, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
class ForestTableSearchView(SearchView):
    """Forest table search view"""

    search_form_factory = ForestTableSearchForm


@adapter_config(context=(IForestTable, IAdminLayer, Interface),
                provides=IPageHeader)
class ForestTableHeaderAdapter(ReferenceTableHeaderAdapter):
    """Forest table header adapter"""


@view_config(name='forest-search-results.html',
             context=IForestTable, request_type=IPyAMSLayer,
             permission=VIEW_SYSTEM_PERMISSION)
class ForestTableSearchResultsView(AdminView, SearchResultsView):
    """Forest table search results view"""

    title = _("Forest search results")
    search_form_factory = ForestTableSearchForm

    sortOn = None
    startBatchingAt = 1000

    def __init__(self, context, request):
        super().__init__(context, request)
        request.registry.notify(PageletCreatedEvent(self))

    def get_element_id(self, element):
        return '{}::{}'.format(self.id, element[0].id_nat_frt)

    @reify
    def data_attributes(self):
        attributes = super().data_attributes
        attributes.setdefault('table', {}).update({
            'data-ams-location': absolute_url(self.context, self.request),
            'data-ams-datatable-sorting': '1,asc',
            'data-ams-attribute-switcher': 'switch-forest-visibility.json'
        })
        attributes.setdefault('tr', {}).update({
            'data-ams-element-name': lambda x, col: x[0].id_nat_frt
        })
        return attributes

    @reify
    def search_form(self):
        return self.search_form_factory(self.context, self.request)


@adapter_config(context=(IForestTable, IPyAMSLayer, ForestTableSearchResultsView),
                provides=IValues)
class SearchResultsViewValuesAdapter(ContextRequestViewAdapter):
    """Search results view values adapter"""

    @property
    def values(self):
        return self.view.search_form.get_search_results() or ()


@adapter_config(name='visible',
                context=(IForestTable, IPyAMSLayer, ForestTableSearchResultsView),
                provides=IColumn)
class ForestVisibleColumn(VisibilitySwitcherColumn):
    """Forest visible column"""

    permission = MANAGE_SITE_ROOT_PERMISSION

    def get_icon(self, item):
        if isinstance(item, (list, tuple)):
            item = item[1]
        if item is None:
            return '<i class="{0}"></i>'.format(self.on_icon_class)
        return super().get_icon(item)

    def has_permission(self, item):
        return super().has_permission(self.table.context)

    def renderCell(self, item):
        permission = self.permission
        if permission and not self.request.has_permission(permission, context=self.table.context):
            return self.get_icon(item)
        return super(AttributeSwitcherColumn, self).renderCell(item[1])


@view_config(name='switch-forest-visibility.json',
             context=IForestTable, request_type=IPyAMSLayer,
             renderer='json', xhr=True,
             permission=MANAGE_SITE_ROOT_PERMISSION)
def switch_forest_visibility(request):
    """Switch forest visibility"""
    params = request.params
    id_nat_foret = params.get('object_name')
    if not id_nat_foret:
        raise HTTPNotFound()
    session = get_user_session(FORET_SESSION)
    foret, info = session.query(Foret, InformationForet) \
        .outerjoin(InformationForet, Foret.id_nat_frt == InformationForet.id_nat_frt) \
        .filter(Foret.id_nat_frt == id_nat_foret) \
        .first()
    if foret is None:
        raise HTTPNotFound()
    if info is None:
        info = InformationForet(id_nat_frt=id_nat_foret)
        info.visible = False
        session.add(info)
    else:
        info.visible = not info.visible
    return {
        'status': 'success',
        'visible': info.visible
    }


@adapter_config(name='name',
                context=(IForestTable, IPyAMSLayer, ForestTableSearchResultsView),
                provides=IColumn)
class ForestLabelColumn(NameColumn):
    """Forest label column"""

    _header = _("Usage label")
    attrName = 'libelle_usage'

    def getValue(self, obj):
        foret, info, code = obj
        return (info.libelle if info is not None else None) or foret.libelle_usage


@adapter_config(name='exception',
                context=(IForestTable, IPyAMSLayer, ForestTableSearchResultsView),
                provides=IColumn)
class ForestExceptionColumn(ImageColumn):
    """Forest exception column"""

    _header = _("Exception")
    icon_class = 'fa-leaf'
    icon_hint = _("Exception forest")

    weight = 20

    def get_icon_hint(self, item):
        if (item is None) or (item.statut_exception == StatutForetException.f):
            return ''
        if item.statut_exception == StatutForetException.c:
            return self.request.localizer.translate(_("Candidate exception forest"))
        return super().get_icon_hint(item)

    def renderCell(self, item):
        info = item[1]
        if (info is None) or (info.statut_exception == StatutForetException.f):
            return ''
        result = super().renderCell(info)
        if info.statut_exception == StatutForetException.t:
            result = result.replace(self.icon_class, '{} text-success'.format(self.icon_class))
        return result


@adapter_config(name='id_nat_frt',
                context=(IForestTable, IPyAMSLayer, ForestTableSearchResultsView),
                provides=IColumn)
class ForestIDColumn(NameColumn):
    """Forest ID column"""

    _header = _("National ID")
    attrName = 'id_nat_frt'

    weight = 30

    def getValue(self, obj):
        return super().getValue(obj[0])


@adapter_config(name='category',
                context=(IForestTable, IPyAMSLayer, ForestTableSearchResultsView),
                provides=IColumn)
class ForestCategoryColumn(NameColumn):
    """Forest owner category column"""

    _header = _("Owner category")
    attrName = 'libelle'

    weight = 40

    def getValue(self, obj):
        return super().getValue(obj[2])


@adapter_config(name='department',
                context=(IForestTable, IPyAMSLayer, ForestTableSearchResultsView),
                provides=IColumn)
class ForestDeptColumn(NameColumn):
    """Forest department column"""

    _header = _("Dept")
    attrName = 'departement'

    weight = 50

    def getValue(self, obj):
        return super().getValue(obj[0])


@adapter_config(context=(Interface, IAdminLayer, ForestTableSearchResultsView),
                provides=ITableElementEditor)
class ForestElementEditor(ContextRequestViewAdapter):
    """Forest info element editor"""

    view_name = 'forest-properties.html'

    @property
    def url(self):
        foret = self.context[0]
        return resource_url(self.view.context, self.request, self.view_name,
                            query={'id_nat_frt': foret.id_nat_frt})

    modal_target = True


@pagelet_config(name='forest-properties.html',
                context=IForestTable, layer=IPyAMSLayer,
                permission=MANAGE_SITE_ROOT_PERMISSION)
@ajax_config(name='forest-properties.json',
             context=IForestTable, layer=IPyAMSLayer)
class ForestPropertiesEditForm(AdminDialogEditForm):
    """Forest properties edit form"""

    prefix = 'forest_properties.'

    legend = _("Edit forest properties")

    fields = Fields(IForet).select('libelle_usage') + \
        Fields(IInformationForet)
    ajax_handler = 'forest-properties.json'

    edit_permission = MANAGE_SITE_ROOT_PERMISSION

    @property
    def id_foret(self):
        id_foret = self.request.params.get('id_nat_frt')
        if not id_foret:
            id_foret = self.request.params.get('{}widgets.id_nat_frt'.format(self.prefix))
        return id_foret

    def check_mode(self):
        if not self.request.has_permission(MANAGE_SITE_ROOT_PERMISSION, context=self.context):
            return DISPLAY_MODE
        return INPUT_MODE

    def getContent(self):
        id_foret = self.id_foret
        if id_foret is None:
            raise HTTPNotFound()
        session = get_user_session(FORET_SESSION)
        foret, info = session.query(Foret, InformationForet) \
            .outerjoin(InformationForet, Foret.id_nat_frt == InformationForet.id_nat_frt) \
            .filter(Foret.id_nat_frt == id_foret) \
            .first()
        if foret is None:
            raise HTTPNotFound()
        if info is None:
            info = InformationForet(id_nat_frt=foret.id_nat_frt,
                                    libelle=foret.libelle_usage)
            session.add(info)
        return info

    def updateWidgets(self, prefix=None):
        super().updateWidgets(prefix)
        id_foret = self.widgets.get('id_nat_frt')
        if id_foret is not None:
            id_foret.mode = HIDDEN_MODE
        libelle_usage = self.widgets.get('libelle_usage')
        if libelle_usage is not None:
            libelle_usage.mode = DISPLAY_MODE
