#
# Copyright (c) 2012 Thierry Florac <tflorac AT onf.fr>
# All Rights Reserved.
#

import re
from datetime import date, timedelta

from geoalchemy2 import Geometry
from sqlalchemy import Boolean, Column, Date, Enum, Integer, Numeric, Text, Unicode
from sqlalchemy.ext.declarative import AbstractConcreteBase, declared_attr
from sqlalchemy.orm import relation, relationship
from sqlalchemy.orm.session import object_session
from sqlalchemy.schema import ForeignKey, PrimaryKeyConstraint
from sqlalchemy.sql import and_, or_
from sqlalchemy.sql.expression import func
from zope.interface import implementer, provider
from zope.schema.interfaces import ITitledTokenizedTerm

from onf_website.reference.client.model import Client
from onf_website.reference.forest.model import PARENT_SCHEMA, PARENT_SESSION
from onf_website.reference.forest.model.code import get_libelle
from onf_website.reference.forest.model.interfaces.foret import \
    IAbstractBaseForetRegroupementModel, IForet, \
    IForetModel, IInformationForet, IRegroupement, IRegroupementModel, StatutForetException
from onf_website.reference.insee.model import Commune, Departement
from onf_website.reference.orga.model import Structure
from pyams_alchemy import Base
from pyams_alchemy.engine import get_user_session
from pyams_alchemy.mixin import DynamicSchemaMixin
from pyams_gis.interfaces import WGS84, WGS84WM
from pyams_utils.adapter import adapter_config
from pyams_utils.unicode import translate_string


# Table REGION_IFN

class RegionIFN(DynamicSchemaMixin, Base):
    """IFN region class"""

    __tablename__ = 'region_ifn'
    __schema__ = PARENT_SCHEMA

    code = Column(Unicode(3), primary_key=True)
    codifn = Column(Unicode(4))
    libelle = Column(Unicode(50))

    @property
    def title(self):
        return '%s - %s' % (self.code, self.libelle)


# Polymorphic base classes

REG_ID_NAT_FRT = re.compile('F([0-9]{5})[A-Z]')


@provider(IAbstractBaseForetRegroupementModel)
@implementer(ITitledTokenizedTerm)
class AbstractBaseForetRegroupement(AbstractConcreteBase, Base):
    """Base model class for forests and groups"""

    __entity__ = 'RGPT+FRT'

    id_nat_frt = Column(Unicode(7), primary_key=True)
    code_frt = Column(Unicode(25))
    date_saisie = Column(Date)
    date_debut_validite = Column(Date)
    date_fin_validite = Column(Date)
    est_domaniale = Column(Boolean)
    regime = Column(Unicode(2))
    cadre_legal_gestion = Column(Unicode(2))
    frais_garderie = Column(Unicode(3))
    libelle_preposition = Column(Unicode(3))
    libelle_strict = Column(Unicode(60))
    libelle_usage = Column(Unicode(120))
    libelle_recherche = Column(Unicode(120))
    surface_cadastrale = Column(Numeric(15, 4))
    code_service_gestionnaire = Column('service_gestionnaire', Unicode(6))
    code_structure_rh_gestionnaire = Column('structure_rh_gestionnaire', Unicode(6))
    code_structure_rh_sign = Column('structure_rh_sign', Unicode(6))
    centre_profit = Column(Unicode(10))
    commentaire = Column(Unicode(250))

    # declared attributes
    @declared_attr
    def departement(self):
        return Column(Unicode(5), ForeignKey(Departement.dep))

    @declared_attr
    def departement_insee(self):
        return relationship(Departement)

    @declared_attr
    def code_region_ifn(self):
        return Column('region_ifn', Unicode(3), ForeignKey(RegionIFN.code))

    @declared_attr
    def region_ifn(self):
        return relationship(RegionIFN)

    @property
    def value(self):
        return self.id_nat_frt

    @property
    def token(self):
        return self.id_nat_frt

    @property
    def title(self):
        return u'%s (%s)' % (self.libelle_usage, self.departement)

    @property
    def longname(self):
        return u'%s (%s)' % (self.libelle_usage, self.departement_insee.nccenr)

    @property
    def invalide(self):
        return self.date_fin_validite < date.today()

    @property
    def date_debut_invalidite(self):
        return self.date_fin_validite + timedelta(days=1)

    @property
    def libelle_cadre_gestion(self):
        return get_libelle(u'CADRE_LEGAL_GESTION', self.cadre_legal_gestion) if self.cadre_legal_gestion else u''

    @property
    def libelle_classement(self):
        return get_libelle(u'CLASSEMENT_AGRICOLE', self.frais_garderie) if self.frais_garderie else u''

    @property
    def preposition(self):
        return get_libelle(u'LIBELLE_PREPOSITION', self.libelle_preposition) if self.libelle_preposition else u'--'

    @property
    def service_gestionnaire(self):
        return Structure.get(self.code_service_gestionnaire).first()

    @property
    def libelle_service_gestionnaire(self):
        return self.service_gestionnaire.title if self.code_service_gestionnaire else u'--'

    @property
    def structure_rh_gestionnaire(self):
        return Structure.get(self.code_structure_rh_gestionnaire).first()

    @property
    def libelle_structure_gestionnaire(self):
        return self.structure_rh_gestionnaire.title if self.code_structure_rh_gestionnaire else u'--'

    @property
    def inactive(self):
        return (self.cadre_legal_gestion == 'NG') or (self.date_fin_validite < date.today())

    @classmethod
    def find(cls, query, session=PARENT_SESSION, ignore_disabled=False):
        return Regroupement.find(query, session, ignore_disabled), \
            Foret.find(query, session, ignore_disabled)

    @classmethod
    def find_by_id(cls, id_nat_frt, session=PARENT_SESSION, ignore_disabled=False):
        return Regroupement.find_by_id(id_nat_frt, session, ignore_disabled), \
            Foret.find_by_id(id_nat_frt, session, ignore_disabled)

    @classmethod
    def find_by_label(cls, query, session=PARENT_SESSION, ignore_disabled=False):
        return Regroupement.find_by_label(query, session, ignore_disabled), \
            Foret.find_by_label(query, session, ignore_disabled)


# Table RDF_REGROUPEMENT

@provider(IRegroupementModel)
@implementer(IRegroupement)
class Regroupement(DynamicSchemaMixin, AbstractBaseForetRegroupement):
    """Model class for RDF_REGROUPEMENT table"""

    __tablename__ = 'rdf_regroupement'
    __schema__ = PARENT_SCHEMA
    __mapper_args__ = {
        'polymorphic_identity': 'regroupement',
        'concrete': True
    }

    __entity__ = 'RGPT'

    id_nat_frt = Column(Unicode(7), primary_key=True)

    @classmethod
    def get(cls, code, session=PARENT_SESSION):
        if isinstance(code, str):
            code = code.split(',')
        session = get_user_session(session)
        return session.query(Regroupement).filter(Regroupement.id_nat_frt.in_(code))

    @classmethod
    def find(cls, query, session=PARENT_SESSION, ignore_disabled=False):
        if isinstance(query, dict):
            id_nat_frt = query.get('id_nat_frt')
            label = query.get('label')
        else:
            query = query.strip()
            if REG_ID_NAT_FRT.match(query.upper()):
                id_nat_frt = query.upper()
                label = None
            else:
                id_nat_frt = None
                label = query
        session = get_user_session(session)
        if id_nat_frt:
            return Regroupement.find_by_id(id_nat_frt, session, ignore_disabled)
        elif label:
            return Regroupement.find_by_label(label, session, ignore_disabled)

    @classmethod
    def find_by_id(cls, id_nat_frt, session=PARENT_SESSION, ignore_disabled=False):
        if isinstance(id_nat_frt, (list, tuple, set)):
            params = Regroupement.id_nat_frt.in_(id_nat_frt)
        else:
            params = Regroupement.id_nat_frt == id_nat_frt
        session = get_user_session(session)
        return session.query(Regroupement).filter(params)

    @classmethod
    def find_by_label(cls, query, session=PARENT_SESSION, ignore_disabled=False):
        query = '%' + translate_string(query, keep_chars="_-.'").upper() + '%'
        query = query.replace('-', ' ').replace('_', '|_')
        params = [Regroupement.libelle_recherche.like(query, escape='|')]
        if ignore_disabled:
            today = date.today()
            params.append(Regroupement.date_debut_validite <= today)
            params.append(Regroupement.date_fin_validite >= today)
        session = get_user_session(session)
        return session.query(Regroupement) \
            .filter(and_(*params))

    @property
    def communes(self):
        session = object_session(self)
        return session.query(ForetSurCommune.code_insee,
                             ForetSurCommune.libelle_insee,
                             func.sum(ForetSurCommune.surface_cadastrale)) \
            .join(Foret, ForetSurCommune.id_nat_frt == Foret.id_nat_frt) \
            .join(Regroupement, Foret.id_nat_frt_regroupement == Regroupement.id_nat_frt) \
            .filter(Regroupement.id_nat_frt == self.id_nat_frt) \
            .group_by(ForetSurCommune.code_insee,
                      ForetSurCommune.libelle_insee) \
            .order_by(ForetSurCommune.libelle_insee)


# Table RDF_FORET

@provider(IForetModel)
@implementer(IForet)
class Foret(DynamicSchemaMixin, AbstractBaseForetRegroupement):
    """Model class for RDF_FORET table"""

    __tablename__ = 'rdf_foret'
    __schema__ = PARENT_SCHEMA
    __mapper_args__ = {
        'polymorphic_identity': 'foret',
        'concrete': True
    }

    __entity__ = 'FRT'

    id_nat_frt = Column(Unicode(7), primary_key=True)
    date_expiration_contrat = Column(Date)
    surface_retenue_gestion = Column(Numeric(15, 4))
    surface_sylviculture = Column(Numeric(15, 4))
    code_tgpe = Column(Unicode(13))
    surface_tgpe = Column(Numeric(15, 4))
    pefc_num_cert = Column(Unicode(15))
    pefc_debut_cert = Column(Date)
    pefc_fin_cert = Column(Date)
    marquage_pefc = Column(Boolean)
    pefc_num_marq = Column(Unicode(15))
    pefc_debut_marq = Column(Date)
    pefc_fin_marq = Column(Date)
    date_fin_gestion = Column(Date)
    id_nat_frt_regroupement = Column(Unicode(7), ForeignKey(Regroupement.id_nat_frt))
    date_liaison_groupement = Column(Date)

    regroupement = relation(Regroupement, backref='forets')

    @classmethod
    def get(cls, code, session=PARENT_SESSION):
        if isinstance(code, str):
            code = code.split(',')
        session = get_user_session(session)
        return session.query(Foret).filter(Foret.id_nat_frt.in_(code))

    @classmethod
    def find(cls, query, session=PARENT_SESSION, ignore_disabled=False, with_info=False):
        if isinstance(query, dict):
            id_nat_frt = query.get('id_nat_frt')
            label = query.get('label')
        else:
            query = query.strip()
            if REG_ID_NAT_FRT.match(query.upper()):
                id_nat_frt = query.upper()
                label = None
            else:
                id_nat_frt = None
                label = query
        session = get_user_session(session)
        if id_nat_frt:
            return Foret.find_by_id(id_nat_frt, session, ignore_disabled, with_info)
        elif label:
            return Foret.find_by_label(label, session, ignore_disabled, with_info)

    @classmethod
    def find_by_id(cls, id_nat_frt, session=PARENT_SESSION, ignore_disabled=False, with_info=False):
        if isinstance(id_nat_frt, (list, tuple, set)):
            params = Foret.id_nat_frt.in_(id_nat_frt)
        else:
            params = Foret.id_nat_frt == id_nat_frt
        session = get_user_session(session)
        if with_info:
            return session.query(Foret, InformationForet) \
                .outerjoin(Foret, Foret.id_nat_frt == InformationForet.id_nat_frt) \
                .filter(params)

        return session.query(Foret) \
            .filter(params)

    @classmethod
    def find_by_label(cls, query, session=PARENT_SESSION, ignore_disabled=False, with_info=False):
        query = translate_string(query, keep_chars="_-.'").upper()
        query = query.replace('-', ' ').replace('_', '|_')
        params = [or_(Foret.libelle_recherche.contains(query, escape='|'),
                      InformationForet.libelle.contains(query, escape='|'))]
        if ignore_disabled:
            today = date.today()
            params.append(Foret.date_debut_validite <= today)
            params.append(Foret.date_fin_validite >= today)
        session = get_user_session(session)
        if with_info:
            return session.query(Foret, InformationForet) \
                .outerjoin(Foret, Foret.id_nat_frt == InformationForet.id_nat_frt) \
                .filter(and_(*params))
        return session.query(Foret) \
            .outerjoin(InformationForet, Foret.id_nat_frt == InformationForet.id_nat_frt) \
            .filter(and_(*params))

    @classmethod
    def find_by_location(cls, query, session=PARENT_SESSION, ignore_disabled=False):
        longitude = query.get('longitude', None)
        latitude = query.get('latitude', None)
        if not (longitude and latitude):
            return ()
        session = get_user_session(session)
        distance = query.get('distance', 50.0) * 1000
        forest_types = query.get('forestTypes', '1').split(',')
        marker = session.scalar(
            func.ST_Transform(
                func.ST_GeomFromText('POINT({} {})'.format(longitude, latitude), WGS84), WGS84WM))
        params = [
            func.ST_Distance(ForetGeom.geom_3857, marker) < distance,
            ProprietaireForet.categorie.in_(forest_types)
        ]
        return session.query(Foret, func.ST_Distance(ForetGeom.geom_3857, marker)) \
            .join(ForetGeom, Foret.id_nat_frt == ForetGeom.id_nat_frt) \
            .join(ProprietaireForet, Foret.id_nat_frt == ProprietaireForet.id_nat_frt) \
            .filter(and_(*params)) \
            .order_by(func.ST_Distance(ForetGeom.geom_3857, marker))

    @classmethod
    def find_with_infos(cls, ids, session=PARENT_SESSION, ignore_disabled=True):
        session = get_user_session(session)
        result = {}
        for forest_id in ids:
            params = [Foret.id_nat_frt == forest_id]
            if ignore_disabled:
                today = date.today()
                params.append(Foret.date_debut_validite <= today)
                params.append(Foret.date_fin_validite >= today)
            for forest, prop, info in session.query(Foret, ProprietaireForet, InformationForet) \
                    .join(ProprietaireForet, Foret.id_nat_frt == ProprietaireForet.id_nat_frt) \
                    .outerjoin(InformationForet, Foret.id_nat_frt == InformationForet.id_nat_frt) \
                    .filter(and_(*params)) \
                    .order_by(ProprietaireForet.pourcentage_part, ProprietaireForet.categorie):
                result[forest_id] = (forest, prop, info)
        return result

    @property
    def libelle_chorus(self):
        return self.code_tgpe or u'--'


class ForetGeom(DynamicSchemaMixin, Base):
    """Model class for RDF_FORET_GEOM table with geometry"""

    __tablename__ = 'rdf_foret_geom'
    __schema__ = PARENT_SCHEMA

    gid = Column(Integer, primary_key=True)
    id_nat_frt = Column(Unicode(7), ForeignKey(Foret.id_nat_frt))
    geom = Column(Geometry('MULTIPOLYGON', WGS84))
    geom_3857 = Column(Geometry('MULTIPOLYGON', WGS84WM))


Foret.ref_geom = relation(ForetGeom,
                          primaryjoin=Foret.id_nat_frt == ForetGeom.id_nat_frt,
                          uselist=False)


# Table RDF_COMMUNE

class ForetSurCommune(DynamicSchemaMixin, Base):
    """Model class for RDF_COMMUNE table"""

    __tablename__ = 'rdf_commune'
    __schema__ = PARENT_SCHEMA

    @declared_attr
    def __table_args__(cls):
        return (
            PrimaryKeyConstraint('code_insee', 'id_nat_frt'),
            cls.get_schema()
        )

    code_insee = Column(Unicode(5), ForeignKey(Commune.code))
    id_nat_frt = Column(Unicode(7), ForeignKey(Foret.id_nat_frt))
    libelle_insee = Column(Unicode(70))
    surface_cadastrale = Column(Numeric(15, 4))

    commune = relation(Commune, backref='forets')
    foret = relation(Foret, backref='communes')


# Table RDF_PROPRIETAIRE

class ProprietaireForet(DynamicSchemaMixin, Base):
    """Model class for RDF_PROPRIETAIRE table"""

    __tablename__ = 'rdf_proprietaire'
    __schema__ = PARENT_SCHEMA

    @declared_attr
    def __table_args__(cls):
        return (
            PrimaryKeyConstraint('clecli', 'id_nat_frt'),
            cls.get_schema()
        )

    clecli = Column(Unicode(30), ForeignKey(Client.clecli))
    id_nat_frt = Column(Unicode(7), ForeignKey(Foret.id_nat_frt))
    categorie = Column(Unicode(2))
    pourcentage_part = Column(Numeric(5, 2))
    type_copropriete = Column(Unicode(3))
    type_lot = Column(Unicode(2))
    commentaire = Column('commentaire_rdf', Unicode(250))
    numerateur_part = Column(Integer)
    denominateur_part = Column(Integer)

    client = relation(Client)
    foret = relation(Foret, backref='proprietaires')

    @property
    def libelle_type(self):
        return get_libelle(u'TYPE_COPROPRIETE', self.type_copropriete) if self.type_copropriete else u''

    @property
    def libelle_categorie(self):
        return get_libelle(u'CATEGORIE_PROPRIETAIRE', self.categorie) if self.categorie else u''


# Table RDF_REPRESENTANT

class RepresentantForet(DynamicSchemaMixin, Base):
    """Model class for RDF_REPRESENTANT table"""

    __tablename__ = 'rdf_representant'
    __schema__ = PARENT_SCHEMA

    @declared_attr
    def __table_args__(cls):
        return (
            PrimaryKeyConstraint('clecli', 'id_nat_frt'),
            cls.get_schema()
        )

    clecli = Column(Unicode(30), ForeignKey(Client.clecli))
    id_nat_frt = Column(Unicode(7), ForeignKey(Foret.id_nat_frt))
    categorie = Column(Unicode(2))
    commentaire = Column('commentaire_rdf', Unicode(250))

    client = relation(Client)
    foret = relation(Foret, backref='representants')

    @property
    def libelle_categorie(self):
        return get_libelle(u'REPRESENTANT_PROPRIETAIRE', self.categorie) if self.categorie else u''


RepresentantForet.regroupement = relation(Regroupement,
                                          foreign_keys=RepresentantForet.id_nat_frt,
                                          primaryjoin=RepresentantForet.id_nat_frt == Regroupement.id_nat_frt,
                                          backref='representants')


# Table RDF_FORET_CUSTOM_INFO

@implementer(IInformationForet)
class InformationForet(DynamicSchemaMixin, Base):
    """Model class for RDF_FORET_CUSTOM_INFO"""

    __tablename__ = 'rdf_foret_custom_info'
    __schema__ = PARENT_SCHEMA

    id_nat_frt = Column(Unicode(7), ForeignKey(Foret.id_nat_frt), primary_key=True)
    libelle = Column(Unicode(255))
    visible = Column(Boolean(), nullable=False, default=True)
    header = Column(Text())
    statut_exception = Column(Enum(StatutForetException), nullable=False,
                              default=StatutForetException.f)
    acces_foretmapper = Column(Boolean(), nullable=False, default=False)

    foret = relation(Foret, backref='informations')

    @classmethod
    def find_foretmapper_forests(cls, session=PARENT_SESSION):
        session = get_user_session(session)
        today = date.today()
        yield from session.query(Foret) \
            .join(InformationForet, Foret.id_nat_frt == InformationForet.id_nat_frt) \
            .filter(and_(Foret.date_debut_validite <= today,
                         Foret.date_fin_validite >= today,
                         InformationForet.acces_foretmapper == True))


@adapter_config(required=IInformationForet,
                provides=IForet)
def information_foret_adapter(context):
    """Forest adapter for forest information"""
    if context.foret is not None:
        return context.foret
    return get_user_session(PARENT_SESSION).query(Foret) \
        .filter(Foret.id_nat_frt == context.id_nat_frt) \
        .first()
