#
# Copyright (c) 2015-2023 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

__docformat__ = 'restructuredtext'

from cornice import Service
from cornice.validators import colander_validator
from pyramid.httpexceptions import HTTPForbidden, HTTPOk, HTTPServiceUnavailable, HTTPUnauthorized
from pyramid.security import Authenticated

from onf_website.reference.forest import Foret, InformationForet
from onf_website.reference.forest.api.interfaces import REST_FORESTS_ROUTE, REST_FORESTS_ROUTE_DEFAULT, \
    REST_FORESTS_ROUTE_SETTING, REST_FORETMAPPER_ROUTE
from onf_website.reference.forest.api.schema import ForestMapperConfigRequest, ForestsGetterRequest, \
    ForestsGetterResponse, ForetMapperConfigResponse
from onf_website.shared.activity import IActivityManager
from onf_website.shared.common.interfaces import REST_CONTENT_PUBLIC_SEARCH_ROUTE_DEFAULT, \
    REST_CONTENT_PUBLIC_SEARCH_ROUTE_SETTING
from onf_website.shared.forest import IForestManager
from pyams_content.component.illustration import ILinkIllustration
from pyams_content.features.json import IJSONConverter, to_json
from pyams_content.interfaces import USE_PUBLIC_API_PERMISSION
from pyams_content.shared.alert import ALERT_CONTENT_TYPE, IAlertTypesManager, IAlertsManager
from pyams_content.shared.common.interfaces.types import ITypedDataManager
from pyams_security.rest import check_cors_origin, set_cors_headers
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION
from pyams_utils.registry import query_utility
from pyams_utils.rest import STATUS, http_error, rest_responses

#
# API ForetMapper
#

foretmapper_config_service = Service(name=REST_FORETMAPPER_ROUTE,
                                     pyramid_route=REST_FORETMAPPER_ROUTE,
                                     description='ONF website API for ForetMapper mobile application')


@foretmapper_config_service.options(validators=(check_cors_origin, set_cors_headers))
def foretmapper_config_options(request):
    return ''


foretmapper_config_responses = rest_responses.copy()
foretmapper_config_responses[HTTPOk.code] = ForetMapperConfigResponse(
    description="ForetMapper service settings getter")


@foretmapper_config_service.get(schema=ForestMapperConfigRequest(),
                                validators=(check_cors_origin, colander_validator, set_cors_headers),
                                response_schemas=foretmapper_config_responses)
def foretmapper_config_getter(request):
    """ForetMapper application settings"""
    if Authenticated not in request.effective_principals:
        return http_error(request, HTTPUnauthorized)
    forests = query_utility(IForestManager)
    if forests is None:
        return http_error(request, HTTPServiceUnavailable)
    if not request.has_permission(USE_PUBLIC_API_PERMISSION, context=forests):
        return http_error(request, HTTPForbidden)
    settings = request.registry.settings
    default_route = settings.get(REST_CONTENT_PUBLIC_SEARCH_ROUTE_SETTING,
                                 REST_CONTENT_PUBLIC_SEARCH_ROUTE_DEFAULT)
    result = {
        'status': STATUS.SUCCESS.value,
        'forests_route': settings.get(REST_FORESTS_ROUTE_SETTING, REST_FORESTS_ROUTE_DEFAULT),
        'alerts_route': default_route.format(content_type=ALERT_CONTENT_TYPE),
        'active_forests': []
    }
    params = request.validated.get('querystring')
    # list of active forests in mobile application
    for forest in InformationForet.find_foretmapper_forests():
        output = {
            'id_nat_frt': forest.id_nat_frt,
            'libelle_usage': forest.libelle_usage,
            'libelle_strict': forest.libelle_strict
        }
        illustration = ILinkIllustration(forest, None)
        if illustration is not None:
            converter = request.registry.queryMultiAdapter((illustration, request), IJSONConverter)
            if converter is not None:
                output['illustration'] = converter.to_json(params)
        result['active_forests'].append(output)
    # list of alerts types
    alerts_manager = query_utility(IAlertsManager)
    if alerts_manager is not None:
        alerts_types = IAlertTypesManager(alerts_manager, None)
        if alerts_types is not None:
            result['alert_types'] = [
                to_json(alert_type, request)
                for alert_type in alerts_types.values()
            ]
    activity_manager = query_utility(IActivityManager)
    if activity_manager is not None:
        # list of activity types
        activity_types = ITypedDataManager(activity_manager, None)
        if activity_types is not None:
            result['activity_types'] = [
                to_json(data_type, request)
                for data_type in activity_types.values()
            ]
        # list of activity adapted themes
        adaptations_theme = activity_manager.foretmapper_adaptations_themes
        if adaptations_theme is not None:
            result['activity_adaptations'] = [
                to_json(term, request)
                for term in adaptations_theme.specifics
            ]
    return result


#
# Forests references table services
#

forests_service = Service(name=REST_FORESTS_ROUTE,
                          pyramid_route=REST_FORESTS_ROUTE,
                          description='ONF website forests API')


@forests_service.options(validators=(check_cors_origin, set_cors_headers))
def forests_service_options(request):
    return ''


forests_getter_responses = rest_responses.copy()
forests_getter_responses[HTTPOk.code] = ForestsGetterResponse(description="Forests search results")


@forests_service.get(schema=ForestsGetterRequest(),
                     validators=(check_cors_origin, colander_validator, set_cors_headers),
                     response_schemas=forests_getter_responses)
def forests_service_getter(request):
    """Forests search service"""
    if Authenticated not in request.effective_principals:
        return http_error(request, HTTPUnauthorized)
    forests = query_utility(IForestManager)
    if forests is None:
        return http_error(request, HTTPServiceUnavailable)
    if not request.has_permission(VIEW_SYSTEM_PERMISSION, context=forests):
        return http_error(request, HTTPForbidden)
    params = request.validated.get('querystring', {})
    return {
        'status': STATUS.SUCCESS.value,
        'results': [{
            'id_nat_frt': forest.id_nat_frt,
            'libelle_usage': forest.libelle_usage,
            'libelle_strict': forest.libelle_strict,
            'distance': distance / 1000
        } for forest, distance in Foret.find_by_location(params, ignore_disabled=True)]
    }
