#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'


# import standard library

# import interfaces
from onf_website.reference.client.model.interfaces import IClient
from zope.schema.interfaces import ITitledTokenizedTerm

# import packages
from onf_website.reference.insee.model import Commune, CodePostal
from pyams_alchemy import Base
from pyams_alchemy.engine import get_user_session
from pyams_alchemy.mixin import DynamicSchemaMixin
from sqlalchemy import Column, Unicode, ForeignKey
from sqlalchemy.orm import relation, joinedload
from sqlalchemy.sql import func, or_
from zope.interface import implementer


PARENT_SCHEMA = 'commun'
PARENT_SESSION = 'COMMUN'


@implementer(IClient, ITitledTokenizedTerm)
class Client(DynamicSchemaMixin, Base):
    """Content class for 'CLIENT_NAT' records"""

    __tablename__ = 'client_nat'
    __schema__ = PARENT_SCHEMA

    clecli = Column('clecli', Unicode, primary_key=True)
    motdir = Column('motdir', Unicode)
    raisoc = Column('raisoc', Unicode)
    nomcli = Column('nomcli', Unicode)
    titcli = Column('titcli', Unicode)
    numtel = Column('numtel', Unicode)
    numfax = Column('numfax', Unicode)
    codpay = Column('codpay', Unicode)
    codedt = Column('codedt', Unicode)
    adr1dom = Column('adr1dom', Unicode)
    adr2dom = Column('adr2dom', Unicode)
    cposdom = Column('cposdom', Unicode)
    lvildom = Column('lvildom', Unicode)
    actina = Column('actina', Unicode)
    nclisap = Column('nclisap', Unicode)
    insee_commune = Column('commune_insee', Unicode, ForeignKey(Commune.code))

    commune = relation(Commune)

    @property
    def value(self):
        return self.clecli

    @property
    def token(self):
        return str(self.clecli)

    @property
    def shortname(self):
        return self.nomcli

    @property
    def title(self):
        if self.insee_commune:
            return '%s (%s)' % (self.nomcli, self.commune.dep)
        else:
            return self.nomcli

    @property
    def longname(self):
        return '%s - %s' % (self.titcli, self.title)

    @classmethod
    def get(cls, key, session=PARENT_SESSION):
        return Client.find_by_key(key, session)

    @classmethod
    def find(cls, query, session=PARENT_SESSION):
        params = []
        if isinstance(query, dict):
            code_client = query.get('code_client')
            if code_client:
                params.append(Client.clecli == code_client)
            nom = query.get('nom')
            if nom:
                params.append(func.upper(Client.nomcli).like(u'%' + nom.upper() + u'%'))
            code_insee = query.get('code_insee')
            if code_insee:
                params.append(Client.insee_commune == code_insee)
            code_postal = query.get('code_postal')
            if code_postal:
                params.append(CodePostal.code_postal == code_postal)
        else:
            try:
                _key = int(query)
                params.append(Client.clecli == query)
                params.append(Client.insee_commune == query)
                params.append(CodePostal.code_postal == query)
            except:
                params.append(func.upper(Client.nomcli).like(u'%' + query.upper() + u'%'))
        session = get_user_session(session)
        return session.query(Client) \
                      .options(joinedload(Client.commune)) \
                      .join(Commune, Client.insee_commune == Commune.code) \
                      .join(CodePostal, Commune.code == CodePostal.code_insee) \
                      .filter(or_(*params))

    @classmethod
    def find_by_key(cls, key, session=PARENT_SESSION):
        if isinstance(key, (list, tuple, set)):
            params = Client.clecli.in_(key)
        else:
            params = Client.clecli == key
        session = get_user_session(session)
        return session.query(Client).filter(params)
