#
# Copyright (c) 2015-2023 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

import json

from pyramid.decorator import reify
from sqlalchemy.sql import func
from zope.interface import Interface, implementer
from zope.intid.interfaces import IIntIds
from zope.location.interfaces import ISublocations
from zope.schema.fieldproperty import FieldProperty
from zope.traversing.interfaces import ITraversable

from onf_website.component.location.skin.map import BaseMapMarkersRendererMixin
from onf_website.features.search.skin import AggregatedSearchResultsRenderer
from onf_website.features.search.skin.interfaces import IMapWithHeadFiltersSearchResultsPortletRendererSettings
from onf_website.features.search.skin.interfaces.filter import IFilterAggregate, IFilterProcessor
from onf_website.features.search.skin.interfaces.nearme import INearmeFilterContainer, INearmeFilterContainerTarget, \
    INearmeMapSearchResultsPortletRendererSettings, INearmeSearchResultsPortletRendererSettings, \
    NEARME_FILTER_CONTAINER_ANNOTATION_KEY
from onf_website.reference.forest import InformationForet
from onf_website.reference.forest.model import PARENT_SESSION as RDF_SESSION
from onf_website.reference.forest.model.foret import Foret, ForetGeom, REG_ID_NAT_FRT
from onf_website.reference.insee.model import Commune, CommuneGeom, Departement, DepartementGeom, \
    PARENT_SESSION as INSEE_SESSION, REG_COMMUNE_CODE, REG_DEPT_CODE
from onf_website.reference.universe import IUniverse
from onf_website.reference.universe.interfaces import IUniverseTable
from onf_website.skin.public import aside_sticky
from onf_website.skin.public.layer import IONFBaseLayer
from pyams_alchemy.engine import get_user_session
from pyams_content.features.search.portlet.interfaces import IAggregatedPortletRenderer, ISearchResultsPortletSettings
from pyams_content.shared.common import IWfSharedContent
from pyams_default_theme.features.search.portlet import SearchResultsPortletRendererBaseSettings
from pyams_default_theme.features.search.portlet.interfaces import ISearchResultRenderer, ISearchResultTarget, \
    ISearchResultTitle
from pyams_default_theme.interfaces import ISearchResultsView
from pyams_file.property import FileProperty
from pyams_gis.configuration import MapConfiguration
from pyams_gis.interfaces import WGS84
from pyams_gis.interfaces.configuration import IMapConfiguration
from pyams_gis.interfaces.utility import IMapManager
from pyams_i18n.interfaces import II18n
from pyams_portal.interfaces import IPortalContext, IPortletRenderer
from pyams_skin.layer import IPyAMSLayer, IPyAMSUserLayer
from pyams_template.template import template_config
from pyams_utils.adapter import ContextAdapter, adapter_config, get_annotation_adapter
from pyams_utils.container import BTreeOrderedContainer
from pyams_utils.factory import factory_config
from pyams_utils.registry import get_utility, query_utility
from pyams_utils.request import check_request
from pyams_utils.url import canonical_url
from pyams_viewlet.viewlet import ViewContentProvider, contentprovider_config

__docformat__ = 'restructuredtext'

from onf_website import _


#
# Near-me filters container
#

@factory_config(INearmeFilterContainer)
class NearmeFilterContainer(BTreeOrderedContainer):
    """Near-me filter container"""

    def add(self, obj):
        intids = get_utility(IIntIds)
        self[hex(intids.register(obj))[2:]] = obj

    def get_visible_filters(self):
        yield from filter(lambda x: x.visible, self.values())


@adapter_config(required=INearmeFilterContainerTarget,
                provides=INearmeFilterContainer)
def nearme_filter_container(context):
    """Near-me filter container adapter"""
    return get_annotation_adapter(context, NEARME_FILTER_CONTAINER_ANNOTATION_KEY,
                                  INearmeFilterContainer, name='++filter.nearme++')


@adapter_config(name='filter.nearme',
                required=INearmeFilterContainerTarget,
                provides=ITraversable)
class NearmeFilterContainerTraverser(ContextAdapter):
    """Near-me filter container traverser"""

    def traverse(self, name, furtherPath=None):
        container = INearmeFilterContainer(self.context, None)
        if name:
            return container.get(name)
        return container


@adapter_config(name='filter.nearme',
                required=INearmeFilterContainerTarget,
                provides=ISublocations)
class NearmeFilterContainerSublocations(ContextAdapter):
    """Near-me filter container sublocations"""

    def sublocations(self):
        return INearmeFilterContainer(self.context).values()


@factory_config(INearmeSearchResultsPortletRendererSettings)
@implementer(INearmeFilterContainerTarget, IAggregatedPortletRenderer)
class NearmeSearchResultsPortletRendererSettings(SearchResultsPortletRendererBaseSettings):
    """Search results portlet advanced filters renderer settings"""

    description = FieldProperty(INearmeSearchResultsPortletRendererSettings['description'])
    dropdown_menu_title = FieldProperty(INearmeSearchResultsPortletRendererSettings['dropdown_menu_title'])
    map_mode_label = FieldProperty(INearmeSearchResultsPortletRendererSettings['map_mode_label'])
    list_mode_label = FieldProperty(INearmeSearchResultsPortletRendererSettings['list_mode_label'])
    empty_results_title = FieldProperty(INearmeSearchResultsPortletRendererSettings['empty_results_title'])
    empty_results_message = FieldProperty(INearmeSearchResultsPortletRendererSettings['empty_results_message'])
    help_title = FieldProperty(INearmeSearchResultsPortletRendererSettings['help_title'])
    help_message = FieldProperty(INearmeSearchResultsPortletRendererSettings['help_message'])
    open_links_in_new_tabs = FieldProperty(INearmeSearchResultsPortletRendererSettings['open_links_in_new_tabs'])

    @property
    def filters(self):
        yield from INearmeFilterContainer(self, {}).get_visible_filters()

    def get_labels_for_tag_ids(self, tag_data):
        intids = get_utility(IIntIds)
        result = {}
        for item in tag_data:
            try:
                tag_id = int(item['key'])
                tag_object = intids.queryObject(tag_id)
                if tag_object is not None:
                    result[tag_id] = tag_object.label
            except (KeyError, ValueError, TypeError):
                continue
        return result

    @property
    def universe_table_config(self):
        table = query_utility(IUniverseTable)
        return {
            "dropdown_menu_title": II18n(table).query_attribute('dropdown_menu_title'),
            "inactive_info_title": II18n(table).query_attribute('inactive_info_title'),
            "inactive_info_text": II18n(table).query_attribute('inactive_info_text'),
            "online_help_title": II18n(table).query_attribute('online_help_title'),
            "online_help_text": II18n(table).query_attribute('online_help_text'),
        }

    @property
    def universes_list(self):
        table = query_utility(IUniverseTable)
        if table is not None:
            request = check_request()
            terms = [
                {
                    'title': II18n(v).query_attribute('title', request=request),
                    'url': canonical_url(v, request, view_name=request.view_name)
                }
                for v in table.get_visible_items()
            ]
        else:
            terms = []
        return terms

    def processed_filters(self, request, aggregations):
        processed = []
        for filter in self.filters:
            processor = request.registry.queryAdapter(filter, IFilterProcessor)
            if processor is not None:
                processed_data = processor.process(request, aggregations)
                processed.append(processed_data)
        return processed

    @property
    def aggregates(self):
        return [
            IFilterAggregate(filter)
            for filter in self.filters
        ]


#
# Nearme search results portlet map renderer
#


class NearmeSearchResultsPortletBaseRenderer(AggregatedSearchResultsRenderer):
    """Nearme search results portlet base renderer"""

    @reify
    def locations(self):
        result = {}
        user_locations = self.request.params.getall('user_location')
        if user_locations:
            for user_location in user_locations:
                user_location = user_location.strip().upper()
                if REG_ID_NAT_FRT.match(user_location):
                    try:
                        session = get_user_session(RDF_SESSION)
                        foret, centroid, info_foret, dept = session.query(Foret,
                                                                          func.ST_Centroid(ForetGeom.geom),
                                                                          InformationForet,
                                                                          Departement) \
                                .join(ForetGeom, Foret.id_nat_frt==ForetGeom.id_nat_frt) \
                                .outerjoin(InformationForet, Foret.id_nat_frt==InformationForet.id_nat_frt) \
                                .join(Departement, Foret.departement==Departement.dep) \
                                .filter(Foret.id_nat_frt==user_location) \
                                .first()
                    except TypeError:  # null result
                        pass
                    else:
                        if foret is not None:
                            longitude = session.scalar(func.ST_X(centroid))
                            latitude = session.scalar(func.ST_Y(centroid))
                            result[user_location] = {
                                'weight': 30,
                                'type': 'forest',
                                'info': (foret, info_foret),
                                'label': getattr(info_foret, 'libelle', None) or foret.title,
                                'geom': (longitude, latitude),
                                'dept': foret.departement,
                                'dept_label': dept.title
                            }
                elif REG_COMMUNE_CODE.match(user_location):
                    try:
                        session = get_user_session(INSEE_SESSION)
                        commune, centroid, dept = session.query(Commune,
                                                                func.ST_Centroid(CommuneGeom.geom),
                                                                Departement) \
                                .join(CommuneGeom, Commune.code==CommuneGeom.code) \
                                .join(Departement, Commune.dep==Departement.dep) \
                                .filter(Commune.code==user_location) \
                                .first()
                    except TypeError:
                        pass
                    else:
                        if commune is not None:
                            longitude = session.scalar(func.ST_X(centroid))
                            latitude = session.scalar(func.ST_Y(centroid))
                            result[user_location] = {
                                'weight': 20,
                                'type': 'commune',
                                'info': commune,
                                'label': commune.title,
                                'geom': (longitude, latitude),
                                'dept': commune.dep,
                                'dept_label': dept.title
                            }
                elif REG_DEPT_CODE.match(user_location):
                    try:
                        session = get_user_session(INSEE_SESSION)
                        dept, centroid = session.query(Departement, func.ST_Centroid(DepartementGeom.geom)) \
                            .join(DepartementGeom, Departement.dep==DepartementGeom.code) \
                            .filter(Departement.dep==user_location) \
                            .first()
                    except TypeError:
                        pass
                    else:
                        if dept is not None:
                            longitude = session.scalar(func.ST_X(centroid))
                            latitude = session.scalar(func.ST_Y(centroid))
                            result[user_location] = {
                                'weight': 10,
                                'type': 'department',
                                'info': dept,
                                'label': dept.title,
                                'geom': (longitude, latitude),
                                'dept': dept.dep,
                                'dept_label': dept.title
                            }
                else:  # GPS coordinates
                    try:
                        longitude, latitude = user_location.split()
                    except ValueError:
                        continue
                    else:
                        marker = f'POINT({longitude} {latitude})'
                        session = get_user_session(RDF_SESSION)
                        for foret, centroid, info_foret, dept in session.query(Foret,
                                                                               func.ST_Centroid(ForetGeom.geom),
                                                                               InformationForet,
                                                                               Departement) \
                                .join(ForetGeom, Foret.id_nat_frt == ForetGeom.id_nat_frt) \
                                .outerjoin(InformationForet, Foret.id_nat_frt == InformationForet.id_nat_frt) \
                                .join(Departement, Foret.departement == Departement.dep) \
                                .filter(ForetGeom.geom.ST_Intersects(func.ST_GeomFromText(marker, WGS84))):
                            result[foret.id_nat_frt] = {
                                'weight': 30,
                                'type': 'forest',
                                'info': (foret, info_foret),
                                'label': getattr(info_foret, 'libelle', None) or foret.title,
                                'geom': (longitude, latitude),
                                'dept': foret.departement,
                                'dept_label': dept.title
                            }
                        session = get_user_session(INSEE_SESSION)
                        for commune, centroid, dept in session.query(Commune,
                                                                     func.ST_Centroid(CommuneGeom.geom),
                                                                     Departement) \
                                .join(CommuneGeom, Commune.code==CommuneGeom.code) \
                                .join(Departement, Commune.dep==Departement.dep) \
                                .filter(CommuneGeom.geom.ST_Intersects(func.ST_GeomFromText(marker, WGS84))):
                            result[commune.code] = {
                                'weight': 20,
                                'type': 'commune',
                                'info': commune,
                                'label': commune.title,
                                'geom': (longitude, latitude),
                                'dept': commune.dep,
                                'dept_label': dept.title
                            }
        return result

    @property
    def values_map(self):
        result = {}
        for key, location in self.locations.items():
            result[key] = location.get('label')
        return json.dumps(result)

    def clean_url(slef , url):
        url = url.split('?')[0]
        url = url.split('#')[0]
        if url.endswith('/preview.html'):
            url = url[:-13]
        return url


NEARME_SEARCH_PARAMS_SESSION_KEY = 'onf_website.nearme.map.search_params'


@factory_config(INearmeMapSearchResultsPortletRendererSettings)
class NearmeMapSearchResultsPortletRendererSettings(NearmeSearchResultsPortletRendererSettings, MapConfiguration):
    """Nearme map search results portlet renderer settings"""

    display_illustrations = FieldProperty(
            IMapWithHeadFiltersSearchResultsPortletRendererSettings['display_illustrations'])
    marker_image = FileProperty(
            IMapWithHeadFiltersSearchResultsPortletRendererSettings['marker_image'])
    display_clusters = FieldProperty(
            IMapWithHeadFiltersSearchResultsPortletRendererSettings['display_clusters'])

    _use_default_map_configuration = FieldProperty(
            INearmeMapSearchResultsPortletRendererSettings['use_default_map_configuration'])

    map_height = FieldProperty(
            IMapWithHeadFiltersSearchResultsPortletRendererSettings['map_height'])
    map_distance = FieldProperty(
            INearmeMapSearchResultsPortletRendererSettings['map_distance'])

    allow_pagination = False

    @property
    def use_default_map_configuration(self):
        return self._use_default_map_configuration

    @use_default_map_configuration.setter
    def use_default_map_configuration(self, value):
        self._use_default_map_configuration = value

    @property
    def no_use_default_map_configuration(self):
        return not bool(self.use_default_map_configuration)

    @no_use_default_map_configuration.setter
    def no_use_default_map_configuration(self, value):
        self.use_default_map_configuration = not bool(value)

    @property
    def configuration(self):
        if self.use_default_map_configuration:
            manager = get_utility(IMapManager)
            return IMapConfiguration(manager).get_configuration()
        return self.get_configuration()


@adapter_config(name='nearme-map-filters',
                context=(IPortalContext, IPyAMSLayer, Interface, ISearchResultsPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/nearme-search-map.pt', layer=IPyAMSLayer)
class NearmeMapSearchResultsPortletRenderer(BaseMapMarkersRendererMixin,
                                            NearmeSearchResultsPortletBaseRenderer):
    """Search results portlet nearme map renderer"""

    label = _("ONF: 'Near me' search map")

    settings_interface = INearmeMapSearchResultsPortletRendererSettings
    weight = 72

    default_page_length = 999

    def update(self):
        super().update()
        self.request.GET['map_distance'] = str(self.renderer_settings.map_distance)
        self.request.session[NEARME_SEARCH_PARAMS_SESSION_KEY] = self.request.query_string

    @property
    def user_location(self):
        locations = sorted(self.locations.items(), key=lambda x: x[1]['weight'])
        return locations[0][0] if locations else None
    
    @property
    def list_mode_params(self):
        departements = set((location['dept'] for location in self.locations.values()))
        return '&'.join((f'user_location={v}' for v in departements))

    def get_markers(self, results, marker_image=None):
        return super().get_markers(results, self.renderer_settings.marker_image)


#
# Nearme search results portlet list renderer
#

@adapter_config(name='nearme-list-results',
                required=(IPortalContext, IPyAMSLayer, Interface, ISearchResultsPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/nearme-search-list.pt',
                 layer=IPyAMSLayer)
@implementer(ISearchResultsView)
class NearmeListSearchResultsPortletRenderer(NearmeSearchResultsPortletBaseRenderer):
    """Search results portlet nearme results list renderer"""

    label = _("ONF: 'Near me' results list")
    weight = 73

    settings_interface = INearmeSearchResultsPortletRendererSettings

    def update(self):
        super().update()
        aside_sticky.need()

    @property
    def map_mode_params(self):
        return self.request.session.get(NEARME_SEARCH_PARAMS_SESSION_KEY)
    
    @property
    def location_label(self):
        return ', '.join(sorted(set(location.get('dept_label') for location in self.locations.values())))


@adapter_config(context=(IWfSharedContent, IPyAMSUserLayer, NearmeListSearchResultsPortletRenderer),
                provides=ISearchResultRenderer)
@template_config(template='templates/nearme-card.pt', layer=IPyAMSUserLayer)
class WFNearmeSearchResultsPortletCardRenderer(ViewContentProvider):
    """Shared content search result item card renderer"""

    @property
    def url(self):
        return self.request.registry.queryMultiAdapter((self.context, self.request, self.view),
                                                       ISearchResultTarget)

    @property
    def title(self):
        return self.request.registry.queryMultiAdapter((self.context, self.request, self.view),
                                                       ISearchResultTitle)


@contentprovider_config(name='onf_inner_help',
                        context=IUniverse,
                        layer=IONFBaseLayer,
                        view=NearmeMapSearchResultsPortletRenderer)
@contentprovider_config(name='onf_inner_help',
                        context=IUniverse,
                        layer=IONFBaseLayer,
                        view=NearmeListSearchResultsPortletRenderer)
@template_config(template='templates/nearme-help.pt', layer=IONFBaseLayer)
class UniverseInnerHelp(ViewContentProvider):
    """Universe inner help content provider"""

    settings = None

    def update(self, **kwargs):
        super().update()
        self.settings = kwargs.get('settings')

    def render(self, template_name=''):
        i18n = II18n(self.settings)
        if not (i18n.query_attribute('help_title', request=self.request) or
                i18n.query_attribute('help_message', request=self.request)):
            return ''
        return super().render(template_name)
