#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from transaction.interfaces import ITransactionManager
from z3c.form import field
from z3c.form.browser.checkbox import SingleCheckBoxFieldWidget

from onf_website.features.header.interfaces import IDefaultHeaderMenu, \
    IDefaultHeaderMenusContainer, IDefaultHeaderRendererSettings
from pyams_content.features.header.zmi import HeaderSettingsRendererSettingsEditForm
from pyams_content.features.menu.zmi import IMenuLinksView, IMenusView, LinksTable, \
    MenuLinksView, MenusTable, MenusView
from pyams_form.group import NamedWidgetsGroup
from pyams_form.interfaces.form import IInnerSubForm
from pyams_portal.interfaces import MANAGE_TEMPLATE_PERMISSION
from pyams_skin.event import get_json_widget_refresh_event
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import adapter_config
from pyams_zmi.form import InnerAdminEditForm


__docformat__ = 'restructuredtext'

from onf_website import _


@adapter_config(name='default-header-properties',
                context=(IDefaultHeaderRendererSettings, IPyAMSLayer, HeaderSettingsRendererSettingsEditForm),
                provides=IInnerSubForm)
class DefaultHeaderPropertiesEditForm(InnerAdminEditForm):
    """Default header properties edit form"""

    legend = None

    edit_permission = MANAGE_TEMPLATE_PERMISSION

    weight = 1

    @property
    def fields(self):
        fields = field.Fields(IDefaultHeaderRendererSettings).omit('use_context_illustration')
        if not self.context.can_use_context_illustration:
            fields = fields.omit('no_use_context_illustration')
        else:
            fields['no_use_context_illustration'].widgetFactory = SingleCheckBoxFieldWidget
        fields['show_frontispiece'].widgetFactory = SingleCheckBoxFieldWidget
        fields['show_subscriber_menu'].widgetFactory = SingleCheckBoxFieldWidget
        fields['search_menu'].widgetFactory = SingleCheckBoxFieldWidget
        fields['contact_menu'].widgetFactory = SingleCheckBoxFieldWidget
        return fields

    def updateGroups(self):
        if self.context.can_use_context_illustration:
            group = NamedWidgetsGroup(self, 'background', self.widgets,
                                      ('no_use_context_illustration', 'background'),
                                      legend=_("Don't use context illustration as background"),
                                      css_class='inner',
                                      switch=True,
                                      checkbox_switch=True,
                                      checkbox_mode='hide',
                                      checkbox_field=IDefaultHeaderRendererSettings[
                                          'no_use_context_illustration'])
        else:
            group = NamedWidgetsGroup(self, 'background', self.widgets,
                                      ('background',))
        self.add_group(group)
        self.add_group(NamedWidgetsGroup(self, 'properties', self.widgets,
                                         ('heading', 'marianne', 'logo', 'social_links_header',
                                          'accessibility_menu')))
        self.add_group(NamedWidgetsGroup(self, 'frontispiece', self.widgets,
                                         ('show_frontispiece', 'show_frontispiece_on_childs',
                                          'show_frontispiece_on_shared_contents'),
                                         legend=_("Show frontispiece"),
                                         css_class='inner',
                                         switch=True,
                                         checkbox_switch=True,
                                         checkbox_mode='hide',
                                         checkbox_field=IDefaultHeaderRendererSettings[
                                             'show_frontispiece']))
        self.add_group(NamedWidgetsGroup(self, 'subscriber', self.widgets,
                                         ('show_subscriber_menu', 'subscriber_menu_label',
                                          'subscriber_menu_header'),
                                         legend=_("Show subscriber menu"),
                                         css_class='inner',
                                         switch=True,
                                         checkbox_switch=True,
                                         checkbox_mode='hide',
                                         checkbox_field=IDefaultHeaderRendererSettings[
                                             'show_subscriber_menu']))
        self.add_group(NamedWidgetsGroup(self, 'search', self.widgets,
                                         ('search_menu', 'search_target', 'experiential_engine'),
                                         legend=_("Show search menu"),
                                         css_class='inner',
                                         switch=True,
                                         checkbox_switch=True,
                                         checkbox_mode='hide',
                                         checkbox_field=IDefaultHeaderRendererSettings[
                                             'search_menu']))
        self.add_group(NamedWidgetsGroup(self, 'contact', self.widgets,
                                         ('contact_menu', 'contact_target'),
                                         legend=_("Show contact link"),
                                         css_class='inner',
                                         switch=True,
                                         checkbox_switch=True,
                                         checkbox_mode='hide',
                                         checkbox_field=IDefaultHeaderRendererSettings[
                                             'contact_menu']))
        super(DefaultHeaderPropertiesEditForm, self).updateGroups()

    def get_ajax_output(self, changes):
        output = super(DefaultHeaderPropertiesEditForm, self).get_ajax_output(changes)
        updated = changes.get(IDefaultHeaderRendererSettings, ())
        if updated:
            ITransactionManager(self.context).commit()
            if 'background' in updated:
                output.setdefault('events', []).append(
                    get_json_widget_refresh_event(self.context, self.request,
                                                  DefaultHeaderPropertiesEditForm, 'background'))
            if 'marianne' in updated:
                output.setdefault('events', []).append(
                    get_json_widget_refresh_event(self.context, self.request,
                                                  DefaultHeaderPropertiesEditForm, 'marianne'))
            if 'logo' in updated:
                output.setdefault('events', []).append(
                    get_json_widget_refresh_event(self.context, self.request,
                                                  DefaultHeaderPropertiesEditForm, 'logo'))
        return output


#
# Header frontispiece links table view
#

class FrontispieceLinksAssociationsTable(LinksTable):
    """Default header frontispiece links associations table"""

    associations_name = 'front'


@adapter_config(name='default-header-front-links',
                context=(IDefaultHeaderRendererSettings, IPyAMSLayer,
                         HeaderSettingsRendererSettingsEditForm),
                provides=IInnerSubForm)
@adapter_config(name='++ass++front',
                context=(IDefaultHeaderMenu, IPyAMSLayer),
                provides=IMenuLinksView)
class DefaultHeaderFrontispieceLinksView(MenuLinksView):
    """Default header frontispiece links view"""

    title = _("Frontispiece")

    table_class = FrontispieceLinksAssociationsTable
    weight = 4


#
# Header subscribers links table view
#

class SubscribersLinksAssociationsTable(LinksTable):
    """Default header subscribers links associations table"""

    associations_name = 'subs'


@adapter_config(name='default-header-subs-links',
                context=(IDefaultHeaderRendererSettings, IPyAMSLayer,
                         HeaderSettingsRendererSettingsEditForm),
                provides=IInnerSubForm)
@adapter_config(name='++ass++subs',
                context=(IDefaultHeaderMenu, IPyAMSLayer),
                provides=IMenuLinksView)
class DefaultHeaderSubscribersLinksView(MenuLinksView):
    """Default header subscribers links view"""

    title = _("Subscribers links")

    table_class = SubscribersLinksAssociationsTable
    weight = 6


#
# Header top links table view
#

class TopLinksAssociationsTable(LinksTable):
    """Default header top links associations table"""

    associations_name = 'top'


@adapter_config(name='default-header-top-links',
                context=(IDefaultHeaderRendererSettings, IPyAMSLayer,
                         HeaderSettingsRendererSettingsEditForm),
                provides=IInnerSubForm)
@adapter_config(name='++ass++top',
                context=(IDefaultHeaderMenu, IPyAMSLayer),
                provides=IMenuLinksView)
class DefaultHeaderTopLinksView(MenuLinksView):
    """Default header top links view"""

    title = _("Top row links")

    table_class = TopLinksAssociationsTable
    weight = 10


#
# Header menus table view
#

@adapter_config(name='default-header-menus',
                context=(IDefaultHeaderRendererSettings, IPyAMSLayer,
                         HeaderSettingsRendererSettingsEditForm),
                provides=IInnerSubForm)
@adapter_config(name='++ass++menus',
                context=(IDefaultHeaderMenusContainer, IPyAMSLayer),
                provides=IMenusView)
class DefaultHeaderMenusView(MenusView):
    """Default header middle groups view"""

    title = _("Header menus")

    table_class = MenusTable
    weight = 11


#
# Header social links table view
#

class SocialLinksAssociationsTable(LinksTable):
    """Default header social links associations table"""

    associations_name = 'social'


@adapter_config(name='default-header-social-links',
                context=(IDefaultHeaderRendererSettings, IPyAMSLayer,
                         HeaderSettingsRendererSettingsEditForm),
                provides=IInnerSubForm)
@adapter_config(name='++ass++social',
                context=(IDefaultHeaderMenu, IPyAMSLayer),
                provides=IMenuLinksView)
class DefaultHeaderSocialLinksView(MenuLinksView):
    """Default header social links view"""

    title = _("Social networks links")

    table_class = SocialLinksAssociationsTable
    weight = 15
