#
# Copyright (c) 2008-2017 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from sqlalchemy.orm import object_session
from sqlalchemy.sql import func

from onf_website.component.paragraph.interfaces.contact import IContactParagraph
from onf_website.reference.insee.model import Commune, CommuneGeom
from pyams_content.features.renderer import IContentRenderer
from pyams_default_theme.component.paragraph.contact import \
    ContactParagraphDefaultRenderer as BaseContactParagraphDefaultRenderer
from pyams_gis.point import GeoPoint
from pyams_skin.layer import IPyAMSLayer
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config


@adapter_config(name='default',
                context=(IContactParagraph, IPyAMSLayer),
                provides=IContentRenderer)
@template_config(template='templates/contact-default.pt',
                 layer=IPyAMSLayer)
@template_config(name='PARA:title-left',
                 template='templates/contact-title-left.pt',
                 layer=IPyAMSLayer)
class ContactParagraphDefaultRenderer(BaseContactParagraphDefaultRenderer):
    """Context paragraph default renderer"""

    def __new__(cls, *args, **kwargs):
        if hasattr(cls, 'context_attrs'):
            cls.context_attrs = cls.context_attrs + ('city',)
        return super().__new__(cls)

    @property
    def gps_location(self):
        result = super().gps_location
        if result:
            return result
        city = self.context.city
        if city is not None:
            commune = Commune.find_by_insee_code(city) \
                .join(CommuneGeom, Commune.code==CommuneGeom.code) \
                .first()
            if commune is not None:
                session = object_session(commune)
                centroid = session.scalar(func.ST_Centroid(commune.ref_geom.geom))
                return GeoPoint(longitude=session.scalar(func.ST_X(centroid)),
                                latitude=session.scalar(func.ST_Y(centroid)))
        return None
