#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'


from pyams_content.component.map.interfaces import have_gis
if have_gis:

    import json
    import sys

    from pyramid.decorator import reify
    from zope.schema.fieldproperty import FieldProperty

    from onf_website.component.location import ILocationInfo, ILocationTarget
    from onf_website.component.map.interfaces import IMapParagraph
    from onf_website.component.map.skin.interfaces import \
        IMapParagraphDefaultRendererSettings
    from pyams_content.features.renderer import IContentRenderer
    from pyams_default_theme.component.map import MAP_DEFAULT_RENDERER_SETTINGS_KEY, \
        MapParagraphDefaultRenderer as BaseMapParagraphDefaultRenderer, \
        MapParagraphDefaultRendererSettings as BaseMapParagraphDefaultRendererSettings
    from pyams_gis.layer import GeoJSONLayer
    from pyams_skin.layer import IPyAMSLayer
    from pyams_template.template import template_config
    from pyams_utils.adapter import adapter_config, get_annotation_adapter
    from pyams_utils.factory import factory_config
    from pyams_utils.interfaces import ICacheKeyValue
    from pyams_utils.traversing import get_parent
    from pyams_utils.url import absolute_url


    @factory_config(IMapParagraphDefaultRendererSettings)
    class MapParagraphDefaultRendererSettings(BaseMapParagraphDefaultRendererSettings):
        """Map paragraph default renderer settings"""

        display_context_forests = FieldProperty(
            IMapParagraphDefaultRendererSettings['display_context_forests'])
        set_bounds_to_forests = FieldProperty(
            IMapParagraphDefaultRendererSettings['set_bounds_to_forests'])


    @adapter_config(context=IMapParagraph,
                    provides=IMapParagraphDefaultRendererSettings)
    def map_paragraph_default_renderer_settings_factory(context):
        """Map paragraph default renderer settings factory"""
        return get_annotation_adapter(context, MAP_DEFAULT_RENDERER_SETTINGS_KEY,
                                      IMapParagraphDefaultRendererSettings)


    #
    # Map paragraph default renderer
    #

    class BaseMapRendererMixin:
        """Base map renderer mixin class"""

        @staticmethod
        def get_bounds(locations):
            """Get bounds of given locations"""
            if not locations:
                return None
            bounds = {
                'x1': sys.maxsize,
                'y1': sys.maxsize,
                'x2': -sys.maxsize,
                'y2': -sys.maxsize
            }
            for location in locations:
                bounds['x1'] = min(location.longitude, bounds['x1'])
                bounds['x2'] = max(location.longitude, bounds['x1'])
                bounds['y1'] = min(location.latitude, bounds['y1'])
                bounds['y2'] = max(location.latitude, bounds['y2'])
            return bounds

        @staticmethod
        def get_location_target(context):
            return get_parent(context, ILocationTarget)

        def get_gps_locations(self, context, settings, renderer_settings):
            if settings.use_context_location:
                parent = self.get_location_target(context)
                if parent is None:
                    return None
                locations = ILocationInfo(parent).locations
                if not locations:
                    return None
            else:
                if not settings.gps_location:
                    return None
                locations = [settings.gps_location]
            return locations

        def get_map_configuration(self, context, settings, renderer_settings):
            configuration = renderer_settings.configuration
            fit_layer = None
            if renderer_settings.display_context_forests:
                parent = self.get_location_target(context)
                if parent is not None:
                    location = ILocationInfo(parent)
                    if location.forests:
                        key = ICacheKeyValue(parent)
                        layer = GeoJSONLayer()
                        layer.name = '{}-forests'.format(key)
                        layer.url = absolute_url(parent, self.request,
                                                 'get-location-forests.json')
                        layer.style = json.dumps({
                            'color': '#00dd00',
                            'weight': '3',
                            'opacity': '0.5'
                        })
                        configuration.setdefault('layers', []).append(
                            layer.get_configuration())
                        if renderer_settings.set_bounds_to_forests:
                            fit_layer = configuration['fitLayer'] = layer.name
            if fit_layer is None:
                locations = self.get_gps_locations(context, settings, renderer_settings)
                if locations:
                    if len(locations) == 1:
                        configuration['center'] = {
                            'lat': float(locations[0].latitude),
                            'lon': float(locations[0].longitude)
                        }
                    else:
                        bounds = self.get_bounds(locations)
                        if bounds:
                            configuration['bounds'] = [{
                                'lat': float(bounds['y1']),
                                'lon': float(bounds['x1'])
                            }, {
                                'lat': float(bounds['y2']),
                                'lon': float(bounds['x2'])
                            }]
                            if 'center' in configuration:
                                del configuration['center']
            return json.dumps(configuration)

        def get_markers(self, context, settings, renderer_settings):
            if not settings.display_marker:
                return None
            locations = self.get_gps_locations(context, settings, renderer_settings)
            if not locations:
                return None
            config = {
                'icon': {
                    'url': '/--static--/onf_website/images/leaflet-marker.png',
                    'size': [40, 53],
                    'anchor': [20, 52]
                },
                'clusterClass': 'onfCluster',
                'tooltipClass': 'onfTooltip',
                'markers': []
            }
            append = config['markers'].append
            for idx, location in enumerate(locations):
                append({
                    'id': '{}::{}'.format(ICacheKeyValue(context), idx),
                    'point': location.to_json()
                })
            return json.dumps(config)

        def get_gps_position(self, context, settings, renderer_settings):
            parent = self.get_location_target(context)
            if parent is None:
                return None
            location = ILocationInfo(parent).gps_location
            return location.get_coordinates() \
                if (location and location.longitude and location.latitude) else None


    @adapter_config(name='default',
                    context=(IMapParagraph, IPyAMSLayer),
                    provides=IContentRenderer)
    @template_config(template='templates/map-default.pt', layer=IPyAMSLayer)
    @template_config(name='PARA:title-left',
                     template='templates/map-title-left.pt', layer=IPyAMSLayer)
    class MapParagraphDefaultRenderer(BaseMapRendererMixin,
                                      BaseMapParagraphDefaultRenderer):
        """Mpp paragraph default renderer"""

        settings_interface = IMapParagraphDefaultRendererSettings

        @reify
        def gps_locations(self):
            return self.get_gps_locations(self.context, self.context, self.settings)

        def get_map_configuration(self):
            return super().get_map_configuration(self.context, self.context, self.settings)

        def get_markers(self):
            return super().get_markers(self.context, self.context, self.settings)

        def get_gps_position(self):
            return super().get_gps_position(self.context, self.context, self.settings)
