#
# Copyright (c) 2008-2017 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from elasticsearch_dsl import Q
from hypatia.interfaces import ICatalog
from hypatia.query import Any
from persistent import Persistent
from sqlalchemy import func
from zope.container.contained import Contained
from zope.schema.fieldproperty import FieldProperty

from onf_website.component.location.interfaces import ILocationInfo, IViewLocationSettings, \
    VIEW_LOCATION_SETTINGS_KEY
from onf_website.reference.forest.model import PARENT_SESSION as RDF_SESSION
from onf_website.reference.forest.model.foret import ForetGeom, REG_ID_NAT_FRT
from onf_website.reference.insee.model import Commune, CommuneGeom, PARENT_SESSION as INSEE_SESSION, REG_COMMUNE_CODE, \
    REG_DEPT_CODE
from onf_website.reference.location.interfaces.region import IRegion
from pyams_alchemy.engine import get_user_session
from pyams_content.shared.view import ViewQuery
from pyams_content.shared.view.interfaces import IViewQueryParamsExtension, IViewSettings, IViewUserQuery, IWfView
from pyams_content_es.interfaces import IViewQueryEsParamsExtension
from pyams_content_es.shared.view import EsViewQuery
from pyams_gis.interfaces import WGS84
from pyams_utils.adapter import ContextAdapter, adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config
from pyams_utils.registry import get_utility

try:
    from osgeo.osr import SpatialReference, CoordinateTransformation
    have_gdal = True
except ImportError:
    have_gdal = False


@factory_config(IViewLocationSettings)
class ViewLocationSettings(Persistent, Contained):
    """View location settings"""

    select_context_gps_location = FieldProperty(
        IViewLocationSettings['select_context_gps_location'])
    gps_location = FieldProperty(IViewLocationSettings['gps_location'])
    gps_distance = FieldProperty(IViewLocationSettings['gps_distance'])
    gps_area = FieldProperty(IViewLocationSettings['gps_area'])
    select_context_forests = FieldProperty(IViewLocationSettings['select_context_forests'])
    forests = FieldProperty(IViewLocationSettings['forests'])
    select_context_cities = FieldProperty(IViewLocationSettings['select_context_cities'])
    cities = FieldProperty(IViewLocationSettings['cities'])
    select_context_departments = FieldProperty(
        IViewLocationSettings['select_context_departments'])
    departments = FieldProperty(IViewLocationSettings['departments'])
    select_context_countries = FieldProperty(IViewLocationSettings['select_context_countries'])
    countries = FieldProperty(IViewLocationSettings['countries'])
    select_context_structures = FieldProperty(IViewLocationSettings['select_context_structures'])
    structures = FieldProperty(IViewLocationSettings['structures'])

    @property
    def is_using_context(self):
        return self.select_context_gps_location or self.select_context_forests or \
               self.select_context_cities or self.select_context_departments or \
               self.select_context_countries or self.select_context_structures

    def get_gps_location(self, context):
        if self.select_context_gps_location:
            location = ILocationInfo(context, None)
            if location is not None:
                coords = location.gps_location.wgs_coordinates
                if coords and coords[0] and coords[1]:
                    return {
                        'filter': 'geo_distance',
                        'location': {
                            "lon": coords[0],
                            "lat": coords[1]
                        },
                        'distance': '{0}km'.format(self.gps_distance)
                    }
        else:
            if self.gps_location:
                coords = self.gps_location.wgs_coordinates
                if coords and coords[0] and coords[1]:
                    return {
                        'filter': 'geo_distance',
                        'location': {
                            "lon": coords[0],
                            "lat": coords[1]
                        },
                        'distance': '{0}km'.format(self.gps_distance)
                    }
            elif self.gps_area:
                coords = self.gps_area.wgs_coordinates
                if coords and coords[0] and coords[1]:
                    return {
                        'filter': 'geo_bounding_box',
                        'area': {
                            "top_left": {
                                "lon": coords[0][0],
                                "lat": coords[1][1]
                            },
                            "bottom_right": {
                                "lon": coords[1][0],
                                "lat": coords[0][1]
                            }
                        }
                    }

    def get_forests(self, context):
        forests = set()
        if self.select_context_forests:
            location = ILocationInfo(context, None)
            if location is not None:
                forests |= set(location.forests_index or ())
        if self.forests:
            forests |= set(self.forests)
        return list(forests)

    def get_cities(self, context):
        cities = set()
        if self.select_context_cities:
            location = ILocationInfo(context, None)
            if location is not None:
                cities |= set(location.cities or ())
        if self.cities:
            cities |= set(self.cities)
        return list(cities)

    def get_departments(self, context):
        departments = set()
        if self.select_context_departments:
            location = ILocationInfo(context, None)
            if location is not None:
                departments |= set(location.get_departments())
        if self.departments:
            for value in self.departments:
                if IRegion.providedBy(value):
                    departments |= set(value.departments or ())
                else:
                    departments.add(value)
        return departments

    def get_departments_index(self, context):
        return [dept.insee_code for dept in self.get_departments(context)]

    def get_countries(self, context):
        countries = set()
        if self.select_context_countries:
            location = ILocationInfo(context, None)
            if location is not None:
                countries |= set(location.countries or ())
        if self.countries:
            countries |= self.countries
        return countries

    def get_countries_index(self, context):
        return [country.insee_code for country in self.get_countries(context)]

    def get_structures(self, context):
        structures = set()
        if self.select_context_structures:
            location = ILocationInfo(context, None)
            if location is not None:
                structures |= set(location.structures or ())
        if self.structures:
            structures |= set(self.structures)
        return list(structures)


@adapter_config(required=IWfView,
                provides=IViewLocationSettings)
@adapter_config(name='location',
                required=IWfView,
                provides=IViewSettings)
def view_location_settings_factory(view):
    """View location settings factory"""
    return get_annotation_adapter(view, VIEW_LOCATION_SETTINGS_KEY, IViewLocationSettings,
                                  name='++view:location++')


@adapter_config(name='location',
                required=IWfView,
                provides=IViewQueryParamsExtension)
class ViewLocationQueryParamsExtension(ContextAdapter):
    """View location query params extension"""

    weight = 50

    def get_params(self, context, request=None):
        catalog = get_utility(ICatalog)
        settings = IViewLocationSettings(self.context)
        # check forests
        forests = settings.get_forests(context)
        if forests:
            yield Any(catalog['forests'], forests)
        elif settings.select_context_forests:
            yield None
        # check cities
        cities = settings.get_cities(context)
        if cities:
            yield Any(catalog['cities'], cities)
        elif settings.select_context_cities:
            yield None
        # check departments
        departments = settings.get_departments_index(context)
        if departments:
            yield Any(catalog['departments'], departments)
        elif settings.select_context_departments:
            yield None
        # check countries
        countries = settings.get_countries_index(context)
        if countries:
            yield Any(catalog['countries'], countries)
        elif settings.select_context_countries:
            yield None
        # structures
        structures = settings.get_structures(context)
        if structures:
            yield Any(catalog['structures'], structures)
        elif settings.select_context_structures:
            yield None


@adapter_config(name='location',
                required=IWfView,
                provides=IViewQueryEsParamsExtension)
class ViewLocationQueryEsParamsExtension(ContextAdapter):
    """View location query params extension for Elasticsearch"""

    weight = 50

    def get_es_params(self, context, request=None):
        settings = IViewLocationSettings(self.context)
        # check GPS location
        gps_filter = None
        location = settings.get_gps_location(context)
        if location:
            filter = location.pop('filter')
            filter_args = None
            if filter == 'geo_distance':
                filter_args = {
                    'location.coords': location['location'],
                    'distance': location['distance']
                }
            elif filter == 'geo_bounding_box':
                filter_args = {
                    'location.coords': location['area']
                }
            if filter_args:
                gps_filter = Q(filter, **filter_args)
        if gps_filter:
            yield gps_filter
        elif settings.select_context_gps_location:
            yield None
        # check forests
        forests = settings.get_forests(context)
        if forests:
            yield Q('terms', **{'location.forests': forests})
        elif settings.select_context_forests:
            yield None
        # check cities
        cities = settings.get_cities(context)
        if cities:
            yield Q('terms', **{'location.cities': cities})
        elif settings.select_context_cities:
            yield None
        # check departments
        departments = settings.get_departments_index(context)
        if departments:
            yield Q('terms', **{'location.departments': departments})
        elif settings.select_context_departments:
            yield None
        # check countries
        countries = settings.get_countries_index(context)
        if countries:
            yield Q('terms', **{'location.countries': countries})
        elif settings.select_context_countries:
            yield None
        # check structures
        structures = settings.get_structures(context)
        if structures:
            yield Q('terms', **{'location.structures': structures})
        elif settings.select_context_structures:
            yield None


@adapter_config(name='forest',
                required=ViewQuery,
                provides=IViewUserQuery)
class ViewForestQuery(ContextAdapter):
    """View forest query"""

    @staticmethod
    def get_user_params(request):
        forests = request.params.getall('forest')
        if forests:
            catalog = get_utility(ICatalog)
            yield Any(catalog['forests'], forests)


@adapter_config(name='forest',
                required=EsViewQuery,
                provides=IViewUserQuery)
class EsViewForestQuery(ContextAdapter):
    """View forest query for Elasticsearch"""

    @staticmethod
    def get_user_params(request):
        forests = request.params.getall('forest')
        if forests:
            yield Q('terms', **{'location.forests': forests})


@adapter_config(name='location',
                required=EsViewQuery,
                provides=IViewUserQuery)
class EsViewLocationQuery(ContextAdapter):
    """View location query for Elasticsearch"""

    @staticmethod
    def get_user_params(request):
        locations = request.params.getall('location')
        if locations:
           yield Q('terms', **{'location.departments': locations})


@adapter_config(name='geo-distance',
                required=EsViewQuery,
                provides=IViewUserQuery)
class EsViewDistanceQuery(ContextAdapter):
    """View geo-distance query for Elasticsearch"""

    @staticmethod
    def get_user_params(request):
        map_center = request.params.get('map_center', '').strip()
        if map_center:
            longitude, latitude = map_center.split()
            if longitude and latitude:
                distance = request.params.get('map_distance', 50)
                yield Q('geo_distance', **{
                    'location.coords': {
                        'lon': longitude,
                        'lat': latitude
                    },
                    'distance': f'{distance}km'
                })


@adapter_config(name='user-location',
                required=EsViewQuery,
                provides=IViewUserQuery)
class EsViewUserLocationQuery(ContextAdapter):
    """View user location query for Elasticsearch"""

    @staticmethod
    def get_centroid_params(session, centroids, distance):
        for centroid in centroids:
            longitude = session.scalar(func.ST_X(centroid[0]))
            latitude = session.scalar(func.ST_Y(centroid[0]))
            yield Q('geo_distance', **{
                'location.coords': {
                    'lon': longitude,
                    'lat': latitude
                },
                'distance': f'{distance}km'
            })
    
    @staticmethod
    def get_user_params(request):
        user_locations = request.params.getall('user_location')
        if user_locations:
            search_params = []
            forests = []
            communes = []
            depts = []
            coords = []
            distance = request.params.get('map_distance', 50)
            for user_location in user_locations:
                user_location = user_location.strip().upper()
                if REG_ID_NAT_FRT.match(user_location):
                    forests.append(user_location)
                elif REG_COMMUNE_CODE.match(user_location):
                    communes.append(user_location)
                elif REG_DEPT_CODE.match(user_location):
                    depts.append(user_location)
                else:
                    try:
                        longitude, latitude = user_location.split()
                    except ValueError:
                        continue
                    else:
                        coords.append((longitude, latitude))
                        marker = f'POINT({longitude} {latitude})'
                        session = get_user_session(RDF_SESSION)
                        for forest in session.query(ForetGeom) \
                                .filter(ForetGeom.geom.ST_Intersects(func.ST_GeomFromText(marker, WGS84))):
                            forests.append(forest.id_nat_frt)
                        session = get_user_session(INSEE_SESSION)
                        for commune in session.query(CommuneGeom) \
                                .filter(CommuneGeom.geom.ST_Intersects(func.ST_GeomFromText(marker, WGS84))):
                            communes.append(commune.code)
            if forests:
                session = get_user_session(RDF_SESSION)
                forests_centroids = session.query(func.ST_Centroid(ForetGeom.geom)) \
                        .filter(ForetGeom.id_nat_frt.in_(forests))
                for param in EsViewUserLocationQuery.get_centroid_params(session, forests_centroids, distance):
                    search_params.append(param)
                # check hearing
                search_params.append(Q('terms', **{'hearing.forests': forests}))
            if communes:
                session = get_user_session(INSEE_SESSION)
                communes_centroids = session.query(func.ST_Centroid(CommuneGeom.geom)) \
                        .filter(CommuneGeom.code.in_(communes))
                for param in EsViewUserLocationQuery.get_centroid_params(session, communes_centroids, distance):
                    search_params.append(param)
                # check hearing
                search_params.append(Q('terms', **{'hearing.cities': communes}))
            if depts:
                session = get_user_session(INSEE_SESSION)
                for dept in depts:
                    dept_centroid = session.query(func.ST_Centroid(func.ST_Union(CommuneGeom.geom))) \
                            .join(Commune, CommuneGeom.code == Commune.code) \
                            .filter(Commune.dep == dept)
                    for param in EsViewUserLocationQuery.get_centroid_params(session, dept_centroid, distance):
                        search_params.append(param)
                # check hearing
                search_params.append(Q('terms', **{'hearing.departments': depts}))
            for longitude, latitude in coords:
                search_params.append(Q('geo_distance', **{
                    'location.coords': {
                        'lon': longitude,
                        'lat': latitude
                    },
                    'distance': f'{distance}km'
                }))
            if search_params:
                yield Q('bool', **{'should': search_params})
