#
# Copyright (c) 2015-2021 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

__docformat__ = 'restructuredtext'

import json

from pyramid.view import view_config
from sqlalchemy import func
from zope.interface import Interface

from onf_website.component.location import ILocationInfo, ILocationTarget
from onf_website.reference.forest.model import PARENT_SESSION
from onf_website.reference.forest.model.foret import ForetGeom
from pyams_alchemy.engine import get_user_session
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import ContextRequestViewAdapter, adapter_config
from pyams_utils.interfaces.tales import ITALESExtension


@view_config(name='get-location-forests.json',
             context=ILocationTarget, request_type=IPyAMSLayer, renderer='json')
def getLocationForests(request):
    """Get location forests"""
    location = ILocationInfo(request.context, None)
    if (location is None) or not location.forests:
        return {}
    session = get_user_session(PARENT_SESSION)
    forests = session.query(ForetGeom.geom) \
                     .filter(ForetGeom.id_nat_frt.in_(location.forests))
    return {
        'geometry': [
            json.loads(session.scalar(func.ST_AsGeoJSON(forest.geom)))
            for forest in forests
        ]
    }


@adapter_config(name='location_dept',
                context=(ILocationTarget, IPyAMSLayer, Interface),
                provides=ITALESExtension)
class LocationDepartmentTALESExtension(ContextRequestViewAdapter):
    """Location department TALES extension"""

    def render(self):
        location = ILocationInfo(self.context, None)
        if location is None:
            return None
        value = location.get_region_and_departments()
        return json.dumps(value) if value else None
