#
# Copyright (c) 2015-2023 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

from elasticsearch_dsl import Q

from onf_website.component.location.interfaces import ILocationInfo, ILocationTarget
from pyams_content.features.json import JSONBaseConverter
from pyams_content.features.json.interfaces import IJSONConverter
from pyams_content.shared.common.api.json import JSONSharedContentConverter
from pyams_content.shared.common.interfaces import IWfSharedContentFinderParams
from pyams_content_es.content import EsSharedToolContentFinder
from pyams_gis.interfaces import WGS84
from pyams_gis.transform import transform
from pyams_skin.layer import IPyAMSLayer
from pyams_utils.adapter import ContextAdapter, adapter_config


__docformat__ = 'restructuredtext'


@adapter_config(name='location',
                required=EsSharedToolContentFinder,
                provides=IWfSharedContentFinderParams)
class EsSharedToolContentFinderLocationFilter(ContextAdapter):
    """Elasticsearch shared tool content finder location filter"""

    def get_params(self, query, params, request=None):
        """Search results position location params getter"""
        if ('longitude' in params) and ('latitude' in params):
            srid = params.get('srid')
            coords = transform({
                'longitude': params['longitude'],
                'latitude': params['latitude']
            }, srid, WGS84).get('point')
            query &= Q('geo_distance', **{
                'location.coords': {
                    'lon': coords['longitude'],
                    'lat': coords['latitude']
                },
                'distance': '{}km'.format(params['distance'])
            })
        return query


@adapter_config(name='location',
                required=(ILocationTarget, IPyAMSLayer),
                provides=IJSONConverter)
class JSONLocationConverter(JSONBaseConverter):
    """JSON location converter"""

    is_inner = True

    @property
    def conversion_target(self):
        return ILocationInfo(self.context, None)

    def convert_content(self, params):
        """JSON conversion"""
        result = super().convert_content(params)
        location = ILocationInfo(self.context)
        if location.gps_location:
            srid = params.get('srid', WGS84)
            try:
                coords = location.gps_location.get_coordinates(srid)
                result['gps_location'] = {
                    'longitude': float(coords[0]),
                    'latitude': float(coords[1]),
                    'srid': srid
                }
            except NotImplementedError:  # unknown SRID
                pass
        if location.forests:
            result['forests'] = location.get_forests_list(params)
        if location.cities:
            result['cities'] = location.cities
        departments = location.departments
        if departments:
            result['departments'] = location.departments_index
            if location.all_departments_forests:
                result['all_departments_forests'] = True
                if location.domanials_only:
                    result['domanials_only'] = True
                if location.communals_only:
                    result['communals_only'] = True
        return result
