#
# Copyright (c) 2008-2024 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from pyquery import PyQuery
from pyramid.events import subscriber
from z3c.form.interfaces import IDataExtractedEvent
from zope.interface import Invalid

from pyams_content.component.links.zmi import ExternalLinkAddForm, ExternalLinkPropertiesEditForm
from pyams_content.component.paragraph.zmi.html import HTMLParagraphAddForm, HTMLParagraphPropertiesEditForm
from pyams_content.root import IConfiguration, ISiteRootConfiguration

__docformat__ = 'restructuredtext'

from onf_website import _


@subscriber(IDataExtractedEvent, form_selector=ExternalLinkAddForm)
@subscriber(IDataExtractedEvent, form_selector=ExternalLinkPropertiesEditForm)
def extract_external_link_data(event):
    """Check URL when creating external link"""
    configuration = IConfiguration(event.form.request.root, None)
    if (not ISiteRootConfiguration.providedBy(configuration)) or (not configuration.public_url):
        return
    data = event.data
    if data.get('url').startswith(configuration.public_url):
        event.form.widgets.errors += (Invalid(_("You must use an internal link to create a link to "
                                                "another site content!")),)


@subscriber(IDataExtractedEvent, form_selector=HTMLParagraphAddForm)
@subscriber(IDataExtractedEvent, form_selector=HTMLParagraphPropertiesEditForm)
def extract_html_paragraph_data(event):
    """Check HTML when creating HTML paragraph"""
    configuration = IConfiguration(event.form.request.root, None)
    if (not ISiteRootConfiguration.providedBy(configuration)) or (not configuration.public_url):
        return
    data = event.data
    for lang, body in data.get('body', {}).items():
        html = PyQuery('<html>{0}</html>'.format(body))
        for link in html('a[href]'):
            href = link.attrib['href']
            if href.startswith(configuration.public_url) or href.startswith('/') or href.startswith('../'):
                event.form.widgets.errors += (Invalid(_("You must use an internal link to create a link to "
                                                        "another site content!")),)
                break
