#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from pyramid.events import subscriber
from z3c.form import button, field
from z3c.form.browser.checkbox import CheckBoxFieldWidget, SingleCheckBoxFieldWidget
from z3c.form.interfaces import IDataExtractedEvent, INPUT_MODE
from zope.dublincore.interfaces import IZopeDublinCore
from zope.interface import Interface, Invalid, alsoProvides, implementer
from zope.lifecycleevent import Attributes, ObjectModifiedEvent

from onf_website.component.hearing.interfaces import HEARING_SETTINGS, IHearingDiffusionSitesInfo, \
    IHearingInfo, IHearingLocationsInfo, IHearingManagerSettings, IHearingManagerTarget, \
    IHearingNationalInfo, IHearingOrganizationInfo, IHearingReadershipInfo, IHearingTarget
from onf_website.component.location.interfaces import ILocationTarget
from pyams_content.features.search import ISearchFolder
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION
from pyams_content.shared.common import ISharedContent, IWfSharedContent
from pyams_content.shared.common.zmi import WfSharedContentHeaderAdapter
from pyams_content.shared.site.interfaces import ISiteManager
from pyams_form.form import ajax_config
from pyams_form.group import NamedWidgetsGroup
from pyams_form.help import FormHelp
from pyams_form.interfaces.form import IFormHelp, IInnerSubForm, IWidgetForm
from pyams_form.schema import CloseButton, ResetButton
from pyams_pagelet.pagelet import pagelet_config
from pyams_sequence.reference import get_reference_target
from pyams_skin.interfaces import IInnerPage, IPageHeader
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.viewlet.menu import MenuDivider, MenuItem
from pyams_utils.adapter import adapter_config
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION
from pyams_utils.interfaces.inherit import IInheritInfo
from pyams_utils.registry import get_current_registry
from pyams_utils.traversing import get_parent
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogEditForm, AdminEditForm, InnerAdminEditForm
from pyams_zmi.interfaces.menu import IPropertiesMenu
from pyams_zmi.layer import IAdminLayer

__docformat__ = 'restructuredtext'

from onf_website import _


@viewlet_config(name='hearing.divider',
                context=IHearingTarget, layer=IAdminLayer,
                manager=IPropertiesMenu, weight=309,
                permission=VIEW_SYSTEM_PERMISSION)
class HearingMenuDivider(MenuDivider):
    """Hearing menu divider"""

    def __new__(cls, context, request, view, manager=None):
        if ILocationTarget.providedBy(context):  # avoid duplicated divider
            return None
        return MenuDivider.__new__(cls)


@viewlet_config(name='hearing.menu',
                context=IHearingTarget, layer=IAdminLayer,
                manager=IPropertiesMenu, weight=320,
                permission=VIEW_SYSTEM_PERMISSION)
class HearingMenu(MenuItem):
    """Hearing menu"""

    label = _("Hearing...")
    icon_class = 'fa-volume-up'
    url = '#hearing.html'


class IHearingEditGroup(Interface):
    """Inner hearing edit group"""


class IHearingEditButtons(Interface):
    """Hearing form buttons"""

    reset = ResetButton(name='reset', title=_("Reset"))
    submit = button.Button(name='submit', title=_("Submit"))


class IHearingDialogEditButtons(Interface):
    """Hearing dialog form buttons"""

    cancel = CloseButton(name='cancel', title=_("Cancel"))
    submit = button.Button(name='submit', title=_("Submit"))


class HearingEditFormMixin:
    """Hearing edit form mixin"""

    legend = _("Content hearing")

    @property
    def fields(self):
        if IHearingInfo(self.context).can_inherit:
            fields = field.Fields(IHearingInfo).select('no_inherit')
            fields['no_inherit'].widgetFactory = SingleCheckBoxFieldWidget
        else:
            fields = field.Fields(Interface)
        return fields

    def updateGroups(self):
        if IHearingInfo(self.context).can_inherit:
            group = NamedWidgetsGroup(self, 'hearing', self.widgets,
                                      ('no_inherit',),
                                      legend=_("Don't inherit parent hearing"),
                                      css_class='inner',
                                      switch=True,
                                      checkbox_switch=True,
                                      checkbox_mode='disable',
                                      checkbox_field=IHearingInfo['no_inherit'])
        else:
            group = NamedWidgetsGroup(self, 'hearing', self.widgets, (), css_class='inner')
        alsoProvides(group, IHearingEditGroup)
        self.add_group(group)
        super().updateGroups()


@pagelet_config(name='hearing.html',
                context=IHearingTarget, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@ajax_config(name='hearing.json',
             context=IHearingTarget, layer=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION,
             implementer=IHearingEditGroup)
@implementer(IWidgetForm, IInnerPage)
class HearingEditForm(HearingEditFormMixin, AdminEditForm):
    """Hearing edit form"""

    @property
    def buttons(self):
        if self.mode == INPUT_MODE:
            return button.Buttons(IHearingEditButtons)
        return button.Buttons(Interface)

    def get_ajax_output(self, changes):
        if changes:
            hearing = IHearingInfo(self.getContent())
            descriptions = []
            for interface, names in changes.items():
                descriptions.append(Attributes(interface, *names))
            get_current_registry().notify(ObjectModifiedEvent(hearing, *descriptions))
        output = super(self.__class__, self).get_ajax_output(changes)
        if 'no_inherit' in changes.get(IInheritInfo, ()):
            output.update({
                'status': 'reload',
                'location': '#hearing.html'
            })
        return output


@pagelet_config(name='hearing-dialog.html',
                context=IHearingTarget, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@ajax_config(name='hearing-dialog.json',
             context=IHearingTarget, layer=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION,
             implementer=IHearingEditGroup)
class HearingModalEditForm(HearingEditFormMixin, AdminDialogEditForm):
    """Hearing modal edit form"""

    dialog_class = 'modal-large'

    @property
    def buttons(self):
        if self.mode == INPUT_MODE:
            return button.Buttons(IHearingDialogEditButtons)
        return button.Buttons(Interface)

    def get_ajax_output(self, changes):
        if changes:
            hearing = IHearingInfo(self.getContent())
            get_current_registry().notify(ObjectModifiedEvent(hearing))
        return {
            'status': 'success',
            'message': self.request.localizer.translate(self.successMessage)
        }


class HearingInnerEditForm(InnerAdminEditForm):
    """Hearing inner admin edit form"""

    legend = None
    name = None

    def __new__(cls, context, request, view=None):
        manager = get_parent(context, IHearingManagerTarget)
        if manager is None:  # site root
            manager = request.root
        settings = IHearingManagerSettings(manager).get(cls.name)
        if (not settings) or (not settings.visible):
            return None
        return InnerAdminEditForm.__new__(cls)

    @property
    def main_group_legend(self):
        return HEARING_SETTINGS.get(self.name)

    main_group_class = 'inner switcher no-y-padding'

    def update(self):
        super(HearingInnerEditForm, self).update()
        group = self.groups[0]
        group.switch = True
        context_dc = IZopeDublinCore(self.context)
        hearing_dc = IZopeDublinCore(IHearingInfo(self.context))
        if hearing_dc.modified and (hearing_dc.modified > (hearing_dc.created or context_dc.modified)):
            group.display_mode = 'auto'
        else:
            manager = get_parent(self.context, IHearingManagerTarget)
            if manager is None:
                manager = self.request.root
            setting = IHearingManagerSettings(manager)[self.name]
            group.display_mode = 'always' if setting.unfolded else 'auto'


@adapter_config(name='readership',
                context=(IHearingTarget, IPyAMSLayer, IHearingEditGroup),
                provides=IInnerSubForm)
class HearingReadershipEditSubform(HearingInnerEditForm):
    """Hearing readership edit form"""

    name = 'readership'

    fields = field.Fields(IHearingReadershipInfo).select('targets', 'private_groups')
    fields['targets'].widgetFactory = CheckBoxFieldWidget
    weight = 1


@adapter_config(name='national',
                context=(IHearingTarget, IPyAMSLayer, IHearingEditGroup),
                provides=IInnerSubForm)
class HearingNationalEditSubform(HearingInnerEditForm):
    """Hearing national scope edit form"""

    name = 'national'

    fields = field.Fields(IHearingNationalInfo)
    fields['national_scope'].widgetFactory = SingleCheckBoxFieldWidget
    weight = 2


@adapter_config(name='location',
                context=(IHearingTarget, IPyAMSLayer, IHearingEditGroup),
                provides=IInnerSubForm)
class HearingLocationEditSubform(HearingInnerEditForm):
    """Hearing geographic scope edit form"""

    name = 'location'

    fields = field.Fields(IHearingLocationsInfo)
    fields['departments_forests'].widgetFactory = SingleCheckBoxFieldWidget
    fields['domanials_only'].widgetFactory = SingleCheckBoxFieldWidget
    fields['communals_only'].widgetFactory = SingleCheckBoxFieldWidget
    weight = 3

    def updateWidgets(self, prefix=None):
        super().updateWidgets(prefix)
        widget = self.widgets.get('departments_forests')
        if widget is not None:
            translate = self.request.localizer.translate
            if 'forests' in self.widgets:
                self.widgets['forests'].after_widget_notice = \
                    translate(_("...of departments selected below:"))
            widget.group_css_class = 'form-group col-md-6 no-y-padding'
            widget.label_css_class = 'control-label col-md-6'
            widget.input_css_class = 'col-md-6'
            widget.widget_css_class = 'input padding-left-20'
        widget = self.widgets.get('domanials_only')
        if widget is not None:
            widget.group_css_class = 'form-group col-md-3 no-y-padding'
            widget.label_css_class = 'control-label col-md-1'
            widget.input_css_class = 'col-md-11'
            widget.widget_css_class = 'input padding-left-20'
        widget = self.widgets.get('communals_only')
        if widget is not None:
            widget.group_css_class = 'form-group col-md-3 no-y-padding'
            widget.label_css_class = 'control-label col-md-1'
            widget.input_css_class = 'col-md-11'
            widget.widget_css_class = 'input padding-left-20'


@adapter_config(name='organization',
                context=(IHearingTarget, IPyAMSLayer, IHearingEditGroup),
                provides=IInnerSubForm)
class HearingOrganizationEditSubform(HearingInnerEditForm):
    """Hearing organization edit form"""

    name = 'organization'

    fields = field.Fields(IHearingOrganizationInfo)
    fields['departments_structures'].widgetFactory = SingleCheckBoxFieldWidget
    fields['metropolitan_structures'].widgetFactory = SingleCheckBoxFieldWidget
    fields['corsican_structures'].widgetFactory = SingleCheckBoxFieldWidget
    fields['dom_structures'].widgetFactory = SingleCheckBoxFieldWidget
    weight = 4

    def updateWidgets(self, prefix=None):
        super(HearingOrganizationEditSubform, self).updateWidgets(prefix)
        if 'departments_structures' in self.widgets:
            self.widgets['departments_structures'].group_css_class = 'form-group no-y-padding'
        if 'metropolitan_structures' in self.widgets:
            widget = self.widgets['metropolitan_structures']
            widget.group_css_class = 'form-group col-md-6 no-y-padding'
            widget.label_css_class = 'control-label col-md-6'
            widget.input_css_class = 'col-md-6'
            widget.widget_css_class = 'input padding-left-2'
        if 'corsican_structures' in self.widgets:
            widget = self.widgets['corsican_structures']
            widget.group_css_class = 'form-group col-md-3 no-y-padding'
            widget.label_css_class = 'control-label col-md-1'
            widget.input_css_class = 'col-md-11'
            widget.widget_css_class = 'input padding-left-2'
        if 'dom_structures' in self.widgets:
            widget = self.widgets['dom_structures']
            widget.group_css_class = 'form-group col-md-3 no-y-padding'
            widget.label_css_class = 'control-label col-md-1'
            widget.input_css_class = 'col-md-11'
            widget.widget_css_class = 'input padding-left-2'


@adapter_config(name='sites',
                context=(IHearingTarget, IPyAMSLayer, IHearingEditGroup),
                provides=IInnerSubForm)
class HearingSitesEditSubform(HearingInnerEditForm):
    """Hearing source edit form"""

    name = 'sites'

    fields = field.Fields(IHearingDiffusionSitesInfo)
    weight = 5


@subscriber(IDataExtractedEvent, form_selector=HearingSitesEditSubform)
def extract_hearing_sites_data(event):
    """Extract hearing sites data"""
    source_folder = event.data.get('source_folder')
    if source_folder is not None:
        folder = get_reference_target(source_folder, request=event.form.request)
        if ISiteManager.providedBy(folder) or \
                ISharedContent.providedBy(folder) or \
                (IWfSharedContent.providedBy(folder) and not ISearchFolder.providedBy(folder)):
            event.form.widgets.errors += (Invalid(_("You must select a folder as source folder!")),)
            

@adapter_config(context=(IHearingTarget, IPyAMSLayer, HearingEditForm),
                provides=IFormHelp)
class HearingScopeEditSubformHelpAdapter(FormHelp):
    """Scope edit subform help adapter"""

    message = _("While location defines where this content is related to, hearing defines "
                "contexts where this content will be visible.\n\n"
                "**WARNING:** at first, location settings are copied into hearing until you "
                "modify hearing settings by yourself; afterwards, modifications to location "
                "settings are not copied anymore.")
    message_format = 'rest'


@adapter_config(context=(IHearingTarget, IAdminLayer, HearingEditForm),
                provides=IPageHeader)
class HearingHeaderAdapter(WfSharedContentHeaderAdapter):
    """Shared content hearing header adapter"""

    icon_class = 'fa fa-fw fa-volume-up'
