#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

__docformat__ = 'restructuredtext'

from persistent import Persistent
from persistent.mapping import PersistentMapping
from zope.interface import implementer
from zope.schema.fieldproperty import FieldProperty

from onf_website.component.hearing.interfaces import HEARING_SETTINGS_KEY, HEARING_SETTINGS_KEYS, \
    IHearingManagerSettings, IHearingManagerTarget, IHearingSetting
from pyams_content.root import ISiteRoot
from pyams_utils.adapter import adapter_config, get_annotation_adapter


@implementer(IHearingSetting)
class HearingSetting(object):
    """Hearing component setting"""

    visible = FieldProperty(IHearingSetting['visible'])
    unfolded = FieldProperty(IHearingSetting['unfolded'])

    def __init__(self, name):
        self.name = name


@implementer(IHearingManagerSettings)
class HearingManagerSettings(Persistent):
    """Hearing manager settings"""

    settings = FieldProperty(IHearingManagerSettings['settings'])

    def __init__(self):
        self.settings = PersistentMapping()

    def get(self, key, default=None):
        return self.settings.get(key, default)

    def __setitem__(self, key, value):
        self.settings[key] = value

    def __getitem__(self, key):
        return self.settings[key]


@adapter_config(context=IHearingManagerTarget, provides=IHearingManagerSettings)
def hearing_manager_settings_factory(context):
    """Hearing manager target settings factory"""

    def settings_factory():
        settings = HearingManagerSettings()
        for key in HEARING_SETTINGS_KEYS:
            settings[key] = HearingSetting(key)
        return settings

    return get_annotation_adapter(context, HEARING_SETTINGS_KEY, factory=settings_factory)


@adapter_config(context=ISiteRoot, provides=IHearingManagerSettings)
def site_root_hearing_manager_settings_factory(context):
    """Site root hearing manager target settings factory"""

    def settings_factory():
        settings = HearingManagerSettings()
        for key in HEARING_SETTINGS_KEYS:
            settings[key] = HearingSetting(key)
            settings[key].visible = key in ('national', 'location')
        return settings

    return get_annotation_adapter(context, HEARING_SETTINGS_KEY, factory=settings_factory)
