#
# Copyright (c) 2008-2015 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from persistent import Persistent
from pyramid.events import subscriber
from sqlalchemy.sql import and_, or_
from zope.container.contained import Contained
from zope.intid.interfaces import IIntIds
from zope.lifecycleevent import IObjectModifiedEvent, IObjectMovedEvent
from zope.schema.fieldproperty import FieldProperty

from onf_website.component.hearing.interfaces import HEARING_INFO_KEY, IHearingDiffusionSitesInfo, \
    IHearingInfo, IHearingLocationsInfo, IHearingOrganizationInfo, IHearingTarget
from onf_website.component.location.interfaces import ILocationInfo
from onf_website.reference.forest.model.foret import Foret, PARENT_SESSION as RDF_PARENT_SESSION
from onf_website.reference.insee.model import Commune, PARENT_SESSION as INSEE_PARENT_SESSION
from onf_website.reference.location.interfaces.department import IDepartmentTable
from onf_website.reference.location.interfaces.region import IRegion
from onf_website.reference.location.interfaces.unit import IUnitTable
from onf_website.reference.orga.model import PARENT_SESSION as STRUCT_PARENT_SESSION, Structure
from pyams_alchemy.engine import get_user_session
from pyams_content.features.checker import BaseContentChecker
from pyams_content.features.checker.interfaces import IContentChecker, MISSING_VALUE
from pyams_content.shared.common.interfaces import ISharedSite, IWfSharedContentRoles
from pyams_content_es.interfaces import IDocumentIndexInfo
from pyams_form.interfaces.form import IFormContextPermissionChecker
from pyams_security.interfaces import IProtectedObject
from pyams_sequence.reference import get_reference_target
from pyams_utils.adapter import ContextAdapter, adapter_config, get_annotation_adapter
from pyams_utils.factory import factory_config
from pyams_utils.inherit import BaseInheritInfo, InheritedFieldProperty
from pyams_utils.interfaces import VIEW_PERMISSION
from pyams_utils.list import unique
from pyams_utils.registry import get_utility, query_utility
from pyams_utils.traversing import get_parent
from pyams_utils.zodb import volatile_property


__docformat__ = 'restructuredtext'

from onf_website import _


METROPOLITAN_CODE_FILTER = (Structure.code_sign.like('8%'),)
CORSICAN_CODE_FILTER = (Structure.code_sign.like('9005%'),)
DOM_CODE_FILTER = (Structure.code_sign.like('901%'),
                   Structure.code_sign.like('902%'))


@factory_config(IHearingInfo)
class HearingInfo(BaseInheritInfo, Persistent, Contained):
    """Hearing info persistent class"""

    adapted_interface = IHearingInfo
    target_interface = IHearingTarget

    _targets = FieldProperty(IHearingInfo['targets'])
    targets = InheritedFieldProperty(IHearingInfo['targets'])
    _private = FieldProperty(IHearingInfo['private'])
    _private_groups = FieldProperty(IHearingInfo['private_groups'])
    _national_scope = FieldProperty(IHearingInfo['national_scope'])
    national_scope = InheritedFieldProperty(IHearingInfo['national_scope'])
    _forests = FieldProperty(IHearingInfo['forests'])
    forests = InheritedFieldProperty(IHearingInfo['forests'])
    _departments_forests = FieldProperty(IHearingInfo['departments_forests'])
    departments_forests = InheritedFieldProperty(IHearingInfo['departments_forests'])
    _domanials_only = FieldProperty(IHearingInfo['domanials_only'])
    domanials_only = InheritedFieldProperty(IHearingInfo['domanials_only'])
    _communals_only = FieldProperty(IHearingInfo['communals_only'])
    communals_only = InheritedFieldProperty(IHearingInfo['communals_only'])
    _cities = FieldProperty(IHearingInfo['cities'])
    cities = InheritedFieldProperty(IHearingInfo['cities'])
    _departments = FieldProperty(IHearingInfo['departments'])
    departments = InheritedFieldProperty(IHearingInfo['departments'])
    _countries = FieldProperty(IHearingInfo['countries'])
    countries = InheritedFieldProperty(IHearingInfo['countries'])
    _structures = FieldProperty(IHearingInfo['structures'])
    structures = InheritedFieldProperty(IHearingInfo['structures'])
    _departments_structures = FieldProperty(IHearingInfo['departments_structures'])
    departments_structures = InheritedFieldProperty(IHearingInfo['departments_structures'])
    _metropolitan_structures = FieldProperty(IHearingInfo['metropolitan_structures'])
    metropolitan_structures = InheritedFieldProperty(IHearingInfo['metropolitan_structures'])
    _corsican_structures = FieldProperty(IHearingInfo['corsican_structures'])
    corsican_structures = InheritedFieldProperty(IHearingInfo['corsican_structures'])
    _dom_structures = FieldProperty(IHearingInfo['dom_structures'])
    dom_structures = InheritedFieldProperty(IHearingInfo['dom_structures'])
    _source = FieldProperty(IHearingInfo['source'])
    source = InheritedFieldProperty(IHearingInfo['source'])
    source_folder = FieldProperty(IHearingInfo['source_folder'])
    _diffusion_sites = FieldProperty(IHearingInfo['diffusion_sites'])
    diffusion_sites = InheritedFieldProperty(IHearingInfo['diffusion_sites'])

    @property
    def targets_index(self):
        intids = get_utility(IIntIds)
        return [intids.register(value) for value in self.targets or ()]

    @property
    def private(self):
        return IHearingInfo(self.parent).private if self.inherit else self._private

    @private.setter
    def private(self, value):
        if not (self.can_inherit and self.inherit):
            self._private = value
            security = IProtectedObject(self.__parent__)
            if self._private:
                security.inherit_parent_security = False
                security.everyone_denied = {VIEW_PERMISSION}
                security.authenticated_denied = {VIEW_PERMISSION}
            else:
                security.inherit_parent_security = True
                security.everyone_denied = None
                security.authenticated_denied = None

    @property
    def private_groups(self):
        return IHearingInfo(self.parent).private_groups if self.inherit else self._private_groups

    @private_groups.setter
    def private_groups(self, value):
        if not (self.can_inherit and self.inherit):
            if value:
                self.private = True
            else:
                self.private = False
            self._private_groups = value
            parent = self.__parent__
            if IWfSharedContentRoles.providedBy(parent):
                if self.private:
                    parent.guests = value
                else:
                    parent.guests = None

    @volatile_property
    def get_forests_from_departments(self):
        return self.departments_forests or self.domanials_only or self.communals_only

    @volatile_property
    def forests_index(self):
        forests = set(self.forests or [])
        if self.departments and self.get_forests_from_departments:
            session = get_user_session(RDF_PARENT_SESSION)
            params = [Foret.departement.in_(self.selected_departments_codes)]
            if self.domanials_only:
                params.append(Foret.est_domaniale == True)
            elif self.communals_only:
                params.append(Foret.est_domaniale == False)
            results = set(*zip(*session.query(Foret.id_nat_frt)
                                       .filter(and_(*params))))
            if results:
                forests |= results
        return list(forests)

    def get_departments(self, not_if_forests=True):
        if not_if_forests and self.get_forests_from_departments:
            return
        departments = self.departments or ()
        if departments:
            for value in departments:
                if IRegion.providedBy(value):
                    for dept in value.departments or ():
                        yield dept
                else:
                    yield value
        else:
            departments = get_utility(IDepartmentTable)
            forests = self.forests_index
            if forests:
                rdf_session = get_user_session(RDF_PARENT_SESSION)
                yield from (departments.get(dep)
                            for dep in set(*zip(*rdf_session.query(Foret.departement)
                                                            .filter(Foret.id_nat_frt.in_(forests)))))
            cities = self.cities
            if cities:
                insee_session = get_user_session(INSEE_PARENT_SESSION)
                yield from (departments.get(dep)
                            for dep in set(*zip(*insee_session.query(Commune.dep)
                                                              .filter(Commune.code.in_(cities)))))

    @volatile_property
    def selected_departments_codes(self):
        return [dept.insee_code for dept in self.get_departments(False)]

    @volatile_property
    def departments_index(self):
        return [dept.insee_code for dept in self.get_departments()]

    @volatile_property
    def countries_index(self):
        return [value.insee_code for value in self.countries or ()]

    @volatile_property
    def structures_index(self):
        structures = set(self.structures or [])
        params = []
        if self.departments and self.departments_structures:
            codes = set()
            units_table = get_utility(IUnitTable)
            for dept in self.get_departments():
                for unit in filter(lambda x: dept in (x.departments or ()), units_table.values()):
                    codes.add(unit.internal_code)
                    if unit.agency is not None:
                        codes.add(unit.agency.internal_code)
            if codes:
                params.append(Structure.code_sign.in_(codes))
        if self.metropolitan_structures:
            params.append(or_(*METROPOLITAN_CODE_FILTER))
        if self.corsican_structures:
            params.append(or_(*CORSICAN_CODE_FILTER))
        if self.dom_structures:
            params.append(or_(*DOM_CODE_FILTER))
        if params:
            session = get_user_session(STRUCT_PARENT_SESSION)
            results = set(*zip(*session.query(Structure.code)
                                       .filter(and_(and_(*Structure.get_default_params()),
                                                    or_(*params)))))
            if results:
                structures |= results
        return list(structures)

    @volatile_property
    def source_index(self):
        site = self.get_source_site()
        if site is not None:
            intids = get_utility(IIntIds)
            return intids.register(site)

    def get_source_site(self):
        if self.source is not None:
            return query_utility(ISharedSite, name=self.source)

    def get_source_folder(self):
        if self.source_folder is not None:
            return get_reference_target(self.source_folder)

    @volatile_property
    def diffusion_sites_index(self):
        intids = get_utility(IIntIds)
        return unique([
            intids.register(site)
            for site in self.get_diffusion_sites()
            if site is not None
        ])

    def get_diffusion_sites(self, names_only=False):
        folder = self.get_source_folder()
        if folder is not None:
            site = get_parent(folder, ISharedSite)
            if site is not None:
                yield site
        else:
            site = self.get_source_site()
            if site is not None:
                yield site
        for value in self.diffusion_sites or ():
            yield value if names_only else query_utility(ISharedSite, name=value)

    def is_matching_location(self, other):
        if self.national_scope:
            return True
        other_info = ILocationInfo(other, None)
        if other_info is None:
            return False
        return (set(self.forests_index) & set(other_info.forests_index)) or \
               (set(self.cities or ()) & set(other_info.cities or ())) or \
               (set(self.departments_index) & set(other_info.departments_index)) or \
               (set(self.countries_index) & set(other_info.countries_index))


@adapter_config(context=IHearingTarget, provides=IHearingInfo)
def hearing_info_factory(target):
    """Hearing info factory"""
    return get_annotation_adapter(target, HEARING_INFO_KEY, IHearingInfo,
                                  name='++hearing++')


@adapter_config(context=IHearingInfo, provides=IFormContextPermissionChecker)
class HearingInfoPermissionChecker(ContextAdapter):
    """Hearing info checker"""

    @property
    def edit_permission(self):
        content = get_parent(self.context, IHearingTarget)
        return IFormContextPermissionChecker(content).edit_permission


@adapter_config(name='hearing', context=IHearingTarget, provides=IDocumentIndexInfo)
def hearing_target_index_info(content):
    """Hearing index info"""
    hearing = IHearingInfo(content)
    return {
        'hearing': {
            'targets': hearing.targets_index,
            'national_scope': hearing.national_scope,
            'forests': hearing.forests_index,
            'cities': hearing.cities or [],
            'departments': hearing.departments_index,
            'countries': hearing.countries_index,
            'structures': hearing.structures_index,
            'source_site': hearing.source_index,
            'diffusion_sites': hearing.diffusion_sites_index
        }
    }


@adapter_config(name='hearing', context=IHearingTarget, provides=IContentChecker)
class HearingContentChecker(BaseContentChecker):
    """Hearing content checker"""

    label = _("Hearing")
    weight = 95

    def inner_check(self, request):
        output = []
        translate = request.localizer.translate
        hearing = IHearingInfo(self.context)
        if not hearing.targets:
            output.append(translate(MISSING_VALUE).format(
                field=translate(IHearingInfo['targets'].title)))
        return output


@subscriber(IObjectModifiedEvent, context_selector=IHearingTarget)
def handle_modified_hearing(event):
    """Delete volatile attributes when hearing is modified"""
    hearing = IHearingInfo(event.object)
    for attributes in event.descriptions:
        if attributes.interface is IHearingLocationsInfo:
            del hearing.get_forests_from_departments
            del hearing.forests_index
            del hearing.selected_departments_codes
            del hearing.departments_index
            del hearing.countries_index
        elif attributes.interface is IHearingOrganizationInfo:
            del hearing.structures_index
        elif attributes.interface is IHearingDiffusionSitesInfo:
            del hearing.source_index
            del hearing.diffusion_sites_index


@subscriber(IObjectMovedEvent, context_selector=IHearingTarget)
def handle_moved_hearing(event):
    """Remove volatile 'parent' property when hearing target is moved"""
    hearing = IHearingInfo(event.object)
    del hearing.parent
