#
# Copyright (c) 2008-2018 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

from persistent import Persistent
from zope.container.contained import Contained
from zope.interface import implementer, Interface
from zope.schema.fieldproperty import FieldProperty

from onf_website.shared.casestudy import IWfCaseStudy
from onf_website.shared.service import IWfService
from onf_website.shared.view.skin.interfaces import IONFViewItemsPortletBaseRendererSettings, \
    IONFViewItemsPortletCarouselRendererSettings, IONFViewItemsPortletHalfWidthRendererSettings, \
    IONFViewItemsPortletNewsRendererSettings, IONFViewItemsPortletNewsTightRendererSettings, \
    IONFViewItemsPortletSummaryRendererSettings, IONFViewItemsPortletActivitiesRendererSettings, \
    IONFViewItemsPortletAssociatedRendererSettings, IONFViewItemsPortletTexturedRendererSettings, \
    IONFViewItemsPortletFramedRendererSettings, IONFViewItemsPortletPictogramsMarginRendererSettings
from onf_website.skin.public import owl_carousel_init, owl_navigation
from pyams_content.features.search import ISearchFolder
from pyams_content.features.search.interfaces import IContextUserSearchSettings
from pyams_content.shared.common.interfaces.types import IWfTypedSharedContent
from pyams_content.shared.view.portlet.interfaces import IViewItemsPortletSettings
from pyams_default_theme.features.search.portlet import HIDDEN_HEADER_DISPLAY, \
    START_HEADER_DISPLAY
from pyams_default_theme.shared.view.portlet import BaseViewItemsPortletRenderer
from pyams_i18n.interfaces import II18n
from pyams_i18n.schema import I18nTextField
from pyams_portal.interfaces import IPortalContext, IPortletRenderer
from pyams_sequence.reference import get_reference_target
from pyams_skin.layer import IPyAMSLayer, IPyAMSUserLayer
from pyams_template.template import template_config
from pyams_utils.adapter import adapter_config
from pyams_utils.factory import factory_config
from pyams_utils.request import copy_request
from pyams_utils.text import get_text_start


__docformat__ = 'restructuredtext'

from onf_website import _


@implementer(IONFViewItemsPortletBaseRendererSettings)
class ONFViewItemsPortletBaseRendererSettings(Persistent, Contained):
    """View items portlet base renderer settings"""

    list_reference = FieldProperty(IONFViewItemsPortletBaseRendererSettings['list_reference'])
    link_label = FieldProperty(IONFViewItemsPortletBaseRendererSettings['link_label'])
    include_context_settings = FieldProperty(
        IONFViewItemsPortletNewsRendererSettings['include_context_settings'])

    def get_target(self):
        return get_reference_target(self.list_reference)


class ONFBaseViewItemsPortletRenderer(BaseViewItemsPortletRenderer):
    """ONF base view items portlet renderer"""

    def get_context_settings(self):
        settings = self.renderer_settings
        if not (IONFViewItemsPortletBaseRendererSettings.providedBy(settings) and
                settings.include_context_settings):
            return None
        for name, adapter in self.request.registry.getAdapters((self.context, self.request,),
                                                               IContextUserSearchSettings):
            yield from adapter.get_settings()

    def get_pictogram(self, item):
        content = IWfTypedSharedContent(item, None)
        if content is not None:
            datatype = content.get_data_type()
            if datatype is not None:
                pictogram = datatype.get_pictogram()
                if pictogram is not None:
                    return II18n(pictogram).query_attribute('image', request=self.request)
        return None


#
# View items list renderer
#

@factory_config(provided=IONFViewItemsPortletNewsRendererSettings)
class ONFViewItemsPortletNewsRendererSettings(ONFViewItemsPortletBaseRendererSettings):
    """View items portlet renderer settings"""

    header_results = FieldProperty(IONFViewItemsPortletNewsRendererSettings['header_results'])
    highlight_results = FieldProperty(IONFViewItemsPortletNewsRendererSettings['highlight_results'])
    main_css_class = FieldProperty(IONFViewItemsPortletNewsRendererSettings['main_css_class'])
    inner_css_class = FieldProperty(IONFViewItemsPortletNewsRendererSettings['inner_css_class'])


@adapter_config(name='ONF::view.actus',
                context=(IPortalContext, IPyAMSLayer, Interface, IViewItemsPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/cards.pt', layer=IPyAMSUserLayer)
class ONFViewItemsPortletNewsRenderer(ONFBaseViewItemsPortletRenderer):
    """View items portlet renderer"""

    label = _("ONF: small cards")
    weight = 11

    settings_interface = IONFViewItemsPortletNewsRendererSettings
    resources = (owl_carousel_init, owl_navigation)


#
# View items list tight renderer
#

@factory_config(provided=IONFViewItemsPortletNewsTightRendererSettings)
class ONFViewItemsPortletNewsTightRendererSettings(ONFViewItemsPortletNewsRendererSettings):
    """View items portlet tight renderer settings"""

    main_css_class = FieldProperty(
        IONFViewItemsPortletNewsTightRendererSettings['main_css_class'])
    inner_css_class = FieldProperty(
        IONFViewItemsPortletNewsTightRendererSettings['inner_css_class'])


@adapter_config(name='ONF::view.actus-tight',
                context=(IPortalContext, IPyAMSLayer, Interface, IViewItemsPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/cards-noheading.pt', layer=IPyAMSUserLayer)
class ONFTightViewItemsPortletNewsRenderer(ONFBaseViewItemsPortletRenderer):
    """View items portlet tight renderer"""

    label = _("ONF: small cards (without heading card)")
    weight = 12

    settings_interface = IONFViewItemsPortletNewsTightRendererSettings


#
# Related contents renderer
#

@adapter_config(name='ONF::view.related',
                context=(IPortalContext, IPyAMSLayer, Interface, IViewItemsPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/related.pt', layer=IPyAMSUserLayer)
class ONFViewRelatedItemsPortletRenderer(BaseViewItemsPortletRenderer):
    """Related items portlet renderer"""

    label = _("ONF: related contents")
    weight = 13

    resources = (owl_carousel_init, owl_navigation)


#
# Partners logos renderer
#

@adapter_config(name='ONF::view.partners',
                context=(IPortalContext, IPyAMSLayer, Interface, IViewItemsPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/partners-logos.pt', layer=IPyAMSUserLayer)
class ONFViewPartnersLogosPortletRenderer(BaseViewItemsPortletRenderer):
    """Partners logos portlet renderer"""

    label = _("ONF: services partners logos")
    weight = 14

    @property
    def logos(self):
        settings = self.settings
        for content in settings.get_items(self.request):
            if not (IWfService.providedBy(content) or IWfCaseStudy.providedBy(content)):
                continue
            for logo in content.get_logos():
                if not logo.image:
                    continue
                yield content, logo


#
# Horizontal carousel renderer
#

@factory_config(provided=IONFViewItemsPortletCarouselRendererSettings)
class ONFViewItemsPortletCarouselRendererSettings(Persistent, Contained):
    """View items portlet renderer settings"""

    header = FieldProperty(IONFViewItemsPortletCarouselRendererSettings['header'])
    button_label = FieldProperty(IONFViewItemsPortletCarouselRendererSettings['button_label'])
    button_target = FieldProperty(IONFViewItemsPortletCarouselRendererSettings['button_target'])

    def get_target(self):
        if self.button_target is not None:
            return get_reference_target(self.button_target)


@adapter_config(name='ONF::view.carousel',
                context=(IPortalContext, IPyAMSLayer, Interface, IViewItemsPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/carousel.pt', layer=IPyAMSUserLayer)
class ONFViewCarouselPortletRenderer(ONFBaseViewItemsPortletRenderer):
    """View carousel portlet renderer"""

    label = _("ONF: horizontal carousel")
    weight = 15

    settings_interface = IONFViewItemsPortletCarouselRendererSettings


#
# Summary-like view items renderer
#

@factory_config(provided=IONFViewItemsPortletSummaryRendererSettings)
class ONFViewItemsPortletSummaryRendererSettings(Persistent, Contained):
    """View items portlet summary renderer settings"""

    header_display_mode = FieldProperty(
        IONFViewItemsPortletSummaryRendererSettings['header_display_mode'])
    start_length = FieldProperty(IONFViewItemsPortletSummaryRendererSettings['start_length'])
    button_label = FieldProperty(IONFViewItemsPortletSummaryRendererSettings['button_label'])


@adapter_config(name='ONF::view.summary',
                context=(IPortalContext, IPyAMSLayer, Interface, IViewItemsPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/summary.pt', layer=IPyAMSUserLayer)
class ONFViewItemsPortletSummaryRenderer(BaseViewItemsPortletRenderer):
    """View items portlet summary renderer"""

    label = _("ONF: summary")
    weight = 16

    settings_interface = IONFViewItemsPortletSummaryRendererSettings

    def get_header(self, item):
        settings = self.renderer_settings
        display_mode = settings.header_display_mode
        if display_mode == HIDDEN_HEADER_DISPLAY:
            return ''
        header = II18n(item).query_attribute('header', request=self.request)
        if display_mode == START_HEADER_DISPLAY:
            header = get_text_start(header, settings.start_length)
        return header


#
# Resources view renderer
#

@adapter_config(name='ONF::view.resources',
                context=(IPortalContext, IPyAMSLayer, Interface, IViewItemsPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/resources.pt', layer=IPyAMSUserLayer)
class ONFViewItemsPortletResourcesRenderer(ONFBaseViewItemsPortletRenderer):
    """View items portlet resources renderer"""

    label = _("ONF: internal resources")
    weight = 17


#
#
#

@factory_config(provided=IONFViewItemsPortletHalfWidthRendererSettings)
class ONFViewItemsPortletHalfWidthRendererSettings(ONFViewItemsPortletBaseRendererSettings):
    """Onf view items half-width renderer settings"""

    empty_message = FieldProperty(
        IONFViewItemsPortletHalfWidthRendererSettings['empty_message'])
    empty_link_label = FieldProperty(
        IONFViewItemsPortletHalfWidthRendererSettings['empty_link_label'])


@adapter_config(name='ONF::view.actus-half-width',
                context=(IPortalContext, IPyAMSLayer, Interface, IViewItemsPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/cards-half-width.pt', layer=IPyAMSUserLayer)
class ONFHalfWidthViewItemsPortletNewsRenderer(ONFBaseViewItemsPortletRenderer):
    """Half-width view items portlet renderer"""

    label = _("ONF: half-width cards with carousel (without heading card)")
    weight = 18

    settings_interface = IONFViewItemsPortletHalfWidthRendererSettings
    resources = (owl_carousel_init, owl_navigation)

    def has_target_results(self):
        target = self.renderer_settings.get_target()
        if (target is None) or (not ISearchFolder.providedBy(target)):
            return False
        request = copy_request(self.request)
        settings = self.get_context_settings()
        if settings is not None:
            for key, value in settings:
                request.GET[key] = value
        results, count, aggs = target.get_results(target, get_count=True, limit=1, length=1,
                                                  request=request)
        return count > 0


#
# Activities view renderer
#

@factory_config(provided=IONFViewItemsPortletActivitiesRendererSettings)
class ONFViewItemsPortletActivitiesRendererSettings(ONFViewItemsPortletBaseRendererSettings):
    """Onf view items activities renderer settings"""


@adapter_config(name='ONF::view.activities',
                context=(IPortalContext, IPyAMSLayer, Interface, IViewItemsPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/activities.pt', layer=IPyAMSUserLayer)
class ONFViewItemsPortletActivitiesRenderer(ONFBaseViewItemsPortletRenderer):
    """View items portlet activities renderer"""

    label = _("ONF: activities")
    weight = 19

    settings_interface = IONFViewItemsPortletActivitiesRendererSettings


#
# Associated contents view renderer
#

@factory_config(provided=IONFViewItemsPortletAssociatedRendererSettings)
class ONFViewItemsPortletAssociatedRendererSettings(ONFViewItemsPortletBaseRendererSettings):
    """Onf view items associated renderer settings"""


@adapter_config(name='ONF::view.associated',
                context=(IPortalContext, IPyAMSLayer, Interface, IViewItemsPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/associated.pt', layer=IPyAMSUserLayer)
class ONFViewItemsPortletAssociatedRenderer(ONFBaseViewItemsPortletRenderer):
    """View items portlet associated links renderer"""

    label = _("ONF: associated links")
    weight = 20

    settings_interface = IONFViewItemsPortletAssociatedRendererSettings


#
# Textured view renderer
#

@factory_config(provided=IONFViewItemsPortletTexturedRendererSettings)
class ONFViewItemsPortletTexturedRendererSettings(ONFViewItemsPortletBaseRendererSettings):
    """Onf view items textured renderer settings"""

    intro_text = FieldProperty(IONFViewItemsPortletTexturedRendererSettings['intro_text'])


@adapter_config(name='ONF::view.textured',
                context=(IPortalContext, IPyAMSLayer, Interface, IViewItemsPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/textured.pt', layer=IPyAMSUserLayer)
class ONFViewItemsPortletTexturedRenderer(ONFBaseViewItemsPortletRenderer):
    """View items portlet textured renderer"""

    label = _("ONF: textured card")
    weight = 21

    settings_interface = IONFViewItemsPortletTexturedRendererSettings


#
# Framed view renderer
#

@factory_config(provided=IONFViewItemsPortletFramedRendererSettings)
class ONFViewItemsPortletFramedRendererSettings(ONFViewItemsPortletBaseRendererSettings):
    """Onf view items framed renderer settings"""

    intro_text = FieldProperty(IONFViewItemsPortletFramedRendererSettings['intro_text'])
    frame_title = FieldProperty(IONFViewItemsPortletFramedRendererSettings['frame_title'])
    frame_body = FieldProperty(IONFViewItemsPortletFramedRendererSettings['frame_body'])


@adapter_config(name='ONF::view.framed',
                context=(IPortalContext, IPyAMSLayer, Interface, IViewItemsPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/framed.pt', layer=IPyAMSUserLayer)
class ONFViewItemsPortletFramedRenderer(ONFBaseViewItemsPortletRenderer):
    """View items portlet framed renderer"""

    label = _("ONF: framed card with introduction text")
    weight = 22

    settings_interface = IONFViewItemsPortletFramedRendererSettings


#
# Pictograms with margins view renderer
#

@factory_config(provided=IONFViewItemsPortletPictogramsMarginRendererSettings)
class ONFViewItemsPortletPictogramsMarginRendererSettings(ONFViewItemsPortletBaseRendererSettings):
    """Onf view items with square pictograms and margin renderer settings"""


@adapter_config(name='ONF::view.pictograms-square-margin',
                context=(IPortalContext, IPyAMSLayer, Interface, IViewItemsPortletSettings),
                provides=IPortletRenderer)
@template_config(template='templates/pictograms-margin.pt', layer=IPyAMSUserLayer)
class ONFViewItemsPortletPictogramsMarginRenderer(ONFBaseViewItemsPortletRenderer):
    """View items with square pictograms and margin portlet renderer"""

    label = _("ONF: view items with square pictograms and margin")
    weight = 23

    settings_interface = IONFViewItemsPortletPictogramsMarginRendererSettings
