
"""PyAMS_*** module

"""

from hypatia.catalog import CatalogQuery
from hypatia.interfaces import ICatalog
from hypatia.query import Any, Eq
from pyramid.events import subscriber
from z3c.form.field import Fields
from z3c.form.interfaces import IDataExtractedEvent
from zope.interface import Interface, Invalid

from onf_website.shared.planning import IWfPlanning, PLANNING_CONTENT_TYPE, WfPlanning
from onf_website.shared.planning.interfaces import IPlanningLinkPortletSettings
from onf_website.shared.planning.interfaces import IPlanningManager
from pyams_content.interfaces import CREATE_CONTENT_PERMISSION, MANAGE_CONTENT_PERMISSION
from pyams_content.shared.common.zmi import SharedContentAJAXAddForm, SharedContentAddForm
from pyams_content.shared.common.zmi.properties import SharedContentPropertiesEditForm
from pyams_form.form import AJAXEditForm
from pyams_form.form import ajax_config
from pyams_form.interfaces.form import IInnerTabForm
from pyams_i18n.interfaces import II18n
from pyams_i18n.widget import I18nSEOTextLineFieldWidget
from pyams_pagelet.interfaces import IPagelet
from pyams_pagelet.pagelet import pagelet_config
from pyams_portal.interfaces import IPortletPreviewer
from pyams_portal.portlet import PortletPreviewer
from pyams_portal.zmi.portlet import PortletSettingsEditor, PortletSettingsPropertiesEditor
from pyams_skin.interfaces import IContentTitle
from pyams_skin.interfaces.viewlet import IMenuHeader, IWidgetTitleViewletManager
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.viewlet.toolbar import ToolbarAction
from pyams_template.template import template_config
from pyams_utils.adapter import ContextRequestAdapter, ContextRequestViewAdapter, NullAdapter, \
    adapter_config
from pyams_utils.interfaces import FORBIDDEN_PERMISSION, VIEW_SYSTEM_PERMISSION
from pyams_utils.registry import get_utility
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.interfaces.menu import IContentManagementMenu, IPropertiesMenu
from pyams_zmi.layer import IAdminLayer


__docformat__ = 'restructuredtext'

from onf_website import _


@adapter_config(context=(IWfPlanning, IContentManagementMenu), provides=IMenuHeader)
class PlanningContentMenuHeader(ContextRequestAdapter):
    """Planning content menu header adapter"""

    header = _("This planning")


@adapter_config(context=(IWfPlanning, IPyAMSLayer, Interface), provides=IContentTitle)
class PlanningTitleAdapter(ContextRequestViewAdapter):
    """Planning title adapter"""

    @property
    def title(self):
        translate = self.request.localizer.translate
        return translate(_("Planning « {title} »")).format(
            title=II18n(self.context).query_attribute('title', request=self.request))


@viewlet_config(name='add-shared-content.action',
                context=IPlanningManager, layer=IAdminLayer, view=Interface,
                manager=IWidgetTitleViewletManager, weight=1,
                permission=CREATE_CONTENT_PERMISSION)
class PlanningAddAction(ToolbarAction):
    """Planning adding action"""

    label = _("Add planning")
    label_css_class = 'fa fa-fw fa-plus'
    url = 'add-shared-content.html'
    modal_target = True


@pagelet_config(name='add-shared-content.html',
                context=IPlanningManager, layer=IPyAMSLayer,
                permission=CREATE_CONTENT_PERMISSION)
@ajax_config(name='add-shared-content.json',
             context=IPlanningManager, request_type=IPyAMSLayer,
             permission=CREATE_CONTENT_PERMISSION, base=SharedContentAJAXAddForm)
class PlanningAddForm(SharedContentAddForm):
    """Planning add form"""

    legend = _("Add planning")

    fields = Fields(IWfPlanning).select('title', 'planning_ids', 'notepad')
    fields['title'].widgetFactory = I18nSEOTextLineFieldWidget


@subscriber(IDataExtractedEvent, form_selector=PlanningAddForm)
def handle_planning_add_form_data(event):
    """Check for previous planning with same planning IDs"""
    planning_ids = event.data.get('planning_ids')
    catalog = get_utility(ICatalog)
    params = Eq(catalog['content_type'], PLANNING_CONTENT_TYPE) & \
        Any(catalog['plannings'], planning_ids)
    count, results = CatalogQuery(catalog).query(params)
    if count > 0:
        event.form.widgets.errors += (Invalid(_("A content already exists for this planning!")),)


@pagelet_config(name='properties.html',
                context=IWfPlanning, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@ajax_config(name='properties.json',
             context=IWfPlanning, request_type=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION)
class PlanningPropertiesEditForm(SharedContentPropertiesEditForm):
    """Planning properties edit form"""

    interface = IWfPlanning
    fieldnames = ('title', 'short_name', 'content_url', 'planning_ids',
                  'header', 'description', 'notepad')


#
# Custom planning menus
#

@viewlet_config(name='location.menu',
                context=WfPlanning, layer=IAdminLayer,
                manager=IPropertiesMenu, weight=310,
                permission=FORBIDDEN_PERMISSION)
class PlanningLocationMenu(NullAdapter):
    """Hidden planning location menu"""


@viewlet_config(name='hearing.menu',
                context=WfPlanning, layer=IAdminLayer,
                manager=IPropertiesMenu, weight=320,
                permission=FORBIDDEN_PERMISSION)
class PlanningHearingMenu(NullAdapter):
    """Hidden planning hearing menu"""



#
# Shared content planning link portlet views
#

@pagelet_config(name='properties.html', 
                context=IPlanningLinkPortletSettings, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
class PlanningLinkPortletSettingsEditor(PortletSettingsEditor):
    """Planning Link portlet settings editor"""

    settings = IPlanningLinkPortletSettings

@adapter_config(name='properties', 
                context=(Interface, IPyAMSLayer, PlanningLinkPortletSettingsEditor),
                provides=IInnerTabForm)
class PlanningLinkPortletSettingsPropertiesEditor(PortletSettingsPropertiesEditor):
    """Planning Link portlet settings properties editor"""


@adapter_config(name='properties.json', 
                context=(IPlanningLinkPortletSettings, IPyAMSLayer),
                provides=IPagelet)
class PlanningLinkPortletAJAXEditor(AJAXEditForm, PlanningLinkPortletSettingsEditor):
    """Planning Link portlet settings editor, JSON renderer"""


@adapter_config(context=(Interface, IPyAMSLayer, Interface, IPlanningLinkPortletSettings),
                provides=IPortletPreviewer)
@template_config(template='templates/planninglink-preview.pt', layer=IPyAMSLayer)
class PlanningLinkPortletPreviewer(PortletPreviewer):
    """Planning Link portlet previewer"""
