
"""PyAMS_*** module

"""

from pyramid.encode import urlencode, url_quote
from pyramid.url import QUERY_SAFE
from zope.interface import Interface

from onf_website.shared.planning import FakePlanning, IPlanningManager
from pyams_portal.interfaces import IPortalPage, IPortalTemplate
from pyams_portal.page import PortalPage
from pyams_skin.interfaces.metas import IHTMLContentMetas
from pyams_skin.layer import IPyAMSUserLayer
from pyams_skin.metas import HTMLTagMeta
from pyams_utils.adapter import ContextAdapter, ContextRequestAdapter, ContextRequestViewAdapter, \
    adapter_config
from pyams_utils.interfaces.url import ICanonicalURL, IRelativeURL
from pyams_utils.registry import query_utility
from pyams_utils.unicode import translate_string
from pyams_utils.url import absolute_url


__docformat__ = 'restructuredtext'


@adapter_config(context=FakePlanning, provides=IPortalPage)
class FakePlanningPortalPage(ContextAdapter, PortalPage):
    """Fake planning portal page"""

    can_inherit = False
    use_local_template = True

    @property
    def local_template(self):
        manager = query_utility(IPlanningManager)
        if manager is None:
            return None
        refs = self.context.planning_refs
        if not refs:
            return None
        planning, info = refs[0]
        if (planning is None) or (planning.categorie_proprietaire is None):
            return None
        template = manager.templates.get(planning.categorie_proprietaire)
        if template is not None:
            return query_utility(IPortalTemplate, name=template)
        return IPortalPage(manager).template


@adapter_config(name='title',
                context=(FakePlanning, IPyAMSUserLayer, Interface),
                provides=IHTMLContentMetas)
class SharedContentTitleMetasAdapter(ContextRequestViewAdapter):
    """Title metas adapter"""

    order = 1

    def get_metas(self):
        """Get HTML metas headers"""
        yield HTMLTagMeta('title', self.context.title)


@adapter_config(context=(FakePlanning, IPyAMSUserLayer),
                provides=ICanonicalURL)
class FakePlanningCanonicalUrl(ContextRequestAdapter):
    """Fake planning canonical URL adapter"""

    def get_url(self, view_name=None, query=None):
        context = None
        manager = query_utility(IPlanningManager)
        if manager is not None:
            context = manager.get_default_source()
        if context is None:
            return '{}/+a/{}::{}.html'.format(absolute_url(self.request.root, self.request),
                                              ','.join(self.context.planning_ids),
                                              translate_string(self.context.title, spaces='-'))
        return '{}/++amgt++{}::{}.html'.format(absolute_url(context, self.request),
                                               ','.join(self.context.planning_ids),
                                               translate_string(self.context.title, spaces='-'))


@adapter_config(context=(FakePlanning, IPyAMSUserLayer),
                provides=IRelativeURL)
class FakePlanningRelativeUrl(ContextRequestAdapter):
    """Fake planning relative URL adapter"""

    def get_url(self, display_context=None, view_name=None, query=None):
        if query:
            if isinstance(query, str):
                query = url_quote(query, QUERY_SAFE)
            else:
                query = urlencode(query, doseq=True)
        if display_context is None:
            return '{}/+a/{}::{}.html'.format(absolute_url(self.request.root, self.request),
                                              ','.join(self.context.planning_ids),
                                              translate_string(self.context.title, spaces='-'))
        return absolute_url(display_context, self.request,
                            '++amgt++{}::{}.html'.format(
                                ','.join(self.context.planning_ids),
                                translate_string(self.context.title, spaces='-')))
