#
# Copyright (c) 2015-2022 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

__docformat__ = 'restructuredtext'

from hypatia.catalog import CatalogQuery
from hypatia.interfaces import ICatalog
from hypatia.query import And, Any, Eq
from sqlalchemy import and_, func, or_
from zope.interface import implementer

from onf_website import _
from onf_website.reference.forest.model.foret import Foret, ForetSurCommune, InformationForet
from onf_website.reference.planning import IPlanningTable
from onf_website.reference.planning.model import PARENT_SESSION as RDF_SESSION
from onf_website.reference.planning.model.planning import InformationPlanning, \
    PlanningData as PlanningModel, PlanningForet
from onf_website.shared.planning import FakePlanning, PLANNING_CONTENT_TYPE
from onf_website.shared.planning.interfaces import IPlanningSearchFolder
from pyams_alchemy.engine import get_user_session
from pyams_catalog.query import CatalogResultSet
from pyams_content.component.illustration import IIllustrationTarget, ILinkIllustrationTarget
from pyams_content.features.preview.interfaces import IPreviewTarget
from pyams_content.features.search import ISearchFolderRoles
from pyams_content.shared.view import WfView
from pyams_content.workflow import VISIBLE_STATES
from pyams_portal.interfaces import IPortalContext
from pyams_utils.list import unique_iter
from pyams_utils.registry import get_utility, query_utility
from pyams_utils.unicode import translate_string


def planning_from_id(planning_id, contents):
    """Get planning from ID"""

    content = next((
        item
        for item in contents
        if (planning_id in item.planning_ids)
    ), None)
    if content:
        return content

    fake_planning = FakePlanning()
    fake_planning.planning_ids = [planning_id]
    return fake_planning


def is_latest_planning(planning, plannings):
    """Check if given planning is the last one provided list"""

    return next((item for item in plannings if (
        (item.date_debut_applicabilite > planning.date_debut_applicabilite) and
        (item.id_nat_amgt != planning.id_nat_amgt)
    )), None) is None


@implementer(IPlanningSearchFolder, ISearchFolderRoles, IIllustrationTarget,
             ILinkIllustrationTarget, IPortalContext, IPreviewTarget)
class PlanningSearchFolder(WfView):
    """Planning search folder"""

    content_name = _("Planning search folder")

    handle_short_name = True
    handle_header = True
    handle_description = True

    sequence_name = ''
    sequence_prefix = ''

    @staticmethod
    def is_deletable():
        return True

    def get_results(self, context, sort_index=None, reverse=None, limit=None,
                    start=0, length=None, ignore_cache=False, get_count=False, request=None,
                    aggregates=None, settings=None, **kwargs):

        session = get_user_session(RDF_SESSION)

        table = query_utility(IPlanningTable)

        user_search = request.params.get('user_search', '').strip().upper()
        if not user_search:
            items = []
            total_count = 0
            aggregations = {}
            return unique_iter(items), total_count, aggregations

        translated_search = translate_string(user_search, force_lower=False)
        latest_by_forest = session.query(
            PlanningForet.id_nat_foret,
            func.max(PlanningModel.date_debut_applicabilite).label('date_debut_applicabilite')
        ) \
            .outerjoin(PlanningForet,
                       PlanningForet.id_nat_amgt == PlanningModel.id_nat_amgt) \
            .outerjoin(Foret,
                       Foret.id_nat_frt == PlanningForet.id_nat_foret) \
            .outerjoin(InformationForet,
                       InformationForet.id_nat_frt == Foret.id_nat_frt) \
            .outerjoin(ForetSurCommune,
                       ForetSurCommune.id_nat_frt == Foret.id_nat_frt) \
            .outerjoin(InformationPlanning,
                       InformationPlanning.id_nat_amgt == PlanningModel.id_nat_amgt) \
            .filter(and_(
                PlanningModel.code_categorie_proprietaire.in_(table.visible_owners_ids),
                or_(
                    PlanningModel.libelle_recherche.contains(translated_search),
                    Foret.libelle_recherche.contains(translated_search),
                    ForetSurCommune.libelle_insee.contains(translated_search),
                    func.upper(InformationPlanning.libelle).contains(user_search),
                    func.upper(InformationPlanning.libelle).contains(translated_search),
                    func.upper(InformationForet.libelle).contains(user_search),
                    func.upper(InformationForet.libelle).contains(translated_search)
                )
            )).group_by(
                PlanningForet.id_nat_foret
            ).subquery()

        query = session.query(PlanningModel) \
            .outerjoin(PlanningForet,
                       PlanningForet.id_nat_amgt == PlanningModel.id_nat_amgt) \
            .outerjoin(InformationPlanning,
                       InformationPlanning.id_nat_amgt == PlanningModel.id_nat_amgt) \
            .join(
                latest_by_forest,
                and_(
                    PlanningModel.date_debut_applicabilite == latest_by_forest.c.date_debut_applicabilite,
                    PlanningForet.id_nat_foret == latest_by_forest.c.id_nat_foret
                )
            ) \
            .filter(and_(PlanningModel.code_categorie_proprietaire.in_(table.visible_owners_ids),
                         or_(InformationPlanning.visible == True,
                             InformationPlanning.visible == None))) \
            .order_by(PlanningModel.nom_usage_amenagement) \
            .distinct()

        total_count = query.count()

        query = query.limit(self.limit) if self.limit else query.limit(length).offset(start)

        items = query.all()

        planning_ids = [item.id_nat_amgt for item in items]

        catalog = get_utility(ICatalog)
        query = CatalogQuery(catalog).query(
            And(Eq(catalog['content_type'], PLANNING_CONTENT_TYPE),
                Any(catalog['plannings'], planning_ids),
                Any(catalog['workflow_state'], VISIBLE_STATES)),
            limit=limit)

        contents = CatalogResultSet(query)

        items = [
            planning_from_id(id, contents)
            for id in planning_ids
        ]
        aggregations = {}

        return unique_iter(items), total_count, aggregations
