#
# Copyright (c) 2015-2022 Thierry Florac <tflorac AT ulthar.net>
# All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.1 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE.
#

"""PyAMS_*** module

"""

from z3c.form.field import Fields
from zope.intid.interfaces import IIntIds

from onf_website.shared.hunting.interfaces import IHuntingCalendarSearchFolder
from onf_website.shared.hunting.search import HuntingCalendarSearchFolder
from pyams_content.interfaces import MANAGE_CONTENT_PERMISSION, MANAGE_SITE_PERMISSION
from pyams_content.shared.common.zmi.properties import SharedContentPropertiesEditForm
from pyams_content.shared.site.interfaces import ISiteContainer
from pyams_content.shared.site.zmi import SiteManagerFoldersSelectorFieldWidget
from pyams_content.shared.site.zmi.folder import ISiteFolderAddFormFields
from pyams_content.zmi.interfaces import ISiteTreeTable
from pyams_form.form import AJAXAddForm, ajax_config
from pyams_form.interfaces.form import IInnerSubForm
from pyams_i18n.interfaces import II18n, INegotiator
from pyams_pagelet.pagelet import pagelet_config
from pyams_skin.interfaces.viewlet import IToolbarAddingMenu
from pyams_skin.layer import IPyAMSLayer
from pyams_skin.viewlet.toolbar import ToolbarMenuItem
from pyams_utils.adapter import NullAdapter, adapter_config
from pyams_utils.interfaces import VIEW_SYSTEM_PERMISSION
from pyams_utils.registry import get_utility
from pyams_utils.unicode import translate_string
from pyams_utils.url import absolute_url
from pyams_viewlet.viewlet import viewlet_config
from pyams_zmi.form import AdminDialogAddForm
from pyams_zmi.interfaces.menu import IPropertiesMenu
from pyams_zmi.layer import IAdminLayer

__docformat__ = 'restructuredtext'

from onf_website import _


@viewlet_config(name='add-hunting-calendar-search-folder.menu',
                context=ISiteContainer, layer=IAdminLayer,
                view=ISiteTreeTable, manager=IToolbarAddingMenu,
                permission=MANAGE_SITE_PERMISSION, weight=94)
class HuntingCalendarSearchFolderAddMenu(ToolbarMenuItem):
    """Hunting calendar search folder add menu"""
    
    label = _("hunting-calendar-search-folder-add-menu", default="Add hunting calendars search folder")
    label_css_class = 'fa fa-fw fa-search'
    url = 'add-hunting-calendar-search-folder.html'
    modal_target = True
    
    
@pagelet_config(name='add-hunting-calendar-search-folder.html',
                context=ISiteContainer, layer=IPyAMSLayer,
                permission=MANAGE_SITE_PERMISSION)
@ajax_config(name='add-hunting-calendar-search-folder.json',
             context=ISiteContainer, layer=IPyAMSLayer, base=AJAXAddForm)
class HuntingCalendarSearchFolderAddForm(AdminDialogAddForm):
    """Hunting calendar search folder add form"""

    @property
    def title(self):
        return II18n(self.context).query_attribute('title', request=self.request)

    legend = _("hunting-calendar-search-folder-add-form", default="Add hunting calendars search folder")
    icon_css_class = 'fa fa-fw fa-search'

    fields = Fields(ISiteFolderAddFormFields)
    fields['parent'].widgetFactory = SiteManagerFoldersSelectorFieldWidget

    edit_permission = MANAGE_SITE_PERMISSION

    def updateWidgets(self, prefix=None):
        super().updateWidgets(prefix)
        if 'parent' in self.widgets:
            self.widgets['parent'].permission = MANAGE_SITE_PERMISSION

    def create(self, data):
        return HuntingCalendarSearchFolder()

    def update_content(self, content, data):
        data = data.get(self, data)
        # initialize
        content.title = data['title']
        content.short_name = data['title']
        content.notepad = data['notepad']
        intids = get_utility(IIntIds)
        parent = intids.queryObject(data.get('parent'))
        if parent is not None:
            negotiator = get_utility(INegotiator)
            title = II18n(content).get_attribute('title', lang=negotiator.server_language)
            name = translate_string(title, force_lower=True, spaces='-')
            if name in parent:
                index = 1
                new_name = '{name}-{index:02}'.format(name=name, index=index)
                while new_name in parent:
                    index += 1
                    new_name = '{name}-{index:02}'.format(name=name, index=index)
                name = new_name
            parent[name] = content

    def add(self, content):
        pass

    def nextURL(self):
        return absolute_url(self.context, self.request, 'admin#site-tree.html')

    def get_ajax_output(self, changes):
        return {
            'status': 'reload'
        }


@pagelet_config(name='properties.html',
                context=IHuntingCalendarSearchFolder, layer=IPyAMSLayer,
                permission=VIEW_SYSTEM_PERMISSION)
@ajax_config(name='properties.json',
             context=IHuntingCalendarSearchFolder, layer=IPyAMSLayer,
             permission=MANAGE_CONTENT_PERMISSION)
class HuntingCalendarSearchFolderPropertiesEditForm(SharedContentPropertiesEditForm):
    """Hunting calendar search folder properties edit form"""

    def updateWidgets(self, prefix=None):
        # don't override base widgets update method
        super(SharedContentPropertiesEditForm, self).updateWidgets(prefix)


@adapter_config(name='view-settings',
                required=(IHuntingCalendarSearchFolder, IPyAMSLayer, SharedContentPropertiesEditForm),
                provides=IInnerSubForm)
class HuntingCalendarSearchFolderSettingsEditSubform(NullAdapter):
    """Hunting calendar search folder hidden settings edit sub-form extension"""


@viewlet_config(name='hearing.menu',
                context=IHuntingCalendarSearchFolder, layer=IAdminLayer,
                manager=IPropertiesMenu)
class HuntingCalendarSearchFolderHearingMenu(NullAdapter):
    """Hunting calendar search folder hidden hearing menu"""


@viewlet_config(name='location.menu',
                context=IHuntingCalendarSearchFolder, layer=IAdminLayer,
                manager=IPropertiesMenu, permission=VIEW_SYSTEM_PERMISSION)
class HuntingCalendarSearchFolderLocationMenu(NullAdapter):
    """Hunting calendar search folder hidden location menu"""


@viewlet_config(name='tags.menu',
                context=IHuntingCalendarSearchFolder, layer=IAdminLayer,
                manager=IPropertiesMenu, permission=VIEW_SYSTEM_PERMISSION)
class HuntingCalendarSearchFolderTagsMenu(NullAdapter):
    """Hunting calendar search folder hidden tags menu"""


@viewlet_config(name='themes.menu',
                context=IHuntingCalendarSearchFolder, layer=IAdminLayer,
                manager=IPropertiesMenu, permission=VIEW_SYSTEM_PERMISSION)
class HuntingCalendarSearchFolderThemesMenu(NullAdapter):
    """Hunting calendar search folder hidden themes menu"""


@viewlet_config(name='collections.menu',
                context=IHuntingCalendarSearchFolder, layer=IAdminLayer,
                manager=IPropertiesMenu, permission=VIEW_SYSTEM_PERMISSION)
class HuntingCalendarSearchFolderCollectionsMenu(NullAdapter):
    """Hunting calendar search folder hidden collections menu"""
